import os
import tempfile
from inspect import ArgSpec, getfullargspec
from logging import Formatter, StreamHandler, getLogger
from unittest.mock import patch

import invoke

log_fmt = Formatter(
    "%(asctime)s %(name)s L%(lineno)d [%(levelname)s][%(funcName)s] %(message)s "
)
logger = getLogger(__name__)
handler = StreamHandler()
handler.setLevel("INFO")
handler.setFormatter(log_fmt)

logger.setLevel("INFO")
logger.addHandler(handler)
logger.propagate = False

REPOSITORY_ROOT = os.path.dirname(__file__)


def fix_invoke_annotations() -> None:
    def patched_inspect_getargspec(func):
        spec = getfullargspec(func)
        return ArgSpec(*spec[0:4])

    org_task_argspec = invoke.tasks.Task.argspec

    def patched_task_argspec(*args, **kwargs):
        with patch(target="inspect.getargspec", new=patched_inspect_getargspec):
            return org_task_argspec(*args, **kwargs)

    invoke.tasks.Task.argspec = patched_task_argspec


fix_invoke_annotations()

# --------------------------


@invoke.task
def setup(c: invoke.Context):
    c.run("poetry install --no-root")
    with tempfile.TemporaryDirectory() as tmp_d:
        with c.cd(tmp_d):
            c.run(
                "wget http://persoal.citius.usc.es/manuel.fernandez.delgado/papers/jmlr/data.tar.gz"
            )
            c.run(f"tar -xvzf data.tar.gz -C {REPOSITORY_ROOT}/tree_ntk/data")
    logger.info("setup done")


@invoke.task(default=True)
def run(
    c: invoke.Context,
    model: str = "tree_ntk",
    max_tot: int = 5000,
    max_depth: int = 5,
    alpha: float = 1.0,
    search: bool = False,
):
    with c.cd("./tree_ntk"):
        c.run(
            f"poetry run python main.py -max_tot {max_tot} -max_depth {max_depth} -model {model} -alpha {alpha} {'-search' if search else ''}"
        )


def train(
    c: invoke.Context,
    max_depth: int,
    alpha: float,
    model: str,
    max_tot: int,
    reg_coef: float,
    name: str,
    search: bool,
):
    result = c.run(
        f"poetry run python main.py -max_tot {max_tot} -max_depth {max_depth} -model {model} -alpha {alpha} -reg_coef {reg_coef} -name {name} {'-search' if search else ''}"
    )
    return result.ok
