import json
import re

# This is the mapping for the symptom fields. We map the options into numeric values for the finetuning.
# Define the choices dictionaries
negation_choices = {
    "A": 3,
    "C": 1,
    "D": 2,
    "B": -1,
    "E": 4,
    "A. Present": 3,
    "C. Never": 1,
    "D. Negated": 2,
    "B. Unclear": -1,
    "E. Previously Present, Documented as part of copy forward": 4,
}

consistency_choices = {
    "B": 1,
    "C": 2,
    "A": -1,
    "D": 3,
    "B. Intermittent": 1,
    "C. Constant": 2,
    "A. Unclear": -1,
    "D. Frequent": 3,
}

impact_choices = {
    "G": -1,
    "A": 1,
    "B": 2,
    "C": 3,
    "D": 4,
    "E": 5,
    "F": 0,
    "G. Unclear": -1,
    "A. Systemic treatment dosage was reduced due to the symptom": 1,
    "B. Systemic therapy was delayed or temporarily held due to the symptom": 2,
    "C. Systemic treatment was stopped due to the symptom": 3,
    "D. Supportive medications were added or adjusted for the symptom": 4,
    "E. Diagnostics were done for the symptom": 5,
    "F. No change in systemic treatment, supportive medications, or diagnostics was done for this symptom during this visit": 0,
}

cause_options = {
    "A": 2,
    "B": 1,
    "D": -1,
    "A. Cancer": 2,
    "B. Cancer treatment": 1,
    "D. Unclear/Other": -1,
}

daily_func_options = {
    "A. Unclear - The impact on daily functioning is not specified.": -1,
    "B. No Impact - The symptom does not affect daily activities.": 0,
    "C. Mild Impact - Some difficulty but can perform tasks.": 1,
    "D. Moderate Impact - Noticeable difficulty with tasks.": 2,
    "E. Severe Impact - Unable to perform daily tasks.": 3,
    "A": -1,
    "B": 0,
    "C": 1,
    "D": 2,
    "E": 3,
}

emotional_options = {
    "A": -1,
    "B": 0,
    "C": 1,
    "D": 2,
    "E": 3,
    "A. Unclear - The impact on emotional or mental health is not specified.": -1,
    "B. No Impact - The symptom does not affect emotional or mental health.": 0,
    "C. Mild Distress - Some emotional distress but can cope.": 1,
    "D. Moderate Distress - Noticeable emotional distress impacting well-being.": 2,
    "E. Severe Distress - Significant emotional distress requiring intervention.": 3,
}

mobility_options = {
    "A": -1,
    "B": 0,
    "C": 1,
    "D": 2,
    "E": 3,
    "A. Unclear - The impact on mobility is not specified.": -1,
    "B. No Impact - The symptom does not affect mobility.": 0,
    "C. Mild Limitation - Some limitation in mobility but can perform physical activities.": 1,
    "D. Moderate Limitation - Noticeable limitation in mobility.": 2,
    "E. Severe Limitation - Unable to perform physical activities.": 3,
}

chronicity_choices = {
    "A": -1,
    "B": 1,
    "C": 2,
    "D": 3,
    "E": 4,
    "F": 5,
    "A. Unclear": -1,
    "B. Predates Cancer Development/Diagnosis": 1,
    "C. Present at time of Cancer Diagnosis": 2,
    "D. Started subsequent to treatment, but not new": 3,
    "E. Historic/Resolved": 4,
    "F. New": 5,
}

acceleration_choices = {
    "A": -1,
    "D": 3,
    "B": 1,
    "E": 0,
    "C": 2,
    "F": -1,
    "A. Unclear": -1,
    "D. Improving": 3,
    "B. Worsening": 1,
    "E. Resolved": 0,
    "C. Stable": 2,
    "F. N/A": -1,
}


# this is the dictionary that keeps the mapping of the fields
symptom_fields_mapping = {
    "Negation": negation_choices,
    "Symptom Consistency": consistency_choices,
    "Symptom Chronicity": chronicity_choices,
    "Symptom Acceleration": acceleration_choices,
    "daily_func": daily_func_options,
    "mobility": mobility_options,
    "mental_health": emotional_options,
    "Cause": cause_options,
    "Symptom Impact": impact_choices,
    "Negation Reason": None,
}

# this is the list of the symptoms
symptom_choices = [
    "(A): Diarrhea",
    "(B): Constipation",
    "(C): Nausea",
    "(D): Vomiting",
    "(E): Abdominal Pain",
    "(F): Abdominal Distension",
    "(G): Fatigue",
    "(H): Allergic reaction",
    "(I): Weight loss",
    "(J): Erythema",
    "(K): Hair loss",
    "(L): Neutropenia",
    "(M): Anemia",
    "(N): Abnormal liver function",
    "(O): Dyspnea",
    "(P): Appetite Loss",
    "(Q): Fever",
    "(R): Chills",
    "(S): Jaundice",
    "(T): Thrombocytopenia",
    "(U): Sensory Neuropathy",
    "(V): Motor Neuropathy",
    "(W): Cold-induced Neuropathy",
]

symptom_choices_mapping = {
    "(A)": "(A): Diarrhea",
    "(B)": "(B): Constipation",
    "(C)": "(C): Nausea",
    "(D)": "(D): Vomiting",
    "(E)": "(E): Abdominal Pain",
    "(F)": "(F): Abdominal Distension",
    "(G)": "(G): Fatigue",
    "(H)": "(H): Allergic reaction",
    "(I)": "(I): Weight loss",
    "(J)": "(J): Erythema",
    "(K)": "(K): Hair loss",
    "(L)": "(L): Neutropenia",
    "(M)": "(M): Anemia",
    "(N)": "(N): Abnormal liver function",
    "(O)": "(O): Dyspnea",
    "(P)": "(P): Appetite Loss",
    "(Q)": "(Q): Fever",
    "(R)": "(R): Chills",
    "(S)": "(S): Jaundice",
    "(T)": "(T): Thrombocytopenia",
    "(U)": "(U): Sensory Neuropathy",
    "(V)": "(V): Motor Neuropathy",
    "(W)": "(W): Cold-induced Neuropathy",
}


def gt_to_response(data_dict):
    """
    Args:
        gt (dict): a dictionary with (gt column, gt value) pairs from one row in the data
    Returns:
        str: the intended response for the LLM
    """

    # Function to reverse map numeric values to their corresponding descriptions
    def map_values(field, value):
        field_map = symptom_fields_mapping.get(field, {})
        for desc, num in field_map.items():
            if num == value:
                return desc
        return "Unknown"  # Return 'Unknown' if no match found

    # Initialize the result dictionary
    result = {}

    # Process each symptom entry in the input dictionary
    found_in_note = False
    for key, value in data_dict.items():
        if "found_in_note" in key:
            found_in_note = value

        if not found_in_note:
            continue

        symptom_code = key.split(": ")[1].split("_")[0]  # Handle specific renaming
        # field is to remove the symptom_code and '_'
        field = key[len(key.split("_")[0]) + 1 :]

        # Create an entry in the result if the symptom hasn't been added yet
        if symptom_code not in result:
            result[symptom_code] = {}

        # Map the value to the corresponding description
        if "found_in_note" not in key:
            if field == "Negation Reason":
                # keep the string as is
                result[symptom_code][field] = value
            elif field in ["Symptom Impact", "Cause"] and isinstance(value, str):
                # map the individual values in the list to the corresponding code
                value = value.split(", ")
                result[symptom_code][field] = []
                for val in value:
                    result[symptom_code][field].append(map_values(field, val))
            else:
                result[symptom_code][field] = map_values(field, value)

        result[symptom_code]["Symptom Map"] = key.split(": ")[0]
    return str(result)


def parse_response(response, response_start="", verbose=False):
    """
    Args:
        response (str): the raw LLM response in its entirety
        response_start (str): the prefix to the response, to be removed from response
    Outputs
        dict: a dictionary with (gt column, gt value) pairs from one row in the data
    """
    # Remove everything before and including "My answer is: "
    response = re.sub(r".*?My answer is: ", "", response)
    response = response.replace("'", '"')  # Ensure JSON uses double quotes
    response = response.replace(
        "E. Previously Present, Documented as part of copy forward ",
        "E. Previously Present, Documented as part of copy forward",
    )

    # Regular expression to extract the JSON content
    json_match = re.search(r"\{.*\}", response, re.DOTALL)
    answers = {}

    # initialize the answers dictionary, input the default values, usually 0 for flag of presenting symptoms, -1 for attributes or empty list
    for k, symptom in symptom_choices_mapping.items():
        answers[f"{symptom}_found_in_note"] = 0
        answers[f"{symptom}_Symptom Impact"] = []
        answers[f"{symptom}_Negation Reason"] = []
        answers[f"{symptom}_Cause"] = []

        for attr in symptom_fields_mapping.keys():
            answers[f"{symptom}_{attr}"] = -1

    # Check if the response contains the 'Symptom Map' key, if not then the answer is the N/A
    if "Symptom Map" not in response:
        return answers

    if json_match:
        json_str = json_match.group(0)  # Extracted JSON string
        # Load the JSON
        try:
            parsed_data = json.loads(json_str)
            # Access individual symptoms and their attributes
            for symptom, details in parsed_data.items():
                if "Symptom Map" not in details:
                    continue

                if details["Symptom Map"] not in symptom_choices_mapping.keys():
                    continue

                symptom_key = symptom_choices_mapping[details["Symptom Map"]]

                answers[f"{symptom_key}_found_in_note"] = 1
                for attr in symptom_fields_mapping.keys():
                    # check if the attribute is present in the details
                    if attr not in details:
                        continue

                    if (
                        attr == "Negation Reason"
                    ):  # for negation reason, we keep the string as is
                        attr_value = details[attr]
                    elif (
                        attr in ["Symptom Impact", "Cause"]
                    ):  # for Symptom Impact and Cause, we map the individual values in the list to the corresponding code
                        if isinstance(details[attr], list):
                            attr_value = []
                            for detail in details[attr]:
                                if detail in symptom_fields_mapping[attr].keys():
                                    attr_value.append(
                                        symptom_fields_mapping[attr][detail]
                                    )
                        elif (
                            details[attr] in symptom_fields_mapping[attr].keys()
                        ):  # sometime the model outputs a single string instead of a list
                            attr_value = [symptom_fields_mapping[attr][details[attr]]]
                    else:  # for the rest of the attributes, we map the value to the corresponding code
                        details[attr] = details[attr].replace(":", ".")
                        if "N/A" in details[attr]:
                            attr_value = -1
                        elif details[attr] not in symptom_fields_mapping[attr]:
                            print(f"Unknown value for {attr}: {details[attr]}")
                            attr_value = -1
                        else:
                            attr_value = symptom_fields_mapping[attr][details[attr]]

                    answers[f"{symptom_key}_{attr}"] = attr_value

        except json.JSONDecodeError as e:
            print("Error parsing JSON:", e)
            if verbose:
                print(response)
    else:
        print("No JSON found in the text.")
        if verbose:
            print(response)
    return answers


# Example usage of the function

if __name__ == "__main__":
    response = """
    some free text here
    {
       "Nausea":{"Negation":"D. Negated","Negation Reason":"N/A","Symptom Impact":["E. Diagnostics were done for the symptom","D. Supportive medications were added or adjusted for the symptom"],"Symptom Consistency":"B. Intermittent","Symptom Chronicity":"C. Present at time of Cancer Diagnosis","Symptom Acceleration":"C. Stable","Symptom Map":"(C)","Cause":["B. Cancer treatment","D. Unclear/Other"],"daily_func":"C. Mild Impact - Some difficulty but can perform tasks.","mobility":"B. No Impact - The symptom does not affect mobility.","mental_health":"D. Moderate Distress - Noticeable emotional distress impacting well-being."},
       "Abdominal Distension":{"Negation":"E. Previously Present, Documented as part of copy forward ","Negation Reason":"DD/MM/YYYY | Cycle # | Regimen Name","Symptom Impact":["D. Supportive medications were added or adjusted for the symptom","C. Systemic treatment was stopped due to the symptom"],"Symptom Consistency":"B. Intermittent","Symptom Chronicity":"C. Present at time of Cancer Diagnosis","Symptom Acceleration":"E. Resolved","Symptom Map":"(F)","Cause":["B. Cancer treatment"],"daily_func":"C. Mild Impact - Some difficulty but can perform tasks.","mobility":"B. No Impact - The symptom does not affect mobility.","mental_health":"D. Moderate Distress - Noticeable emotional distress impacting well-being."}
    }
    some free text here
    """

    ans = parse_response(response)

    print(ans)
