import json
import re

# Define the choices dictionaries
negation_choices = {"A. Present": 3, "C. Never": 1, "D. Negated": 2, "B. Unclear": -1}
severity_choices = {"A. Unclear": -1, "D. Severe": 3, "B. Mild": 1, "C. Moderate": 2}
consistency_choices = {"B. Intermittent": 1, "C. Constant": 2, "A. Unclear": -1}
chronicity_choices = {
    "A. Unclear": -1,
    "B. New": 1,
    "E. Chronic": 4,
    "C. Subacute": 2,
    "D. Nonchronic": 3,
}
acceleration_choices = {
    "A. Unclear": -1,
    "D. Improving": 3,
    "B. Worsening": 1,
    "E. Resolved": 0,
    "C. Stable": 2,
}
led_to_further_diagnosis_choices = {"C. No": 0, "A. Unclear": -1, "B. Yes": 1}

symptom_fields_mapping = {
    "Negation": negation_choices,
    "Symptom Severity": severity_choices,
    "Symptom Consistency": consistency_choices,
    "Symptom Chronicity": chronicity_choices,
    "Symptom Acceleration": acceleration_choices,
    "Led to further Diagnostics": led_to_further_diagnosis_choices,
}

symptom_fields = [
    "Negation",
    "Symptom Severity",
    "Symptom Consistency",
    "Symptom Chronicity",
    "Symptom Acceleration",
    "Led to further Diagnostics",
]

symptom_choices = [
    "(B): LowerBackPain",
    "(C): WeightLoss",
    "(D): AppetiteLoss",
    "(E): Jaundice",
    "(F): Pruritus",
    "(G): Indigestion",
    "(H): Steatorrhea",
    "(I): UrineColorChange",
    "(J): Constipation",
    "(K): Nausea",
    "(L): Vomiting",
    "(M): Diarrhea",
    "(N): GasorBloating",
    "(O): FatigueMalaiseLethargy",
    "(P): EarlySatiety",
    "(Q): BloodGlucose",
    "(R): GI_Bleed",
    "(S): Melena",
    "(T): BRBPR",
    "(U): AbdominalPain",
    "(V): UpperMidBackPain",
]


def gt_to_response(data_dict):
    # Function to reverse map numeric values to their corresponding descriptions
    def map_values(field, value):
        field_map = symptom_fields_mapping.get(field, {})
        for desc, num in field_map.items():
            if num == value:
                return desc
        return "Unknown"  # Return 'Unknown' if no match found

    # Initialize the result dictionary
    result = {}

    # Process each symptom entry in the input dictionary
    found_in_note = False
    for key, value in data_dict.items():
        print(key, value)

        if "found_in_note" in key:
            found_in_note = value

        if not found_in_note:
            continue

        symptom_code = key.split(": ")[1].split("_")[0]  # Handle specific renaming
        field = key.split("_")[1]

        # Create an entry in the result if the symptom hasn't been added yet
        if symptom_code not in result:
            result[symptom_code] = {}

        # Map the value to the corresponding description
        if "found_in_note" not in key:
            result[symptom_code][field] = map_values(field, value)

    # Add 'Symptom Map' entries
    for symptom_code in result.keys():
        map_key = [k for k in data_dict.keys() if symptom_code.replace(" ", "") in k][
            0
        ].split("_")[0]
        result[symptom_code]["Symptom Map"] = map_key

    return str(result)


def parse_response(response, response_start="", verbose=False):
    """
    Parse the response and dynamically generate the structure for each symptom.
    Convert the result into a DataFrame.
    """
    response = response.replace(response_start, "")
    if '"' in response and "'" in response:
        response = response.replace("'", "")
    else:
        response = response.replace("'", '"')
    response = response.replace("\\n", "")
    response = response.replace(
        "Led to Further Diagnostics", "Led to further Diagnostics"
    )
    answers = {}
    # initialize the answers with {symptom_key}_found_in_note 0, other fields with -1
    for symptom in symptom_choices:
        answers[f"{symptom}_found_in_note"] = 0
        for attr in symptom_fields:
            answers[f"{symptom}_{attr}"] = -1

    json_match = re.search(r"\{.*\}", response, re.DOTALL)

    if json_match:
        json_str = json_match.group(0)  # Extracted JSON string
        # Load the JSON
        try:
            parsed_data = json.loads(json_str)
            # Access individual symptoms and their attributes
            for symptom, details in parsed_data.items():
                symptom_key = details["Symptom Map"]
                # remove the space in between the symptom key
                # eg. (I): Urine Color Change -> (I): UrineColorChange, but keeping the space after the ":"
                symptom_key = re.sub(r"\s+", "", symptom_key)
                symptom_key = symptom_key.replace(":", ": ")
                if symptom_key in symptom_choices:
                    answers[f"{symptom_key}_found_in_note"] = 1
                    for attr in symptom_fields:
                        # check if the attribute is present in the details
                        if attr in details:
                            attr_value = symptom_fields_mapping[attr][details[attr]]
                            answers[f"{symptom_key}_{attr}"] = attr_value
        except json.JSONDecodeError as e:
            print(response)
            print("Error parsing JSON:", e)
    else:
        print("No JSON found in the text.")

    return answers