import numpy as np
from sklearn.gaussian_process import GaussianProcessRegressor
from sklearn.gaussian_process.kernels import RBF, WhiteKernel, ConstantKernel
from sklearn.preprocessing import StandardScaler


def generate_dag(num_nodes=4, sparsity=0.5):
    # only possible edges i -> j for i < j
    dag = np.zeros((num_nodes, num_nodes), dtype=bool)
    for i in range(num_nodes):
        for j in range(i + 1, num_nodes):
            dag[i, j] = np.random.rand() < sparsity  # 50% probability for each edge
    return dag


def sample_from_gp(X, mean_function, kernel):
    gp = GaussianProcessRegressor(kernel=kernel)
    y_sample = gp.sample_y(X, random_state=None)[:, 0] + mean_function
    return y_sample


def create_data(sample_size, num_nodes=4, sparsity=0.5):
    # Initialize data matrix
    X = np.random.normal(0, 1, (sample_size, num_nodes))

    # Initialize DAG
    dag = generate_dag(num_nodes, sparsity)

    # Process each variable according to the DAG
    for j in range(num_nodes):
        parents = np.where(dag[:, j])[0]
        if len(parents) > 0:
            weights = np.random.uniform(-2, 2, len(parents))
            mean_function = np.dot(X[:, parents], weights)

            # Creating a Gaussian Process kernel
            length_scales = np.random.uniform(0.1, 0.6, len(parents))
            kernel = ConstantKernel(1.0) * RBF(length_scale=length_scales) + WhiteKernel(noise_level=np.sqrt(0.1))

            X[:, j] = sample_from_gp(X[:, parents], mean_function, kernel)
            X = StandardScaler().fit_transform(X)
    return X, dag


def apply_nonlinear_function(X):
    functions = [np.sin, np.cos, np.tanh, lambda x: 1 / (1 + np.exp(-x)),
                 lambda x: np.power(x, 2), lambda x: np.power(x, 3)]
    selected_function = np.random.choice(functions)

    return selected_function(X)


def create_data_2(sample_size, num_nodes=4, sparsity=0.5):
    # Initialize data matrix
    X = np.random.normal(0, 1, (sample_size, num_nodes))

    # Initialize DAG
    dag = generate_dag(num_nodes, sparsity)

    # Process each variable according to the DAG
    for j in range(num_nodes):
        parents = np.where(dag[:, j])[0]
        if len(parents) > 0:
            noise_level = np.random.choice([0.2, 0.5])
            X[:, j] = np.random.normal(0, noise_level, sample_size)
            for i in parents:
                is_linear = np.random.random() < .5
                if is_linear:
                    X[:, j] += (np.random.choice([0.5, -1.5]) + np.random.random()) * X[:, i]
                else:
                    X[:, j] += apply_nonlinear_function(X[:, i]) + apply_nonlinear_function(X[:, i])
        X[:, j] = (X[:, j] - X[:, j].mean()) / X[:, j].std()

    return X, dag
