import torch.nn as nn
import torch.nn.functional as F
import math
import numpy as np
import torch
from torch.nn.parameter import Parameter
from torch.nn.modules.module import Module
from torch_geometric.nn import GATConv
from .base_model import BaseModel
from torch_sparse import coalesce, SparseTensor
from .base_model import BatchNorm1d_plus


class GAT(BaseModel):

    def __init__(self, nfeat, nhid, nclass, heads=8, output_heads=1, dropout=0., lr=0.01,
            nlayers=2, with_bn=False, weight_decay=5e-4, device=None, args=None):

        super(GAT, self).__init__()
        from utils import eval_acc, eval_f1, eval_rocauc
        if args.dataset == 'twitch-e':
            self.eval_func = eval_rocauc
        elif args.dataset == 'elliptic':
            self.eval_func = eval_f1
        elif args.dataset in ['cora', 'amazon-photo', 'ogb-arxiv', 'fb100', 'ogb-products']:
            self.eval_func = eval_acc
        else:
            raise NotImplementedError

        assert device is not None, "Please specify 'device'!"
        self.device = device
        self.nclass = nclass

        if with_bn:
            self.bns = nn.ModuleList()
            self.bns.append(BatchNorm1d_plus(nhid*heads))

        self.convs = nn.ModuleList()
        self.convs.append(GATConv(
                nfeat,
                nhid,
                heads=heads,
                dropout=dropout))

        for _ in range(nlayers - 2):
            self.convs.append(GATConv(
                    nhid * heads,
                    nhid,
                    heads=heads,
                    dropout=dropout))
            if with_bn:
                self.bns.append(BatchNorm1d_plus(nhid*heads))

        self.convs.append(GATConv(
            nhid * heads,
            nclass,
            heads=output_heads,
            concat=False,
            dropout=dropout))

        self.dropout = dropout
        self.weight_decay = weight_decay
        self.lr = lr
        self.output = None
        self.best_model = None
        self.best_output = None
        self.activation = F.elu
        self.name = 'GAT'
        self.args = args
        self.with_bn = with_bn

    def reset_bn_momentum(self, momentum):
        if type(momentum) is float:
            for bn in self.bns:
                bn.momentum = momentum

    def reset_bn_track_running_stats(self, stats=False):
        for bn in self.bns:
            bn.track_running_stats = stats

    def adaptation_bn_statistic(self, x, edge_index, edge_weight=None, weights=None):
        if len(self.bns) == len(weights):
            x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
            for i in range(len(self.bns)):
                if edge_weight is not None:
                    x = self.convs[i](x, adj)
                else:
                    x = self.convs[i](x, edge_index)

                mean = torch.mean(x, 0)
                var = torch.var(x, 0, unbiased=True)
                weight = weights[i]
                x = self.bns[i](x)

                running_mean = self.bns[i].running_mean * (1 - weight) + mean * weight
                running_var = self.bns[i].running_var * (1 - weight) + var * weight
                self.bns[i].running_mean = running_mean.detach()
                self.bns[i].running_var = running_var.detach()
                x = self.activation(x)
                x = F.dropout(x, p=self.dropout, training=self.training)
            return
        else:
            print("Wrong weights!")
            return

    def get_embed_plus(self, x, edge_index, edge_weight=None, weights=None):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        if edge_weight is not None:
            adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
        for ii, layer in enumerate(self.convs):
            if ii == len(self.convs) - 1:
                return x
            if edge_weight is not None:
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.convs) - 1:
                if self.with_bn:
                    if weights is not None:
                        x = self.bns[ii].forward_plus(x, weights[ii])
                    else:
                        x = self.bns[ii].forward_plus(x, weights)
                x = F.relu(x)
        return x

    def get_embed_ith(self, x, edge_index, edge_weight=None, i=None):
        if i < 1 or i > len(self.convs) - 1:
            print("Wrong i!")
            return
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        for ii, layer in enumerate(self.convs):
            if ii == len(self.convs) - 1:
                return x
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.convs) - 1:
                if self.with_bn:
                    x = self.bns[ii](x)
                if ii == i - 1:
                    return x
                x = F.relu(x)
        return x

    def forward(self, x, edge_index, edge_weight=None, renorm=False):
        if edge_weight is not None:
            adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()

        for i, conv in enumerate(self.convs[:-1]):
            if edge_weight is not None:
                x = conv(x, adj)
            else:
                x = conv(x, edge_index, edge_weight)
            if self.with_bn:
                if renorm:
                    x = self.bns[i].forward_renorm(x)
                else:
                    x = self.bns[i](x)
            x = self.activation(x)
            x = F.dropout(x, p=self.dropout, training=self.training)
        if edge_weight is not None:
            x = self.convs[-1](x, adj)
        else:
            x = self.convs[-1](x, edge_index, edge_weight)
        return x

    def initialize(self):
        self.reset_parameters()

    def reset_parameters(self):
        for conv in self.convs:
            conv.reset_parameters()
        for bn in self.bns:
            bn.reset_parameters()

    def get_embed(self, x, edge_index, edge_weight=None):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        for ii, layer in enumerate(self.convs):
            if ii == len(self.convs) - 1:
                return x
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.convs) - 1:
                if self.with_bn:
                    x = self.bns[ii](x)
                x = F.relu(x)
        return x
