import numpy as np
import torch.nn as nn
import torch.nn.functional as F
import math
import torch
from torch.nn.parameter import Parameter
from torch.nn.modules.module import Module
from torch_geometric.nn import GCNConv
from .base_model import BaseModel
from .base_model import BatchNorm1d_plus
from torch_sparse import coalesce, SparseTensor
import torch.optim as optim


class GCN(BaseModel):

    def __init__(self, nfeat, nhid, nclass, nlayers=2, dropout=0.5, lr=0.01, save_mem=True,
                with_bn=False, weight_decay=5e-4, with_bias=True, device=None, args=None):

        super(GCN, self).__init__()

        assert device is not None, "Please specify 'device'!"
        self.device = device
        self.args = args
        self.nclass = nclass
        from utils import eval_acc, eval_f1, eval_rocauc
        if args.dataset == 'twitch-e':
            self.eval_func = eval_rocauc
        elif args.dataset == 'elliptic':
            self.eval_func = eval_f1
        elif args.dataset in ['cora', 'amazon-photo', 'ogb-arxiv', 'fb100', 'ogb-products']:
            self.eval_func = eval_acc
        else:
            raise NotImplementedError

        self.layers = nn.ModuleList([])
        if with_bn:
            self.bns = nn.ModuleList()

        if nlayers == 1:
            self.layers.append(GCNConv(nfeat, nclass, bias=with_bias, normalize=not save_mem))
        else:
            self.layers.append(GCNConv(nfeat, nhid, bias=with_bias, normalize=not save_mem))
            if with_bn:
                self.bns.append(BatchNorm1d_plus(nhid))
            for i in range(nlayers-2):
                self.layers.append(GCNConv(nhid, nhid, bias=with_bias, normalize=not save_mem))
                if with_bn:
                    self.bns.append(BatchNorm1d_plus(nhid))
            self.layers.append(GCNConv(nhid, nclass, bias=with_bias, normalize=not save_mem))

        self.dropout = dropout
        self.weight_decay = weight_decay
        self.lr = lr
        self.output = None
        self.best_model = None
        self.best_output = None
        self.with_bn = with_bn
        self.name = 'GCN'

    def reset_bn_momentum(self, momentum):
        if type(momentum) is float:
            for bn in self.bns:
                bn.momentum = momentum

    def reset_bn_track_running_stats(self, stats=False):
        for bn in self.bns:
            bn.track_running_stats = stats

    def adaptation_bn_statistic(self, x, edge_index, edge_weight=None, weights=None):
        if len(self.bns) == len(weights):
            x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()

            for i in range(len(self.bns)):
                if edge_weight is not None:
                    x = self.layers[i](x, adj)
                else:
                    x = self.layers[i](x, edge_index)
                mean = torch.mean(x, 0)
                var = torch.var(x, 0, unbiased=True)
                weight = weights[i]

                x = self.bns[i](x)
                running_mean = self.bns[i].running_mean * (1 - weight) + mean * weight
                running_var = self.bns[i].running_var * (1 - weight) + var * weight
                self.bns[i].running_mean = running_mean.detach()
                self.bns[i].running_var = running_var.detach()
                x = F.relu(x)
        else:
            print("Wrong weights!")
            return

    def get_embed_plus(self, x, edge_index, edge_weight=None, weights=None, debug=0):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        if edge_weight is not None:
            adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
        for ii, layer in enumerate(self.layers):
            if ii == len(self.layers) - 1:
                return x
            if edge_weight is not None:
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.layers) - 1:
                if self.with_bn:
                    if weights is not None:
                        x = self.bns[ii].forward_plus(x, weights[ii], debug=debug)
                    else:
                        x = self.bns[ii].forward_plus(x, weights)
                x = F.relu(x)
        return x

    def forward_plus(self, x, edge_index, edge_weight=None, weights=None):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        if edge_weight is not None:
            adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()

        for ii, layer in enumerate(self.layers):
            if edge_weight is not None:
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.layers) - 1:
                if self.with_bn:
                    if weights is not None:
                        x = self.bns[ii].forward_plus(x, weights[ii])
                    else:
                        x = self.bns[ii].forward_plus(x, weights)
                x = F.relu(x)
                x = F.dropout(x, p=self.dropout, training=self.training)
                self.h = x
        return x

    def forward(self, x, edge_index, edge_weight=None, renorm=False):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        if edge_weight is not None:
            adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()

        for ii, layer in enumerate(self.layers):
            if edge_weight is not None:
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.layers) - 1:
                if self.with_bn:
                    if renorm:
                        x = self.bns[ii].forward_renorm(x)
                    else:
                        x = self.bns[ii](x)
                x = F.relu(x)
                x = F.dropout(x, p=self.dropout, training=self.training)
                self.h = x
        return x

    def get_embed(self, x, edge_index, edge_weight=None):
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        for ii, layer in enumerate(self.layers):
            if ii == len(self.layers) - 1:
                return x
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.layers) - 1:
                if self.with_bn:
                    x = self.bns[ii](x)
                x = F.relu(x)
        return x

    def get_embed_ith(self, x, edge_index, edge_weight=None, i=None):
        if i < 1 or i > len(self.layers)-1:
            print("Wrong i!")
            return
        x, edge_index, edge_weight = self._ensure_contiguousness(x, edge_index, edge_weight)
        for ii, layer in enumerate(self.layers):
            if ii == len(self.layers) - 1:
                return x
            if edge_weight is not None:
                adj = SparseTensor.from_edge_index(edge_index, edge_weight, sparse_sizes=2 * x.shape[:1]).t()
                x = layer(x, adj)
            else:
                x = layer(x, edge_index)
            if ii != len(self.layers) - 1:
                if self.with_bn:
                    x = self.bns[ii](x)
                if ii == i-1:
                    return x
                x = F.relu(x)
        return x

    def initialize(self):
        for m in self.layers:
            m.reset_parameters()
        if self.with_bn:
            for bn in self.bns:
                bn.reset_parameters()

