"""
Utility tool to convert AIBB accession numbers to study dates.

Author(s):
    [Anonymous Authors]

Licensed under the MIT License. Copyright 2022 Anonymized Institution.
"""
from datetime import datetime
import os
import pandas as pd
from pathlib import Path
from typing import Optional, Union


class AccessionConverter:
    def __init__(
        self,
        aibb_to_anon_fn: Union[Path, str],
        anon_to_date_fn: Union[Path, str]
    ):
        """
        Utility tool to convert AIBB accession numbers to study dates.
        Args:
            aibb_to_anon_fn: CSV filepath containing the mappings from AIBB
                accession numbers to Anon Inst accession numbers.
            anon_to_date_fn: CSV filepath containing the mappings from Anon
                Inst accession numbers to study dates.
        """
        self.aibb_to_anon_fn = aibb_to_anon_fn
        self.aibb_to_anon = pd.read_csv(
            os.path.abspath(aibb_to_anon_fn),
            error_bad_lines=False,
            warn_bad_lines=False
        )
        self.anon_to_date_fn = anon_to_date_fn
        self.anon_to_date = pd.read_csv(os.path.abspath(anon_to_date_fn))
        self.anon_to_date = self.anon_to_date[
            ["ANON_ACCESSION", "STUDY_DATE"]
        ]

    def get_date(self, aibb_accession_number: int) -> Optional[datetime]:
        """
        Returns the study date associated with a AIBB accession number.
        Input:
            aibb_accession_number: A AIBB accession number.
        Returns:
            The associated study date represented as a datetime object.
        """
        anon_accession_number = self.aibb_to_anon[
            self.aibb_to_anon.AIBB_ACCESSION == int(aibb_accession_number)
        ][["ANON_ACCESSION"]]
        if anon_accession_number.empty:
            return None
        anon_accession_number = anon_accession_number.iat[0, 0]
        study_date = self.anon_to_date[
            self.anon_to_date.ANON_ACCESSION == anon_accession_number
        ][["STUDY_DATE"]]
        if study_date.empty:
            return None
        return datetime.strptime(str(int(study_date.iat[0, 0])), "%Y%m%d")
