
# %% Imports

import pickle
import os
from time import time
import matplotlib.pyplot as plt

import tensorflow as tf
import numpy as np
from tensorflow_probability import distributions as tfd

from pavi.dual.models import (
    PAVEFamily
)
from pavi.utils.callbacks import (TimerCallback, BatchELBOCallback)
from generative_hbms.GM import (
    seed,
    G, N,
    pavi_kwargs,
    val_data,
    na_val_idx,
    na_dataset
)

# %% FRMFamily kwargs

d = 128
num_heads = 4
key_dim = d // num_heads
k = 1
m = 128
n_sabs = 2

rff_kwargs = dict(
    units_per_layers=[128, d]
)

mab_kwargs = dict(
    multi_head_attention_kwargs=dict(
        num_heads=num_heads,
        key_dim=key_dim
    ),
    rff_kwargs=rff_kwargs,
    layer_normalization_h_kwargs=dict(),
    layer_normalization_out_kwargs=dict()
)

isab_kwargs = dict(
    m=m,
    d=d,
    mab_h_kwargs=mab_kwargs,
    mab_out_kwargs=mab_kwargs
)

set_transformer_kwargs = dict(
    embedding_size=d,
    encoder_kwargs=dict(
        type="ISAB",
        kwargs_per_layer=[
            isab_kwargs
        ] * n_sabs
    ),
    decoder_kwargs=dict(
        pma_kwargs=dict(
            k=k,
            d=d,
            rff_kwargs=rff_kwargs,
            mab_kwargs=mab_kwargs,
        ),
        sab_kwargs=mab_kwargs,
        rff_kwargs=rff_kwargs
    )
)

conditional_nf_chain_kwargs = dict(
    nf_type_kwargs_per_bijector=[
        (
            "MAF",
            dict(
                hidden_units=[128, 128],
                kernel_initializer=tf.keras.initializers.RandomNormal(
                    mean=0.0,
                    stddev=1e-4
                ),
                bias_initializer="zeros"
            )
        ),
        (
            "affine",
            dict(
                scale_type="tril",
                rff_kwargs=rff_kwargs
            )
        )
    ],
    with_permute=False,
    with_batch_norm=False
)

flow_posterior_scheme_kwargs = (
    "flow",
    dict(
        conditional_nf_chain_kwargs=conditional_nf_chain_kwargs
    )
)

posterior_schemes_kwargs = {
    "mu": flow_posterior_scheme_kwargs,
    "mu_g": flow_posterior_scheme_kwargs,
    "probs": flow_posterior_scheme_kwargs,
    "x": (
        "observed",
        dict()
    )
}

pavi_family_kwargs = dict(
    posterior_schemes_kwargs=posterior_schemes_kwargs,
    encoding_sizes={
        ('P',): d,
        ('P', 'G'): d,
    },
    embedder_rff_kwargs=rff_kwargs,
    set_transformer_kwargs=set_transformer_kwargs,
    **pavi_kwargs
)


# %% We build our architecture
fbrm_family = PAVEFamily(
    **pavi_family_kwargs
)

# %%

observed_values = {
    "x": tf.expand_dims(val_data["x"][na_val_idx:na_val_idx+1], axis=0)
}

# %% We fit the training data
time_1 = time()

# %% We select the loss used for training

fbrm_family.compile(
    train_method="reverse_KL",
    n_theta_draws=8,
    optimizer=tf.keras.optimizers.Adam(
        learning_rate=tf.keras.optimizers.schedules.ExponentialDecay(
            initial_learning_rate=1e-3,
            decay_steps=200,
            decay_rate=0.9
        )
    )
)

# %% We fit the training data

call_me_maybe = BatchELBOCallback(
    elbo_batches=4,
    p_model=pavi_kwargs["full_hbm"],
    observed_values=observed_values,
    sample_size=32
)

hist_2 = fbrm_family.fit(
    na_dataset,
    batch_size=1,
    epochs=1_000,
    callbacks=[
        TimerCallback(),
        call_me_maybe
    ]
)
time_2 = time()

# %% We compute the ELBO
sample_size = 256
q_sample = fbrm_family.sample(
    sample_shape=(sample_size,),
    observed_values=observed_values,
    return_observed_values=True
)
q = fbrm_family.log_prob(
    q_sample
)
p = pavi_kwargs["full_hbm"].log_prob(
    q_sample
)
ELBO = tf.reduce_mean(p - q).numpy()

print(f"""
    Idx:  {na_val_idx}
    Time: {time_2 - time_1}
    Loss: {- ELBO}
""")

# %%

fig, axs = plt.subplots(
    nrows=1,
    ncols=2,
    figsize=(20, 10)
)
for g in range(G):
    axs[0].scatter(
        val_data["x"][na_val_idx, g, :, 0],
        val_data["x"][na_val_idx, g, :, 1],
        color=f"C{g}",
        alpha=0.5
    )

axs[0].axis("equal")
axs[0].set_ylabel(
    f"{na_val_idx}",
    fontsize=30,
    rotation=0
)
plt.draw()
x_lim = axs[0].get_xlim()
y_lim = axs[0].get_ylim()

axs[1].scatter(
    q_sample["mu"][:, 0, :, 0],
    q_sample["mu"][:, 0, :, 1],
    color="black",
    s=20,
    alpha=0.05
)
for g in range(G):
    axs[1].scatter(
        q_sample["mu_g"][:, 0, g, :, 0],
        q_sample["mu_g"][:, 0, g, :, 1],
        color=f"C{g}",
        s=20,
        alpha=0.05
    )
axs[1].set_xlim(x_lim)
axs[1].set_ylim(y_lim)

axs[0].set_title(
    "Data",
    fontsize=30
)
axs[1].set_title(
    "Posterior samples",
    fontsize=30
)
plt.show()

# %%
base_name = (
    "../data/"
    f"GM_PAVI-E_idx{na_val_idx}_seed{seed}_"
)

pickle.dump(
    {
        **hist_2.history,
        "ELBO": call_me_maybe.ELBOs
    },
    open(
        base_name + "history.p",
        "wb"
    )
)
pickle.dump(
    - ELBO,
    open(
        base_name + "loss.p",
        "wb"
    )
)