
# %% Imports

import pickle
from time import time
import matplotlib.pyplot as plt

import tensorflow as tf
from tensorflow_probability import distributions as tfd

from pavi.dual.models import (
    UIVFamily
)
from pavi.utils.callbacks import (TimerCallback,)
from generative_hbms.GM import (
    seed,
    G, N,
    generative_hbm,
    link_functions,
    val_data,
    na_val_idx,
)

# %% UIVFamily kwargs

nf_chain_kwargs = dict(
    nf_type_kwargs_per_bijector=[
        (
            "affine",
            dict(
                scale_type="diag",
                rff_kwargs=dict(
                    units_per_layers=[64, 64]
                )
            )
        )
    ],
    with_permute=False,
    with_batch_norm=False
)

uivi_kwargs = dict(
    generative_hbm=generative_hbm,
    link_functions=link_functions,
    observed_rv="x",
    conditional_nf_chain_kwargs=nf_chain_kwargs,
    embedding_RV_size=6
)


# %% We build our architecture
ui_family = UIVFamily(
    **uivi_kwargs
)

# %%

observed_values = {
    "x": tf.expand_dims(val_data["x"][na_val_idx], axis=0)
}

# %% We fit the training data
time_1 = time()

# %% We select the loss used for training

ui_family.compile(
    n_theta_draws=8,
    t_mcmc_burn_in=5,
    n_mcmc_samples=5,
    optimizer=tf.keras.optimizers.Adam(
        learning_rate=tf.keras.optimizers.schedules.ExponentialDecay(
            initial_learning_rate=1e-2,
            decay_steps=300,
            decay_rate=0.9
        )
    )
)

# %% We fit the training data
hist_2 = ui_family.fit(
    observed_values,
    batch_size=1,
    epochs=3_000,
    verbose=0,
    callbacks=[
        TimerCallback()
    ]
)
time_2 = time()

# %% We compute the ELBO
sample_size = 256
repeated_x = tf.repeat(
    val_data["x"][na_val_idx:na_val_idx + 1],
    repeats=(sample_size,),
    axis=0
)
z, epsilon = ui_family.sample((sample_size,), return_epsilon=True)
q = ui_family.q_z(z, epsilon)
p = ui_family.generative_hbm.log_prob(
    **z,
    x=repeated_x
)
ELBO = tf.reduce_mean(
    p - q
).numpy()

print(f"""
    Idx:  {na_val_idx}
    Time: {time_2 - time_1}
    Loss: {- ELBO}
""")

# %%

fig, axs = plt.subplots(
    nrows=1,
    ncols=2,
    figsize=(20, 10)
)
for g in range(G):
    axs[0].scatter(
        val_data["x"][na_val_idx, g, :, 0],
        val_data["x"][na_val_idx, g, :, 1],
        color=f"C{g}",
        alpha=0.5
    )

axs[0].axis("equal")
axs[0].set_ylabel(
    f"{na_val_idx}",
    fontsize=30,
    rotation=0
)
plt.draw()
x_lim = axs[0].get_xlim()
y_lim = axs[0].get_ylim()

axs[1].scatter(
    z["mu"][:, :, 0],
    z["mu"][:, :, 1],
    color="black",
    s=20,
    alpha=0.05
)
for g in range(G):
    axs[1].scatter(
        z["mu_g"][:, g, :, 0],
        z["mu_g"][:, g, :, 1],
        color=f"C{g}",
        s=20,
        alpha=0.05
    )
axs[1].set_xlim(x_lim)
axs[1].set_ylim(y_lim)

axs[0].set_title(
    "Data",
    fontsize=30
)
axs[1].set_title(
    "Posterior samples",
    fontsize=30
)
plt.show()

# %%
base_name = (
    "../data/"
    f"GM_UIVI_idx{na_val_idx}_seed{seed}_"
)

pickle.dump(
    hist_2.history,
    open(
        base_name + "history.p",
        "wb"
    )
)
pickle.dump(
    - ELBO,
    open(
        base_name + "loss.p",
        "wb"
    )
)