# https://github.com/kuangliu/pytorch-cifar/blob/master/models/resnet.py

import torch
import torch.nn as nn
import torch.nn.functional as F
from models.layers import SubnetConv, SubnetLinear


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, conv_layer, stride=1, prune_reg='weight', task_mode='harp_prune'):
        super(BasicBlock, self).__init__()
        self.conv1 = conv_layer(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False, prune_reg=prune_reg, task_mode=task_mode
        )
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv_layer(
            planes, planes, kernel_size=3, stride=1, padding=1, bias=False, prune_reg=prune_reg, task_mode=task_mode
        )
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                conv_layer(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                    prune_reg=prune_reg,
                    task_mode=task_mode,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, conv_layer, stride=1, prune_reg='weight', task_mode='harp_prune'):
        super(Bottleneck, self).__init__()
        self.conv1 = conv_layer(in_planes, planes, kernel_size=1, bias=False, prune_reg=prune_reg, task_mode=task_mode)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv_layer(
            planes, planes, kernel_size=3, stride=stride, padding=1, bias=False, prune_reg=prune_reg, task_mode=task_mode
        )
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = conv_layer(
            planes, self.expansion * planes, kernel_size=1, bias=False, prune_reg=prune_reg, task_mode=task_mode
        )
        self.bn3 = nn.BatchNorm2d(self.expansion * planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                conv_layer(
                    in_planes,
                    self.expansion * planes,
                    kernel_size=1,
                    stride=stride,
                    bias=False,
                    prune_reg=prune_reg,
                    task_mode=task_mode,
                ),
                nn.BatchNorm2d(self.expansion * planes),
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, conv_layer, linear_layer, block, num_blocks, mean, std,
                 num_classes=10, prune_reg='weight', task_mode='harp_prune', normalize=False, normalize_features=False, normalize_logits=False):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.conv_layer = conv_layer
        self.normalize = normalize
        #self.mean = torch.Tensor(mean).unsqueeze(1).unsqueeze(1)
        #self.std = torch.Tensor(std).unsqueeze(1).unsqueeze(1)
        self.num_classes = torch.Tensor(num_classes)
        self.normalize_logits = normalize_logits

        self.conv1 = conv_layer(3, 64, kernel_size=3, stride=1, padding=1, bias=False, prune_reg=prune_reg, task_mode=task_mode)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1, prune_reg=prune_reg, task_mode=task_mode)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2, prune_reg=prune_reg, task_mode=task_mode)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2, prune_reg=prune_reg, task_mode=task_mode)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2, prune_reg=prune_reg, task_mode=task_mode)
        self.linear = linear_layer(512 * block.expansion, num_classes, prune_reg=prune_reg, task_mode=task_mode)

    def _make_layer(self, block, planes, num_blocks, stride, prune_reg='weight', task_mode='harp_prune'):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, self.conv_layer, stride, prune_reg=prune_reg, task_mode=task_mode))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        if self.normalize:
            x = (x - self.mean) / self.std
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        if self.normalize_logits:
            out = out - out.mean(dim=-1, keepdim=True)
            out_norms = out.norm(dim=-1, keepdim=True)
            out_norms = torch.max(out_norms, 10**-10 * torch.ones_like(out_norms))
            out = out / out_norms
        return out

class LogitNormalizationWrapper(nn.Module):
    def __init__(self, model, normalize_logits=False):
        super(LogitNormalizationWrapper, self).__init__()
        self.model = model
        self.normalize_logits = normalize_logits

    def forward(self, x):
        out = self.model(x)
        if self.normalize_logits:
            out = out - out.mean(dim=-1, keepdim=True)
            out_norms = out.norm(dim=-1, keepdim=True)
            out_norms = torch.max(out_norms, 10**-10 * torch.ones_like(out_norms))
            out = out / out_norms
        return out

# NOTE: Only supporting default (kaiming_init) initializaition.
def resnet18_subnet(n_cls, model_width=64, normalize_features=False, normalize_logits=False, task_mode=None):
    return ResNet(SubnetConv, SubnetLinear, BasicBlock, [2, 2, 2, 2],num_classes=n_cls, normalize_features=normalize_features, normalize_logits=normalize_logits, mean=None, std=None, task_mode=task_mode)
