import torch
import torch.nn as nn
import torch.nn.functional as F

# Dataset normalization placeholder
class Normalize(nn.Module):
    def __init__(self, mu, std):
        super(Normalize, self).__init__()
        self.mu = mu
        self.std = std

    def forward(self, x):
        return (x - self.mu) / self.std


def make_layers(cfg, batch_norm=True):
    """
    Build VGG feature extractor (conv-BN-ReLU blocks and max pooling) from cfg list.
    """
    layers = []
    in_channels = 3
    for v in cfg:
        if v == 'M':
            layers.append(nn.MaxPool2d(kernel_size=2, stride=2))
        else:
            layers.append(nn.Conv2d(in_channels, v, kernel_size=3, padding=1, bias=False))
            if batch_norm:
                layers.append(nn.BatchNorm2d(v))
            layers.append(nn.ReLU(inplace=True))
            in_channels = v
    return nn.Sequential(*layers)

# VGG configuration dictionaries
cfgs = {
    '16': [64, 64, 'M', 128, 128, 'M',
            256, 256, 256, 'M',
            512, 512, 512, 'M',
            512, 512, 512]
}

class VGG(nn.Module):
    def __init__(
        self,
        cfg,
        n_cls=10,
        batch_norm=True,
        normalize_features=False,
        normalize_logits=False
    ):
        super(VGG, self).__init__()
        # Simple dataset normalization (identity by default)
        self.mu = torch.tensor((0.0, 0.0, 0.0)).view(1, 3, 1, 1)
        self.std = torch.tensor((1.0, 1.0, 1.0)).view(1, 3, 1, 1)
        self.normalize = Normalize(self.mu, self.std)

        self.normalize_features = normalize_features
        self.normalize_logits = normalize_logits

        # Feature extractor
        self.features = make_layers(cfg, batch_norm=batch_norm)
        self.avgpool = nn.AdaptiveAvgPool2d((2, 2))

        # Classifier head
        last = [x for x in cfg if isinstance(x, int)][-1]
        self.classifier = nn.Sequential(
            nn.Linear(last * 2 * 2, 256, bias=True),
            nn.ReLU(True),
            nn.Linear(256, 256, bias=True),
            nn.ReLU(True),
            nn.Linear(256, n_cls, bias=True)
        )

    def forward(self, x):
        # Convolutional features
        x = self.features(x)
        # Pool & flatten
        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        # Optional feature normalization
        if self.normalize_features:
            norms = x.norm(dim=1, keepdim=True).clamp(min=1e-10)
            x = x / norms
        # Classifier
        x = self.classifier(x)
        # Optional logit normalization
        if self.normalize_logits:
            x = x - x.mean(dim=1, keepdim=True)
            norms = x.norm(dim=1, keepdim=True).clamp(min=1e-10)
            x = x / norms
        return x

# Factory functions

class LogitNormalizationWrapper(nn.Module):
    def __init__(self, model, normalize_logits=False):
        super(LogitNormalizationWrapper, self).__init__()
        self.model = model
        self.normalize_logits = normalize_logits

    def forward(self, x):
        out = self.model(x)
        if self.normalize_logits:
            out = out - out.mean(dim=-1, keepdim=True)
            out_norms = out.norm(dim=-1, keepdim=True)
            out_norms = torch.max(out_norms, 10**-10 * torch.ones_like(out_norms))
            out = out / out_norms
        return out



def vgg16(
    n_cls: int = 10,
    batch_norm: bool = False,
    normalize_features: bool = False,
    normalize_logits: bool = False
):
    """
    VGG-16 model adapted for sharpness computation,
    with optional batch-norm, feature, and logit normalization.
    """
    cfg = cfgs['16']
    return VGG(
        cfg,
        n_cls=n_cls,
        batch_norm=batch_norm,
        normalize_features=normalize_features,
        normalize_logits=normalize_logits
    )




def vgg16_bn(
    n_cls: int = 10,
    normalize_features: bool = False,
    normalize_logits: bool = False
):
    return vgg16(
        n_cls=n_cls,
        batch_norm=True,
        normalize_features=normalize_features,
        normalize_logits=normalize_logits
    )
