import torch
import torch.nn as nn
from models.layers import SubnetConv, SubnetLinear


def make_layers(cfg, conv_layer, prune_reg='weight', task_mode='harp_prune', batch_norm=True):
    """
    Build VGG feature extractor using the provided conv_layer (e.g., SubnetConv).
    """
    layers = []
    in_channels = 3
    for v in cfg:
        if v == 'M':
            layers.append(nn.MaxPool2d(kernel_size=2, stride=2))
        else:
            conv2d = conv_layer(in_channels, v, kernel_size=3, padding=1,
                                 bias=False, prune_reg=prune_reg, task_mode=task_mode)
            if batch_norm:
                layers += [conv2d, nn.BatchNorm2d(v), nn.ReLU(inplace=True)]
            else:
                layers += [conv2d, nn.ReLU(inplace=True)]
            in_channels = v
    return nn.Sequential(*layers)


cfgs = {
    '16': [64, 64, 'M', 128, 128, 'M',
            256, 256, 256, 'M',
            512, 512, 512, 'M',
            512, 512, 512]
}


class VGG(nn.Module):
    def __init__(
        self,
        features: nn.Module,
        last_conv_channels: int,
        linear_layer,
        n_cls: int = 10,
        prune_reg: str = 'weight',
        task_mode: str = 'harp_prune',
        normalize: bool = False,
        normalize_features: bool = False,
        normalize_logits: bool = False
    ):
        super(VGG, self).__init__()
        self.features = features
        self.avgpool = nn.AdaptiveAvgPool2d((2, 2))
        self.normalize = normalize
        self.normalize_features = normalize_features
        self.normalize_logits = normalize_logits
        self.classifier = nn.Sequential(
            linear_layer(last_conv_channels * 2 * 2, 256,
                         prune_reg=prune_reg, task_mode=task_mode),
            nn.ReLU(True),
            linear_layer(256, 256,
                         prune_reg=prune_reg, task_mode=task_mode),
            nn.ReLU(True),
            linear_layer(256, n_cls,
                         prune_reg=prune_reg, task_mode=task_mode),
        )

    def forward(self, x):
        # Optional input normalization (e.g., dataset mean/std)
        if self.normalize:
            # assume input already normalized externally or mean/std attributes removed
            x = x
        # Feature extraction
        x = self.features(x)
        # Optional feature normalization
        if self.normalize_features:
            # normalize feature vectors to unit norm per sample
            f = x.flatten(1)
            norms = f.norm(dim=1, keepdim=True).clamp(min=1e-10)
            f = f / norms
            x = f.view_as(x)
        # Pooling and flatten
        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        # Classification
        x = self.classifier(x)
        # Optional logit normalization
        if self.normalize_logits:
            x = x - x.mean(dim=1, keepdim=True)
            norms = x.norm(dim=1, keepdim=True).clamp(min=1e-10)
            x = x / norms
        return x


# Factory functions

def vgg16_subnet(
    n_cls: int,
    prune_reg: str = 'weight',
    task_mode: str = 'harp_prune',
    normalize: bool = False,
    normalize_features: bool = False,
    normalize_logits: bool = False
):
    """
    VGG-16 using SubnetConv and SubnetLinear for sharpness/pruning tasks.
    """
    # last conv channel count for cfg 16
    last = [v for v in cfgs['16'] if isinstance(v, int)][-1]
    features = make_layers(cfgs['16'], SubnetConv,
                           prune_reg=prune_reg, task_mode=task_mode,
                           batch_norm=False)
    model = VGG(
        features,
        last_conv_channels=last,
        linear_layer=SubnetLinear,
        n_cls=n_cls,
        prune_reg=prune_reg,
        task_mode=task_mode,
        normalize=normalize,
        normalize_features=normalize_features,
        normalize_logits=normalize_logits
    )
    return model

class LogitNormalizationWrapper(nn.Module):
    def __init__(self, model, normalize_logits=False):
        super(LogitNormalizationWrapper, self).__init__()
        self.model = model
        self.normalize_logits = normalize_logits

    def forward(self, x):
        out = self.model(x)
        if self.normalize_logits:
            out = out - out.mean(dim=-1, keepdim=True)
            out_norms = out.norm(dim=-1, keepdim=True)
            out_norms = torch.max(out_norms, 10**-10 * torch.ones_like(out_norms))
            out = out / out_norms
        return out


def vgg16_bn_subnet(
    n_cls: int,
    prune_reg: str = 'weight',
    task_mode: str = 'harp_prune',
    normalize: bool = False,
    normalize_features: bool = False,
    normalize_logits: bool = False
):
    last = [v for v in cfgs['16'] if isinstance(v, int)][-1]
    features = make_layers(cfgs['16'], SubnetConv,
                           prune_reg=prune_reg, task_mode=task_mode,
                           batch_norm=True)
    model = VGG(
        features,
        last_conv_channels=last,
        linear_layer=SubnetLinear,
        n_cls=n_cls,
        prune_reg=prune_reg,
        task_mode=task_mode,
        normalize=normalize,
        normalize_features=normalize_features,
        normalize_logits=normalize_logits
    )
    return model
