import math
import torch
import torch.nn as nn
import torch.nn.functional as F


class Normalize(nn.Module):
    def __init__(self, mu, std):
        super(Normalize, self).__init__()
        self.mu = mu
        self.std = std

    def forward(self, x):
        return (x - self.mu) / self.std


class BasicBlockWRN(nn.Module):
    expansion = 1

    def __init__(self, in_planes, out_planes, stride=1, dropRate=0.0):
        super(BasicBlockWRN, self).__init__()
        # BatchNorm and ReLU for pre-activation
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.relu1 = nn.ReLU(inplace=True)
        # First conv layer
        self.conv1 = nn.Conv2d(
            in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False
        )
        # BatchNorm and ReLU after first conv
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu2 = nn.ReLU(inplace=True)
        # Second conv layer
        self.conv2 = nn.Conv2d(
            out_planes, out_planes, kernel_size=3, stride=1, padding=1, bias=False
        )
        self.droprate = dropRate
        # Determine if we need a shortcut conv
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = None
        if not self.equalInOut:
            self.convShortcut = nn.Conv2d(
                in_planes, out_planes, kernel_size=1, stride=stride, bias=False
            )

    def forward(self, x):
        original_x = x
        # Pre-activation
        if not self.equalInOut:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        # Select activation for conv1
        preact = out if self.equalInOut else x
        # Residual path
        out = self.relu2(self.bn2(self.conv1(preact)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        # Shortcut connection
        if self.equalInOut:
            skip = original_x
        else:
            skip = self.convShortcut(x)
        return skip + out

class NetworkBlock(nn.Module):
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock, self).__init__()
        layers = []
        for i in range(int(nb_layers)):
            s = stride if i == 0 else 1
            ip = in_planes if i == 0 else out_planes
            layers.append(block(ip, out_planes, s, dropRate))
        self.layer = nn.Sequential(*layers)

    def forward(self, x):
        return self.layer(x)


class WideResNet(nn.Module):
    def __init__(
        self,
        depth=28,
        n_cls=10,
        widen_factor=4,
        dropRate=0.0,
        normalize_features=False,
        normalize_logits=False,
    ):
        super(WideResNet, self).__init__()
        assert (depth - 4) % 6 == 0, "Depth must be 6n+4"
        n = (depth - 4) // 6
        nChannels = [16, 16 * widen_factor, 32 * widen_factor, 64 * widen_factor]

        # Dataset normalization (identity)
        self.mu = torch.tensor((0.0, 0.0, 0.0)).view(1, 3, 1, 1)
        self.std = torch.tensor((1.0, 1.0, 1.0)).view(1, 3, 1, 1)
        self.normalize = Normalize(self.mu, self.std)
        self.normalize_features = normalize_features
        self.normalize_logits = normalize_logits

        # Initial conv
        self.conv1 = nn.Conv2d(
            3, nChannels[0], kernel_size=3, stride=1, padding=1, bias=False
        )
        # Wide ResNet blocks (match saved model naming)
        self.block1 = NetworkBlock(n, nChannels[0], nChannels[1], BasicBlockWRN, stride=1, dropRate=dropRate)
        self.block2 = NetworkBlock(n, nChannels[1], nChannels[2], BasicBlockWRN, stride=2, dropRate=dropRate)
        self.block3 = NetworkBlock(n, nChannels[2], nChannels[3], BasicBlockWRN, stride=2, dropRate=dropRate)
        self.bn1 = nn.BatchNorm2d(nChannels[3])
        self.relu = nn.ReLU(inplace=True)
        self.nChannels = nChannels[3]
        self.fc = nn.Linear(self.nChannels, n_cls)

        # Weight initialization
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n_val = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2.0 / n_val))
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()
            elif isinstance(m, nn.Linear) and m.bias is not None:
                m.bias.data.zero_()

    def forward(self, x):
        # Optional input normalization
        out = self.conv1(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn1(out))
        out = F.avg_pool2d(out, 8)
        out = out.view(out.size(0), -1)
        # Optional feature normalization
        if self.normalize_features:
            out = out / out.norm(dim=-1, keepdim=True).clamp(min=1e-10)
        logits = self.fc(out)
        # Optional logit normalization
        if self.normalize_logits:
            logits = logits - logits.mean(dim=-1, keepdim=True)
            norms = logits.norm(dim=-1, keepdim=True).clamp(min=1e-10)
            logits = logits / norms
        return logits


class LogitNormalizationWrapper(nn.Module):
    def __init__(self, model, normalize_logits=False):
        super(LogitNormalizationWrapper, self).__init__()
        self.model = model
        self.normalize_logits = normalize_logits

    def forward(self, x):
        out = self.model(x)
        if self.normalize_logits:
            out = out - out.mean(dim=-1, keepdim=True)
            norms = out.norm(dim=-1, keepdim=True).clamp(min=1e-10)
            out = out / norms
        return out


# Factory function

def wrn_28_4(n_cls, widen_factor=4, dropRate=0.0, normalize_features=False, normalize_logits=False):
    """
    Modified WRN-28-4 matching saved model structure (block1/block2/block3), no pruning.
    """
    return WideResNet(
        depth=28,
        n_cls=n_cls,
        widen_factor=widen_factor,
        dropRate=dropRate,
        normalize_features=normalize_features,
        normalize_logits=normalize_logits,
    )
