"""
The MIT License (MIT) Copyright (c) 2020 Andrej Karpathy

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import random
import numpy as np
import torch
from torch.nn import functional as F

target_return_mapping = {"Breakout": 90, "Seaquest": 1150, "Qbert": 14000, "Pong": 20}


def set_seed(seed):
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)


def get_normalized_score(game_name, raw_score):
    if game_name == "Breakout":
        random_score, expert_score = 2, 30
    elif game_name == "Qbert":
        random_score, expert_score = 164, 13455
    elif game_name == "Pong":
        random_score, expert_score = -21, 15
    elif game_name == "Seaquest":
        random_score, expert_score = 68, 42055
    normalized_score = (100 * (raw_score - random_score)) / (
        expert_score - random_score
    )
    return normalized_score


def top_k_logits(logits, k):
    v, ix = torch.topk(logits, k)
    out = logits.clone()
    out[out < v[:, [-1]]] = -float("Inf")
    return out


@torch.no_grad()
def sample(
    model,
    x,
    steps,
    temperature=1.0,
    sample=False,
    top_k=None,
    actions=None,
    rtgs=None,
    timesteps=None,
):
    """
    take a conditioning sequence of indices in x (of shape (b,t)) and predict the next token in
    the sequence, feeding the predictions back into the model each time. Clearly the sampling
    has quadratic complexity unlike an RNN that is only linear, and has a finite context window
    of block_size, unlike an RNN that has an infinite context window.
    """
    block_size = model.get_block_size()
    model.eval()
    for k in range(steps):
        # x_cond = x if x.size(1) <= block_size else x[:, -block_size:] # crop context if needed
        x_cond = (
            x if x.size(1) <= block_size // 3 else x[:, -block_size // 3 :]
        )  # crop context if needed
        if actions is not None:
            actions = (
                actions
                if actions.size(1) <= block_size // 3
                else actions[:, -block_size // 3 :]
            )  # crop context if needed
        rtgs = (
            rtgs if rtgs.size(1) <= block_size // 3 else rtgs[:, -block_size // 3 :]
        )  # crop context if needed
        logits, _ = model(
            x_cond, actions=actions, targets=None, rtgs=rtgs, timesteps=timesteps
        )
        # pluck the logits at the final step and scale by temperature
        logits = logits[:, -1, :] / temperature
        # optionally crop probabilities to only the top k options
        if top_k is not None:
            logits = top_k_logits(logits, top_k)
        # apply softmax to convert to probabilities
        probs = F.softmax(logits, dim=-1)
        # sample from the distribution or take the most likely
        if sample:
            ix = torch.multinomial(probs, num_samples=1)
        else:
            _, ix = torch.topk(probs, k=1, dim=-1)
        # append to the sequence and continue
        # x = torch.cat((x, ix), dim=1)
        x = ix

    return x
