from collections import deque
import numpy as np
import random
import torch

from wrappers.atari_wrapper import LazyFrames
from dataset.expert_dataset import ExpertDataset
from utils.utils import get_concat_samples


class Memory(object):
    def __init__(self, memory_size: int, seed: int = 0) -> None:
        random.seed(seed)
        self.memory_size = memory_size
        self.buffer = deque(maxlen=self.memory_size)
        
    def resize(self, memory_size):
        self.new_buffer = deque(maxlen=memory_size)
        for i in range(len(self.buffer)):
            self.new_buffer.append(self.buffer[i])
        self.buffer = self.new_buffer

    def add(self, experience) -> None:
        self.buffer.append(experience)

    def size(self):
        return len(self.buffer)

    def sample(self, batch_size: int, continuous: bool = True):
        if batch_size > len(self.buffer):
            batch_size = len(self.buffer)
        if continuous:
            rand = random.randint(0, len(self.buffer) - batch_size)
            return [self.buffer[i] for i in range(rand, rand + batch_size)]
        else:
            indexes = np.random.choice(np.arange(len(self.buffer)), size=batch_size, replace=False)
            return [self.buffer[i] for i in indexes]

    def clear(self):
        self.buffer.clear()

    def save(self, path):
        b = np.asarray(self.buffer)
        print(b.shape)
        np.save(path, b)

    def load(self, path, num_trajs, sample_freq, seed):
        # If path has no extension add npy
        if not path.endswith("pkl"):
            path += '.npy'
        data = ExpertDataset(path, num_trajs, sample_freq, seed)
        # data = np.load(path, allow_pickle=True)
        for i in range(len(data)):
            self.add(data[i])

    def get_samples(self, batch_size, device):
        batch = self.sample(batch_size, False)

        batch_state, batch_next_state, batch_action, batch_reward, batch_done = zip(
            *batch)

        # Scale obs for atari. TODO: Use flags
        if isinstance(batch_state[0], LazyFrames):
            # Use lazyframes for improved memory storage (same as original DQN)
            batch_state = np.array(batch_state) / 255.0
        if isinstance(batch_next_state[0], LazyFrames):
            batch_next_state = np.array(batch_next_state) / 255.0
        batch_state = np.array(batch_state)
        batch_next_state = np.array(batch_next_state)
        batch_action = np.array(batch_action)

        batch_state = torch.as_tensor(batch_state, dtype=torch.float, device=device)
        batch_next_state = torch.as_tensor(batch_next_state, dtype=torch.float, device=device)
        batch_action = torch.as_tensor(batch_action, dtype=torch.float, device=device)
        if batch_action.ndim == 1:
            batch_action = batch_action.unsqueeze(1)
        batch_reward = torch.as_tensor(batch_reward, dtype=torch.float, device=device).unsqueeze(1)
        batch_done = torch.as_tensor(batch_done, dtype=torch.float, device=device).unsqueeze(1)

        return batch_state, batch_next_state, batch_action, batch_reward, batch_done
    

    def sample_all(self):
        batch = [self.buffer[i] for i in range(self.size())]

        batch_state, batch_next_state, batch_action, batch_reward, batch_done = zip(
            *batch)

        # Scale obs for atari. TODO: Use flags
        if isinstance(batch_state[0], LazyFrames):
            # Use lazyframes for improved memory storage (same as original DQN)
            batch_state = np.array(batch_state) / 255.0
        if isinstance(batch_next_state[0], LazyFrames):
            batch_next_state = np.array(batch_next_state) / 255.0
        batch_state = np.array(batch_state).astype(np.float32)
        batch_next_state = np.array(batch_next_state).astype(np.float32)
        batch_action = np.array(batch_action).astype(np.float32)
        batch_reward = np.array(batch_reward).astype(np.float32).reshape(-1, 1)
        batch_done = np.array(batch_done).astype(np.float32).reshape(-1, 1)

        return batch_state, batch_next_state, batch_action, batch_reward, batch_done

    def get_sample_np(self, batch_size):
        batch = self.sample(batch_size, False)

        batch_state, batch_next_state, batch_action, batch_reward, batch_done = zip(
            *batch)

        # Scale obs for atari. TODO: Use flags
        if isinstance(batch_state[0], LazyFrames):
            # Use lazyframes for improved memory storage (same as original DQN)
            batch_state = np.array(batch_state) / 255.0
        if isinstance(batch_next_state[0], LazyFrames):
            batch_next_state = np.array(batch_next_state) / 255.0
        batch_state = np.array(batch_state).astype(np.float32)
        batch_next_state = np.array(batch_next_state).astype(np.float32)
        batch_action = np.array(batch_action).astype(np.float32)
        batch_reward = np.array(batch_reward).astype(np.float32).reshape(-1, 1)
        batch_done = np.array(batch_done).astype(np.float32).reshape(-1, 1)

        return batch_state, batch_next_state, batch_action, batch_reward, batch_done

    def get_samples_np(self, batch_size):
        if batch_size < len(self.buffer):
            return self.get_sample_np(batch_size)[:5]
        elif batch_size == len(self.buffer):
            return self.sample_all()
        else:
            data = self.sample_all()
            return get_concat_samples(data, self.get_samples_np(batch_size - len(self.buffer)))[:5]
