import random
from . import tone_sentences

def realistic_misspell(word):
    """
    Misspells a single word.
    """
    if len(word) <= 1:  # Skip very short words to keep the text readable
        return word

    misspelling_type = random.choice(['swap', 'drop', 'repeat', 'replace', 'shuffle_middle'])

    if misspelling_type == 'swap' and len(word) > 1:
        # Swap two adjacent letters
        i = random.randint(0, len(word) - 2)
        return word[:i] + word[i+1] + word[i] + word[i+2:]
    
    elif misspelling_type == 'drop':
        # Drop a random letter
        i = random.randint(0, len(word) - 1)
        return word[:i] + word[i+1:]
    
    elif misspelling_type == 'repeat':
        # Repeat a random letter
        i = random.randint(0, len(word) - 1)
        return word[:i] + word[i] + word[i:]
    
    elif misspelling_type == 'replace':
        # Replace a letter with a nearby letter (simulating a typo)
        keyboard_neighbors = {
            'a': 'qs', 'b': 'vn', 'c': 'vx', 'd': 'sf', 'e': 'wr', 'f': 'dg', 'g': 'fh',
            'h': 'gj', 'i': 'uo', 'j': 'hk', 'k': 'jl', 'l': 'k', 'm': 'n', 'n': 'bm',
            'o': 'ip', 'p': 'o', 'q': 'wa', 'r': 'et', 's': 'ad', 't': 'ry', 'u': 'yi',
            'v': 'cb', 'w': 'qe', 'x': 'zc', 'y': 'tu', 'z': 'x'
        }
        i = random.randint(0, len(word) - 1)
        original_letter = word[i].lower()
        if original_letter in keyboard_neighbors:
            replacement = random.choice(keyboard_neighbors[original_letter])
            return word[:i] + replacement + word[i+1:]
        else:
            return word  # If no suitable replacement, leave it as is
    
    elif misspelling_type == 'shuffle_middle' and len(word) > 3:
        # Shuffle the middle letters of the word
        middle = list(word[1:-1])
        random.shuffle(middle)
        return word[0] + ''.join(middle) + word[-1]
    
    return word


def add_tone(sentence, tone, seed=100):
    """
    Takes a sentence and randomly select a sentence with the given tone to add to the beginning.
    """
    if seed is not None:
        random.seed(seed)
    all_sentences = tone_sentences.tone_dict[tone]
    selected = random.choice(all_sentences)
    return selected + '\n' + sentence



def misspell_sentence(sentence, fraction=0.2, seed=100):
    """
    Takes a sentence and randomly misspells a given fraction of its words.
    Optionally accepts a random seed for reproducible output.
    """
    if seed is not None:
        random.seed(seed)
    words = sentence.split()
    num_words = len(words)
    num_to_misspell = int(num_words * fraction)
    indices_to_misspell = random.sample(range(num_words), num_to_misspell)
    
    for idx in indices_to_misspell:
        words[idx] = realistic_misspell(words[idx])
    return " ".join(words)


def mess_up_spacing_and_punctuation(sentence, fraction=0.2, seed=100):
    """
    Takes a sentence and messes up its spacing and punctuation like how a foreigner might do.
    Optionally accepts a random seed for reproducible output.
    """
    if seed is not None:
        random.seed(seed)
    punctuation_marks = ['.', ',', '!', '?', ';', ':']
    words = sentence.split()
    num_words = len(words)
    num_changes = int(num_words * fraction)
    
    # Randomly decide what type of error to introduce
    for _ in range(num_changes):
        change_type = random.choice(['extra_space', 'missing_space', 'extra_punctuation', 'remove_punctuation'])
        if change_type == 'extra_space' and len(words) > 1:
            # Add extra spaces between words
            index = random.randint(0, len(words) - 2)
            words[index] += ' ' * random.randint(1, 3)
        elif change_type == 'missing_space' and len(words) > 1:
            # Remove space between two words
            index = random.randint(0, len(words) - 2)
            words[index] += words[index + 1]
            del words[index + 1]
        elif change_type == 'extra_punctuation':
            # Add random punctuation to a random word
            index = random.randint(0, len(words) - 1)
            words[index] += random.choice(punctuation_marks) * random.randint(1, 3)
        elif change_type == 'remove_punctuation':
            # Remove punctuation from a random word
            index = random.randint(0, len(words) - 1)
            words[index] = ''.join(char for char in words[index] if char not in punctuation_marks)
    
    return ' '.join(words)


# "Augment prompts"

# import random
# from deep_translator import MyMemoryTranslator


# def realistic_misspell(word):
#     """
#     Misspells a single word.
#     """
#     if len(word) <= 1:  # Skip very short words to keep the text readable
#         return word
    
#     misspelling_type = random.choice(['swap', 'drop', 'repeat', 'replace', 'shuffle_middle'])
    
#     if misspelling_type == 'swap' and len(word) > 1:
#         # Swap two adjacent letters
#         i = random.randint(0, len(word) - 2)
#         return word[:i] + word[i+1] + word[i] + word[i+2:]
    
#     elif misspelling_type == 'drop':
#         # Drop a random letter
#         i = random.randint(0, len(word) - 1)
#         return word[:i] + word[i+1:]
    
#     elif misspelling_type == 'repeat':
#         # Repeat a random letter
#         i = random.randint(0, len(word) - 1)
#         return word[:i] + word[i] + word[i:]
    
#     elif misspelling_type == 'replace':
#         # Replace a letter with a nearby letter (simulating a typo)
#         keyboard_neighbors = {
#             'a': 'qs', 'b': 'vn', 'c': 'vx', 'd': 'sf', 'e': 'wr', 'f': 'dg', 'g': 'fh',
#             'h': 'gj', 'i': 'uo', 'j': 'hk', 'k': 'jl', 'l': 'k', 'm': 'n', 'n': 'bm',
#             'o': 'ip', 'p': 'o', 'q': 'wa', 'r': 'et', 's': 'ad', 't': 'ry', 'u': 'yi',
#             'v': 'cb', 'w': 'qe', 'x': 'zc', 'y': 'tu', 'z': 'x'
#         }
#         i = random.randint(0, len(word) - 1)
#         original_letter = word[i].lower()
#         if original_letter in keyboard_neighbors:
#             replacement = random.choice(keyboard_neighbors[original_letter])
#             return word[:i] + replacement + word[i+1:]
#         else:
#             return word  # If no suitable replacement, leave it as is
    
#     elif misspelling_type == 'shuffle_middle' and len(word) > 3:
#         # Shuffle the middle letters of the word
#         middle = list(word[1:-1])
#         random.shuffle(middle)
#         return word[0] + ''.join(middle) + word[-1]
    
#     return word


# def misspell_sentence(sentence, fraction=0.2):
#     """
#     Takes a sentence and randomly misspells a given fraction of its words.
#     """
#     words = sentence.split()
#     num_words = len(words)
#     num_to_misspell = int(num_words * fraction)
#     indices_to_misspell = random.sample(range(num_words), num_to_misspell)
    
#     for idx in indices_to_misspell:
#         words[idx] = realistic_misspell(words[idx])
#     return " ".join(words)


# def mess_up_spacing_and_punctuation(sentence, fraction=0.2):
#     """
#     Takes a sentence and messes up its spacing and punctuation like how a foreigner might do.
#     """
#     punctuation_marks = ['.', ',', '!', '?', ';', ':']
#     words = sentence.split()
#     num_words = len(words)
#     num_changes = int(num_words * fraction)
    
#     # Randomly decide what type of error to introduce
#     for _ in range(num_changes):
#         change_type = random.choice(['extra_space', 'missing_space', 'extra_punctuation', 'remove_punctuation'])
#         if change_type == 'extra_space' and len(words) > 1:
#             # Add extra spaces between words
#             index = random.randint(0, len(words) - 2)
#             words[index] += ' ' * random.randint(1, 3)
#         elif change_type == 'missing_space' and len(words) > 1:
#             # Remove space between two words
#             index = random.randint(0, len(words) - 2)
#             words[index] += words[index + 1]
#             del words[index + 1]
#         elif change_type == 'extra_punctuation':
#             # Add random punctuation to a random word
#             index = random.randint(0, len(words) - 1)
#             words[index] += random.choice(punctuation_marks) * random.randint(1, 3)
#         elif change_type == 'remove_punctuation':
#             # Remove punctuation from a random word
#             index = random.randint(0, len(words) - 1)
#             words[index] = ''.join(char for char in words[index] if char not in punctuation_marks)
    
#     messed_up_sentence = ' '.join(words)    
#     return messed_up_sentence


# # Example usage:
# if __name__ == "__main__":
#     original_sentence = "hello everyone, my name is john, did you guys eat breakfast yet?"
#     messed_up = partial_translate(original_sentence,
#                       'chinese')
#     print("Original:", original_sentence)
#     print("Messed up:", messed_up)
