#!/bin/bash
# usage: bash process/sh [steps] [graph name]
# needs graphs/[graph name]/graph.txt and graphs/[graph name]/graph.csv

# extra env variables:
# set SEED to change the RNG seed
# set PROCESS_DRY to just output the commands, do not execute them

GRAPH=$2
IDIR=graphs/$GRAPH/
DIR=results/$GRAPH/
DHRG="hyperrogue/hyper -c - -fixx 42 -nogui -s -"
DHRGSEED="hyperrogue/hyper -c - -fixx 42$SEED -nogui -s -"

if [ -n "$SEED" ]
then DIR=results/repeated/$GRAPH$SEED/
GRAPH=r$SEED/$GRAPH
fi
echo "#" DIR is \'$DIR\', SEED is \'$SEED\'

function log {
  echo 1>&2
  echo "######" $GRAPH " : " $1 1>&2
  echo 1>&2
  echo `TZ=Universal date` : $1 >> $DIR/log.txt
  }

function logdo {
  if [ -n "$PROCESS_DRY" ]
  then echo "#" log message: $1
  shift
  echo "$@"
  else  
  LOGMSG="$1"
  shift
  log "$LOGMSG"
  "$@"
  log "$LOGMSG: finished"
  fi
  }

function indir {
  pushd . >/dev/null
  cd $1
  shift
  "$@"
  popd >/dev/null
  }

function poincare {
  indir poincare-embeddings python3 "$@"
  }

function vfile {
  if [[ -f $1 ]]
  then true
  else echo "File missing: $1"; false
  fi
  }

mkdir -p $DIR

# for every prefix (embedder) we can have the following operation names:
# embed -- call the embedder
# enk -- evaluation using NK software
# eval -- full evaluation using our framework
# conv -- convert to the BFKL format
# dhrg -- apply DHRG to improve the embedding
# deval -- discrete evaluation
# ls50 -- convert to a landscape embedding (not explained in the paper, see README.txt)
# ls200 -- convert to a landscape embedding (not explained in the paper, see README.txt)
# penalty -- penalty variant of DHRG

# b: BFKL embedder

if [[ $1 == *-b-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using BFKL" \
hyperbolic-embedder/embedder -seed 123456$SEED -input $IDIR/graph.txt -embed $DIR/bfkl | tee $DIR/log-bfkl.txt
fi

if [[ $1 == *-b-eval-* ]]
then vfile $DIR/bfkl-coordinates.txt && logdo "evaluate bfkl fulleval" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-bfkl-fulleval.txt
fi

if [[ $1 == *-b-dhrg-* ]]
then vfile $DIR/bfkl-coordinates.txt && logdo "improve BFKL" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-bfkl-improve.txt
fi

if [[ $1 == *-b-dgreedy-* ]]
then vfile $DIR/bfkl-dhrg.txt && logdo "evaluate dhrg discrete greedy routing " \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -routing-qdist | tee $DIR/log-dhrg-bfkl-greedy.txt
fi

if [[ $1 == *-b-deval-* ]]
then vfile $DIR/bfkl-dhrg.txt && logdo "evaluate ddhrg bfkl fulleval" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -cellcoords -eval-hdist -dhrg-to-polar $DIR/dhrg-polar.txt | tee $DIR/log-ddhrg-bfkl-fulleval.txt
fi

if [[ $1 == *-b-ls200-* ]]
then vfile $DIR/bfkl-dhrg.txt && logdo "landscape 200D" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -landscape-rank 200 | tee $DIR/log-landscape-200.txt
fi

if [[ $1 == *-b-ls50-* ]]
then vfile $DIR/bfkl-dhrg.txt && logdo "landscape 50D" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -landscape-rank 50 | tee $DIR/log-landscape-50.txt
fi

if [[ $1 == *-b-penalty-* ]]
then vfile $DIR/bfkl-coordinates.txt && logdo "penalty test" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -penalty 2 -iterate 100 -cellcoords -eval-hdist | tee $DIR/log-penalty2-fulleval.txt
fi

# l2: Lorentz 2D embedder by NK

if [[ $1 == *-l2-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Lorentz 2D" poincare embed.py \
  -checkpoint ../$DIR/lorentz2.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold lorentz \
  -dim 2 | tee $DIR/log-lorentz2.txt
fi

if [[ $1 == *-l2-enk-* ]]
then vfile $DIR/lorentz2.bin.best && logdo "evaluate Lorentz 2D" \
poincare kx-evaluate.py ../$DIR/lorentz2.bin.best -rank 1 | tee $DIR/log-lorentz2-evaluate.txt
fi

if [[ $1 == *-l2-conv-* ]]
then vfile $DIR/lorentz2.bin.best && logdo "convert Lorentz 2D" \
poincare kx-evaluate.py ../$DIR/lorentz2.bin.best -output ../$DIR/lorentz2-coordinates.txt | tee $DIR/log-lorentz2-convert.txt
fi

if [[ $1 == *-l2-eval-* ]]
then vfile $DIR/lorentz2-coordinates.txt && vfile $DIR/bfkl-links.txt && logdo "evaluate lorentz2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-lorentz2-fulleval.txt
fi

if [[ $1 == *-l2-dhrg-* ]]
then vfile $DIR/lorentz2-coordinates.txt && vfile $DIR/bfkl-links.txt && logdo "improve Lorentz 2D" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/lorentz2 -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-lorentz2-improve.txt
fi

if [[ $1 == *-l2-deval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/bfkl-links.txt && logdo "evaluate ddhrg lorentz2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -cellcoords -eval-hdist | tee $DIR/log-ddhrg-lorentz2-fulleval.txt
fi

# l3: Lorentz 3D embedder by NK

if [[ $1 == *-l3-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Lorentz 3D" poincare embed.py \
  -checkpoint ../$DIR/lorentz3.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold lorentz \
  -dim 3 | tee $DIR/log-lorentz3.txt
fi

if [[ $1 == *-l3-conv-* ]]
then vfile $DIR/lorentz2.bin.best && logdo "convert Lorentz 3D" poincare kx-evaluate.py ../$DIR/lorentz3.bin.best -output3 ../$DIR/lorentz3.txt | tee $DIR/log-lorentz3-convert.txt
fi

if [[ $1 == *-l3-eval-* ]]
then vfile $DIR/lorentz3.txt && vfile $DIR/lorentz2-dhrg.txt && logdo "evaluate lorentz3 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-lorentz3 $DIR/lorentz3.txt | tee $DIR/log-lorentz3-fulleval.txt
fi

# p2: Poincare 2D embedder by NK

if [[ $1 == *-p2-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Poincare 2D" poincare embed.py \
  -checkpoint ../$DIR/poincare2.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 2 | tee $DIR/log-poincare2.txt
fi

if [[ $1 == *-p2-enk-* ]]
then logdo "evaluate Poincare 2D" poincare kx-evaluate.py ../$DIR/poincare2.bin.best -rank 1 | tee $DIR/log-poincare2-evaluate.txt
fi

if [[ $1 == *-p2-conv-* ]]
then logdo "convert Poincare 2D" poincare kx-evaluate.py ../$DIR/poincare2.bin.best -output3 ../$DIR/poincare2.txt | tee $DIR/log-poincare2-convert.txt
fi

if [[ $1 == *-p2-eval-* ]]
then vfile $DIR/poincare2.txt && logdo "evaluate poincare2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare2 $DIR/poincare2.txt | tee $DIR/log-poincare2-fulleval.txt
fi

if [[ $1 == *-p2-dhrg-* ]]
then vfile $DIR/poincare2.txt && vfile $DIR/bfkl-links.txt && logdo "improve Poincare 2D" \
$DHRG -alt-edges $DIR/bfkl-links.txt -alt-coord-poincare $DIR/poincare2.txt -dhrg $DIR/poincare -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-poincare2-improve.txt
fi

if [[ $1 == *-p2-deval-* ]]
then vfile $DIR/poincare-dhrg.txt && logdo "evaluate ddhrg poincare fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -alt-coord-poincare $DIR/poincare2.txt -eload $DIR/poincare -rdi $IDIR/graph.txt -cellcoords -eval-hdist -dhrg-to-polar $DIR/poincare-polar.txt | tee $DIR/log-ddhrg-poincare-fulleval.txt
fi

# p3: Poincare 3D embedder by NK

if [[ $1 == *-p3-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Poincare 3D" poincare embed.py \
  -checkpoint ../$DIR/poincare3.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 3 | tee $DIR/log-poincare3.txt
fi

if [[ $1 == *-p3-enk-* ]]
then logdo "evaluate Poincare 3D" poincare kx-evaluate.py ../$DIR/poincare3.bin.best -rank 1 | tee $DIR/log-poincare3-evaluate.txt
fi

if [[ $1 == *-p3-conv- ]]
then logdo "convert Poincare 3D" poincare kx-evaluate.py ../$DIR/poincare3.bin.best -output3 ../$DIR/poincare3.txt | tee $DIR/log-poincare3-convert.txt
fi

if [[ $1 == *-p3-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && logdo "evaluate poincare3 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare3 $DIR/poincare3.txt | tee $DIR/log-poincare3-fulleval.txt
fi

# p5: Poincare 5D embedder by NK

if [[ $1 == *-p5-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Poincare 5D" poincare embed.py \
  -checkpoint ../$DIR/poincare5.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 5 | tee $DIR/log-poincare5.txt
fi

if [[ $1 == *-p5-enk-* ]]
then logdo "evaluate Poincare 5D" poincare kx-evaluate.py ../$DIR/poincare5.bin.best -rank 1 | tee $DIR/log-poincare5-evaluate.txt
fi

function peval {
  poincare kx-evaluate.py ../$DIR/poincare$1.bin.best -output3 ../$DIR/poincare$1.txt | tee $DIR/log-poincare$1-convert.txt
  $DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-poincare-any $DIR/poincare$1.txt 5 | tee $DIR/log-poincare$1-fulleval.txt
  }

if [[ $1 == *-p5-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && logdo "evaluate poincare5 fulleval" peval 5
fi

# e50: Euclidean 50D embedder by NK

if [[ $1 == *-e50-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Euclidean 50D" poincare embed.py \
  -checkpoint ../$DIR/euclidean50.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold euclidean \
  -dim 50 | tee $DIR/log-euclidean50.txt
fi

if [[ $1 == *-e50-enk-* ]]
then logdo "evaluate Euclidean 50D" poincare kx-evaluate.py ../$DIR/euclidean50.bin.best -rank 1 | tee $DIR/log-euclidean50-evaluate.txt
fi

function eeval {
  poincare kx-evaluate.py ../$DIR/euclidean$1.bin.best -output3 ../$DIR/euclidean$1.txt | tee $DIR/log-euclidean$1-convert.txt
  $DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-euclid $DIR/euclidean$1.txt $1 | tee $DIR/log-euclidean$1-fulleval.txt
  }

if [[ $1 == *-e50-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/euclidean50.txt && logdo "evaluate euclid50 fulleval" eeval 50
fi

# e200: Euclidean 200D embedder by NK

if [[ $1 == *-e200-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Euclidean 200D" poincare embed.py \
  -checkpoint ../$DIR/euclidean200.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold euclidean \
  -dim 200 | tee $DIR/log-euclidean200.txt
fi

if [[ $1 == *-e200-enk-* ]]
then logdo "evaluate Euclidean 200D" poincare kx-evaluate.py ../$DIR/euclidean200.bin.best -rank 1 | tee $DIR/log-euclidean200-evaluate.txt
fi

if [[ $1 == *-e200-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/euclidean200.bin.best && logdo "evaluate euclid200 fulleval" eeval 200
fi

if [[ $1 == *-e50-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/euclidean50.bin.best && logdo "evaluate euclid50 fulleval" eeval 50
fi

# hv: HypViewer

function gentree {
  hypviewer/gentree < $IDIR/graph.txt > $DIR/hypviewer.txt 2> $DIR/log-hypviewer.txt
  }

if [[ $1 == *-hv-embed-* ]]
then vfile $IDIR/graph.txt && logdo "apply HypViewer" gentree
fi

if [[ $1 == *-hv-enk-* ]]
then vfile $DIR/hypviewer.txt && logdo "evaluate HypViewer" \
poincare kx-evaluate.py ../$DIR/poincare3.bin.best -input3 ../$DIR/hypviewer.txt | tee $DIR/log-evaluate-hypviewer.txt
fi

if [[ $1 == *-hv-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/bfkl-links.txt && vfile $DIR/hypviewer.txt && logdo "evaluate HypViewer using dhrg" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare3 $DIR/hypviewer.txt | tee $DIR/log-hypviewer-fulleval.txt
fi

# mfast: Mercator fast

if [[ $1 == *-mfast-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Mercator fast" mercator/mercator -f -s 123456$SEED -o $DIR/mercator-fast $IDIR/graph.txt
bin/mercator-to-bfkl < $DIR/mercator-fast.inf_coord > $DIR/mercator-fast-coordinates.txt
fi

if [[ $1 == *-mfast-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/mercator-fast-coordinates.txt && logdo "evaluate Mercator fast fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/mercator-fast -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-mercator-fast-fulleval.txt
fi

# mfull: Mercator full

if [[ $1 == *-mfull-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using Mercator full" mercator/mercator -s 123456$SEED -o $DIR/mercator-full $IDIR/graph.txt
bin/mercator-to-bfkl < $DIR/mercator-full.inf_coord > $DIR/mercator-full-coordinates.txt
fi

if [[ $1 == *-mfull-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/mercator-full-coordinates.txt && logdo "evaluate Mercator full fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/mercator-full -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-mercator-full-fulleval.txt
fi

# mdim: 3-dimensional Mercator

if [[ $1 == *-mdim-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using d-Mercator" \
d-mercator/mercator -f -s 123456$SEED -o $DIR/d-mercator $IDIR/graph.txt
fi

if [[ $1 == *-mdim-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/bfkl-links.txt && vfile $DIR/d-mercator.inf_coord && logdo "evaluate d-mercator" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-mercator3 $DIR/d-mercator.inf_coord | tee $DIR/log-d-mercator-fulleval.txt
fi

# rv: RogueViz tree visualization

function rvembed {
  hypviewer/gentree < $IDIR/graph.txt > /dev/null
  $DHRG -tree /tmp/treedata-tol.txt -write-in-dhrg-format $DIR/rogueviz-coordinates.txt | tee $DIR/log-rogueviz.txt
  }

if [[ $1 == *-rv-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using RogueViz" rvembed
fi

function rveval {
  hypviewer/gentree < $IDIR/graph.txt > /dev/null
  $DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/rogueviz -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-rogueviz-fulleval.txt
  }

if [[ $1 == *-rv-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/rogueviz-coordinates.txt && logdo "evaluate RogueViz mAP" rveval
fi

# tr: TreeRep

if [[ $1 == *-tr-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using TreeRep and evaluate" \
julia -t 8 TreeRep/experiment.jl $IDIR/graph.csv $DIR/treerep-edgelist.txt $DIR/treerep-norec.txt $DIR/treerep-wrec.txt | tee $DIR/log-treerep.txt
fi

if [[ $1 == *-tr-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/treerep-edgelist.txt && logdo "evaluate TreeReps with other measures" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-disttable $DIR/treerep-edgelist.txt $DIR/treerep-norec.txt -eval-disttable $DIR/treerep-edgelist.txt $DIR/treerep-wrec.txt | tee $DIR/log-treerep-eval.txt
fi

# lt: ltiling

if [[ $1 == *-lt-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using LTiling_rsgd" indir HyperbolicTiling_Learning python3 embed.py \
  -dset ../$IDIR/graph.csv \
  -epochs 1000 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -train_threads 2 \
  -dampening 1.0 \
  -batchsize 50 \
  -manifold LTiling_rsgd \
  -dim 2 \
  -com_n 1 \
  -lr 0.3 -no-maxnorm \
  -keys ../$DIR/ltiling-keys.txt \
  -disttable ../$DIR/ltiling-disttable.txt
# epochs=1000, train_threads=2
#       -checkpoint ../$DIR/ltiling.bin \
fi

if [[ $1 == *-lt-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/ltiling-keys.txt && logdo "evaluate LTiling_rsgd using other measures" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-disttable $DIR/ltiling-keys.txt $DIR/ltiling-disttable.txt | tee $DIR/log-ltiling-eval.txt
fi

# a2: simulated annealing in 2 dimensions

# note: the 'embed' variants technically performed fulleval but they were launched when less measures were implemented;
# so we include a separate evaluator to compute everything

if [[ $1 == *-a2-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using anneal2" \
$DHRGSEED stamplen=9 -sag-creq 20000 -sag_gdist_load /tmp/graph-g711 -sagformat 1 -sag $IDIR/graph.txt -sag-bridging -rdi $IDIR/graph.txt -sag-fulleval -sag-export-lorentz $DIR/anneal2.txt -exit | tee $DIR/log-anneal2.txt
fi

if [[ $1 == *-a2-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/anneal2.txt && logdo "evaluate anneal2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-lorentz2 $DIR/anneal2.txt | tee $DIR/log-anneal2-fulleval.txt
fi

# a3: simulated annealing in 2 dimensions

if [[ $1 == *-a3-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using anneal3" \
$DHRGSEED stamplen=9 -geo 435h -bch-subcubes 2 -sag-creq 20000 -sag_gdist_load /tmp/graph-g435b2 -sagformat 1 -sag $IDIR/graph.txt -sag-bridging -rdi $IDIR/graph.txt -sag-fulleval -sag-export-lorentz $DIR/anneal3.txt -exit | tee $DIR/log-anneal3.txt
fi

if [[ $1 == *-a3-eval-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/anneal3.txt && logdo "evaluate anneal3 fulleval" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-lorentz3 $DIR/anneal3.txt | tee $DIR/log-anneal3-fulleval.txt
fi

if [[ $1 == *-a2-buggy-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/anneal2.txt && logdo "evaluate anneal2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-lorentz2 <( bin/anneal-recover 2 $DIR/anneal2.txt < $IDIR/graph.txt) | tee $DIR/log-anneal2-fulleval.txt
fi

if [[ $1 == *-a3-buggy-* ]]
then vfile $DIR/bfkl-dhrg.txt && vfile $DIR/anneal3.txt && logdo "evaluate anneal3 fulleval" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-lorentz3 <( bin/anneal-recover 3 $DIR/anneal3.txt < $IDIR/graph.txt) | tee $DIR/log-anneal3-fulleval.txt
fi


# c2: coalescent embedding in 2 dimensions

function c2 {
  bin/graph-to-coalescent 2 < $IDIR/graph.txt > $DIR/coalescent2.m
  indir coalescent_embedding/usage_example octave ../../$DIR/coalescent2.m | grep -v "Python communication" > $DIR/coalescent2-coordinates.txt
  }

if [[ $1 == *-c2-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using coalescent2" c2
fi

if [[ $1 == *-c2-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/coalescent2-coordinates.txt && logdo "evaluate coalescent2 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/coalescent2 -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-coalescent2-fulleval.txt
fi

# c3: coalescent embedding in 3 dimensions

function c3 {
  bin/graph-to-coalescent 3 < $IDIR/graph.txt > $DIR/coalescent3.m
  indir coalescent_embedding/usage_example octave ../../$DIR/coalescent3.m | grep -v "Python communication" > $DIR/coalescent3-coordinates.txt
  }

if [[ $1 == *-c3-embed-* ]]
then vfile $IDIR/graph.txt && logdo "embed using coalescent3" c3
fi

if [[ $1 == *-c3-eval-* ]]
then vfile $DIR/lorentz2-dhrg.txt && vfile $DIR/coalescent3-coordinates.txt && logdo "evaluate coalescent3 fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-lorentz3 $DIR/coalescent3-coordinates.txt | tee $DIR/log-coalescent3-fulleval.txt
fi

# kvk: KVK embedder (HyperLink)

if [[ $1 == *-kvk-embed-* ]]
then vfile $IDIR/graph.txt && vfile $DIR/bfkl-links.txt && logdo "embed using kvk" \
sh kvk/bridging-hyperlink.sh $IDIR/graph.txt $DIR/bfkl-coordinates.txt $DIR/kvk-coordinates.txt $SEED | tee $DIR/log-kvk.txt
fi

if [[ $1 == *-kvk-embgt-* ]]
then vfile $IDIR/graph.txt && vfile $DIR/bfkl-links.txt && logdo "embed using kvk knowing ground truth" \
sh kvk/bridging-hyperlink.sh $IDIR/graph.txt $IDIR/groundtruth-coordinates.txt $DIR/kvk-coordinates.txt $SEED | tee $DIR/log-kvk.txt
fi

if [[ $1 == *-kvk-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/kvk-coordinates.txt && logdo "evaluate kvk fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/kvk -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-kvk-fulleval.txt
fi

# clove: HypCLOVE

if [[ $1 == *-clove-embed-* ]]
then vfile $IDIR/graph.txt && vfile $DIR/bfkl-links.txt && logdo "embed using hypCLOVE" \
python hypCLOVE/import_clove.py $IDIR/graph.txt > $DIR/clove-coordinates.txt "$SEED"1 | tee $DIR/log-clove.txt
fi

if [[ $1 == *-clove-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/clove-coordinates.txt && logdo "evaluate hypCLOVE" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/clove -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-clove-fulleval.txt
fi

if [[ $1 == *-clove-dhrg-* ]]
then vfile $DIR/clove-coordinates.txt && vfile $DIR/bfkl-links.txt &&  logdo "improve hypCLOVE" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/clove -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-clove-improve.txt
fi

if [[ $1 == *-clove-dgreedy-* ]]
then vfile $DIR/clove-dhrg.txt && logdo "evaluate dhrg hypCLOVE fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/clove -rdi $IDIR/graph.txt -routing-qdist | tee $DIR/log-dhrg-clove-greedy.txt
fi

if [[ $1 == *-clove-deval-* ]]
then vfile $DIR/clove-dhrg.txt && logdo "evaluate ddhrg hypCLOVE fulleval" \
$DHRG -alt-edges $DIR/bfkl-links.txt -eload $DIR/clove -rdi $IDIR/graph.txt -cellcoords -eval-hdist -dhrg-to-polar $DIR/dhrg-clove-polar.txt | tee $DIR/log-ddhrg-clove-fulleval.txt
fi

# lpcs: LPCS

function lpcs {
indir LPCS ../bin/graph-to-lpcs "$SEED"1 < $IDIR/graph.txt > LPCS/bridging.m
indir LPCS octave bridging.m > $DIR/lpcs-coordinates.txt
}

function lpcs-remake {
indir LPCS ../bin/lpcs-remake "$SEED"1 < $IDIR/graph.txt > $DIR/lpcs-coordinates.txt
}

if [[ $1 == *-lpcs-embed-* ]]
then vfile $IDIR/graph.txt && vfile $DIR/bfkl-links.txt && vfile $DIR/bfkl-links.txt && logdo "embed using LPCS" lpcs
fi

if [[ $1 == *-lpcs-alt-* ]]
then vfile $IDIR/graph.txt && vfile $DIR/bfkl-links.txt && vfile $DIR/bfkl-links.txt && logdo "embed using alt LPCS" lpcs-remake
fi

if [[ $1 == *-lpcs-eval-* ]]
then vfile $DIR/bfkl-links.txt && vfile $DIR/lpcs-coordinates.txt && logdo "evaluate LPCS" \
$DHRG -alt-edges $DIR/bfkl-links.txt -dhrg $DIR/lpcs -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-lpcs-fulleval.txt
fi

function timer {
  start=`grep "$1" $DIR/log.txt | grep -v finished | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  end=`grep "$1" $DIR/log.txt | grep finished | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  #echo \(START=$start END=$end\)  
  if [ -n "$start" ]
  then if [ "$start" = "$end" ]
    then echo 1
    else echo $((`date "+%s" -d "$end"`-`date "+%s" -d "$start"`))
    fi
  else echo missing >> tmp
  fi
  }

function addtimer {
  start=`grep "$2" $DIR/log.txt | grep -v finished | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  if [ -n "$start" ]
  then echo $((`timer "$2"` + `timer "$1"`))
  else echo missing addtimer >> tmp
  fi
  }

# a part of experiments had to be redone in Sep 1X, 2025 due to an earlier bug
# but they had to be done on a faster setup, so we still use the old time for the statistics
function timer-old {
  start=`grep "$1" $DIR/log.txt | grep -v finished | grep -v "Sep 1" | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  end=`grep "$1" $DIR/log.txt | grep finished | grep -v "Sep 1" | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  echo START=$start END=$end >> tmp
  if [ -n "$start" ]
  then if [ "$start" = "$end" ]
    then echo 1
    else echo $((`date "+%s" -d "$end"`-`date "+%s" -d "$start"`))
    fi
  else echo missing >> tmp
  fi
  }

function stat {

  echo GRAPH=$GRAPH

  echo :: graph $GRAPH 1>&2

  echo n=`cat $DIR/log-bfkl-improve.txt | grep "Vertices by distance" | head -n 1 | sed "s-.*N = --" | sed "s-):--"`
  echo m=`wc -l $IDIR/graph.txt | sed "s- .*--"`
  echo entropy=`cat $DIR/log-bfkl-fulleval.txt | grep "entropy =" | head -n 1 | sed "s-.*entropy = --"`

  function greedy {
    if [ -f $2 ]
    then 
    echo $1_greedy_success=`cat $2 | grep "greedy routing:" | sed "s-.*success = --" | sed "s- .*--"`
    echo $1_greedy_stretch=`cat $2 | grep "greedy routing:" | sed "s-.*stretch = --" | sed "s- .*--"`
    echo $1_greedy_effect=`cat $2 | grep "greedy routing:" | grep efficiency | sed "s-.*efficiency = --" | sed "s- .*--"`
    echo $1_modded_success=`cat $2 | grep "modded routing:" | sed "s-.*success = --" | sed "s- .*--"`
    echo $1_modded_stretch=`cat $2 | grep "modded routing:" | sed "s-.*stretch = --" | sed "s- .*--"`
    echo $1_modded_effect=`cat $2 | grep "modded routing:" | grep efficiency | sed "s-.*efficiency = --" | sed "s- .*--"`
    else echo "greedy file missing: $2" 1>&2
    fi
    }

  function fulleval {
    if [ -f $2 ]
    then echo $1_au=`cat $2 | grep "entropy" | grep "radius =" | tail -n 1 | sed "s-.*radius = --" | sed "s- .*--"`
    echo $1_map=`cat $2 | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`
    echo $1_meanrank=`cat $2 | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
    echo $1_loglik=`cat $2 | grep loglikelihood | sed "s-.*loglikelihood = --"`
    echo $1_control=`cat $2 | grep "control" | tail -n 1 |  sed "s-.*control = --"`
    greedy $1 $2
    else echo "fulleval file missing: $2" 1>&2
    fi
    }

  function nkeval {
    if [ -f $2 ]
    then
    echo $1_meanrank=`cat $2 | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
    echo $1_map=`cat $2 | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
    else echo "fulleval file missing: $2" 1>&2
    fi
    }

  function improve {
    if [ -f $DIR/log-$1-improve.txt ]
    then
    #echo $1_result=`cat $DIR/log-$1-evaluate.txt | grep rank | tail -n 1`
    echo $1_dau=`cat $DIR/log-$1-improve.txt | grep "Vertices by distance" -A 1 | head -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
    echo $1_dmeanrank=`cat $DIR/log-$1-improve.txt | grep meanrank | head -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
    echo $1_dmap=`cat $DIR/log-$1-improve.txt | grep meanrank | head -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
    #echo $1_pre_improve=`cat $DIR/log-$1-improve.txt | grep meanrank | head -n 1`
    echo $1_dhrg_dau=`cat $DIR/log-$1-improve.txt | grep "Vertices by distance" -A 1 | tail -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
    echo $1_dhrg_dmeanrank=`cat $DIR/log-$1-improve.txt | grep meanrank | tail -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
    echo $1_dhrg_dmap=`cat $DIR/log-$1-improve.txt | grep meanrank | tail -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
    #echo $1_post_improve=`cat $DIR/log-$1-improve.txt | grep meanrank | tail -n 1`
    else echo "improve file missing: $1" 1>&2
    fi
    }

  echo : BFKL 1>&2

  echo bfkl_time=`timer "embed using BFKL"`
  fulleval bfkl $DIR/log-bfkl-fulleval.txt
  echo bfkl_dhrg_time=`addtimer "embed using BFKL" "improve BFKL"`
  improve bfkl
  greedy bfkl_post $DIR/log-dhrg-bfkl-greedy.txt
  fulleval bfkl_dhrg $DIR/log-ddhrg-bfkl-fulleval.txt
  fulleval bfkl_penalty2 $DIR/log-penalty2-fulleval.txt

  echo : NK variants 1>&2

  echo poincare2_time=`timer "embed using Poincare 2D"`
  nkeval poincare2_alt $DIR/log-poincare2-evaluate.txt
  fulleval poincare2 $DIR/log-poincare2-fulleval.txt
  echo poincare2_dhrg_time=`addtimer "embed using Poincare 2D" "improve Poincare 2D"`
  improve poincare2
  fulleval poincare2_dhrg $DIR/log-ddhrg-poincare-fulleval.txt

  echo poincare3_time=`timer "embed using Poincare 3D"`
  nkeval poincare3_alt $DIR/log-poincare3-evaluate.txt
  fulleval poincare3 $DIR/log-poincare3-fulleval.txt

  nkeval poincare5_alt $DIR/log-poincare5-evaluate.txt
  fulleval poincare5 $DIR/log-poincare5-fulleval.txt

  echo lorentz2_time=`timer "embed using Lorentz 2D"`
  nkeval lorentz2_alt $DIR/log-lorentz2-evaluate.txt
  echo lorentz2_eval_time=`cat $DIR/log-lorentz2-evaluate.txt | grep rank | tail -n 1 | sed "s-.*time: --"`
  fulleval lorentz2 $DIR/log-lorentz2-fulleval.txt
  echo lorentz2_dhrg_time=`addtimer "embed using Lorentz 2D" "improve Lorentz"`
  improve lorentz2
  fulleval lorentz2_dhrg $DIR/log-ddhrg-lorentz2-fulleval.txt

  echo lorentz3_time=`timer "embed using Lorentz 3D"`
  fulleval lorentz3 $DIR/log-lorentz3-fulleval.txt

  nkeval euclidean50 $DIR/log-euclidean50-evaluate.txt
  fulleval euclidean50 $DIR/log-euclidean50-fulleval.txt

  nkeval euclidean200 $DIR/log-euclidean200-evaluate.txt
  fulleval euclidean200 $DIR/log-euclidean200-fulleval.txt

  echo landscape50_meanrank=`cat $DIR/log-landscape-50.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo landscape50_map=`cat $DIR/log-landscape-50.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`

  echo landscape200_meanrank=`cat $DIR/log-landscape-200.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo landscape200_map=`cat $DIR/log-landscape-200.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`

  echo : Mercator variants 1>&2

  echo mfast_time=`timer "embed using Mercator fast"`
  fulleval mfast $DIR/log-mercator-fast-fulleval.txt

  echo mfull_time=`timer "embed using Mercator full"`
  fulleval mfull $DIR/log-mercator-full-fulleval.txt

  echo mdim_time=`timer "embed using d-Mercator"`
  echo mdim_au=`cat $DIR/log-d-mercator-fulleval.txt | grep radius | sed "s-.*radius = --" | sed "s- .*--"`
  fulleval mdim $DIR/log-d-mercator-fulleval.txt

  echo : other embedders 1>&2

  echo kvk_time=`timer-old "embed using kvk"`
  fulleval kvk $DIR/log-kvk-fulleval.txt

  echo lpcs_time=`timer "embed using LPCS"`
  fulleval lpcs $DIR/log-lpcs-fulleval.txt

  echo : coalescent 1>&2

  echo coalescent2_time=`timer "embed using coalescent2"`
  fulleval coalescent2 $DIR/log-coalescent2-fulleval.txt

  echo coalescent3_time=`timer "embed using coalescent3"`
  fulleval coalescent3 $DIR/log-coalescent3-fulleval.txt

  echo : clove 1>&2

  echo clove_time=`timer "embed using hypCLOVE"`
  fulleval clove $DIR/log-clove-fulleval.txt

  echo clove_dhrg_time=`addtimer "embed using hypCLOVE" "improve hypCLOVE"`
  fulleval clove_dhrg $DIR/log-ddhrg-clove-fulleval.txt
  improve clove

  echo : visualizations 1>&2

  echo hypviewer_au=`cat $DIR/log-hypviewer.txt | grep radius | cut -d "=" -f 2`
  echo hypviewer_meanrank=`cat $DIR/log-evaluate-hypviewer.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo hypviewer_map=`cat $DIR/log-evaluate-hypviewer.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`

  echo rogueviz_au=`cat $DIR/rogueviz-coordinates.txt | tail -n +3 | cut -d " " -f 2 | sort -g | tail -n 1`
  echo rogueviz_meanrank=`cat $DIR/log-rogueviz-fulleval.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo rogueviz_map=`cat $DIR/log-rogueviz-fulleval.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`

  echo : treerep 1>&2

  echo norec_treerep_tu=`cat $DIR/log-treerep-eval.txt | grep "maximum distance" | head -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--"`
  echo norec_treerep_alt_map=`cat $DIR/log-treerep.txt | grep "norec mAP" | sed "s-.* = --"`
  cat $DIR/log-treerep-eval.txt | grep wrec.txt -B 100 > /tmp/norec
  fulleval norec_treerep /tmp/norec | grep -v _au

  echo wrec_treerep_tu=`cat $DIR/log-treerep-eval.txt | grep "maximum distance" | tail -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--"`
  echo wrec_treerep_alt_map=`cat $DIR/log-treerep.txt | grep "wrec mAP" | sed "s-.* = --"`
  cat $DIR/log-treerep-eval.txt | grep wrec.txt -A 100 > /tmp/wrec
  fulleval wrec_treerep /tmp/wrec | grep -v _au

  echo : ltiling 1>&2

  echo ltiling_time=`timer "embed using LTiling_rsgd"`
  fulleval ltiling $DIR/log-ltiling-eval.txt | grep -v ltiling_au
  echo ltiling_au=`cat $DIR/log-ltiling-eval.txt | grep "maximum distance" | tail -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--" | awk -v CONVFMT=%.6g '{print $1 / 2}'`

  echo : anneal 1>&2

  fulleval anneal2 $DIR/log-anneal2-fulleval.txt
  fulleval anneal3 $DIR/log-anneal3-fulleval.txt

  echo : done 1>&2

  #echo landscape199=`cat $DIR/log-landscape-199b.txt | grep MeanRank`
  #echo landscape50=`cat $DIR/log-landscape-50.txt | grep MeanRank`
  #echo BFKL embed time: `grep "embed using BFKL" $DIR/log.txt`
  }

if [ -n "$PROCESS_DRY" ]
then
echo "#" done
else
stat > $DIR/summary.txt
fi

if [[ $1 == *-done ]]
then log "experiment complete"
fi

