#include <string>
#include <vector>
#include <iostream>
#include <map>
#include <cmath>
#include <set>

using namespace std;

vector<string> graphs;

map<string, map<string, double>> values;

map<string, map<string, pair<double, double> >> bars;

bool compute_stats;
string current_stat, current_graph;

map<pair<string, string>, vector<double> > stats;

bool colored = false;
bool view_extra = false;
bool noncompete = false;
bool alt_layout = false;

bool multirows = false, short_multirows = false;

double good_linear(double d) { return d; }
double bad_linear(double d) { return -d; }
double good_inverse(double d) { return 1/d; }
double bad_log(double d) { return -log(d); }

map<string, double(*)(double)> which_good;

map<string, string> what_it_is;

void set_good() {
  which_good["LL"] = good_linear;
  what_it_is["LL"] = "LL: normalized loglikelihood";
  which_good["ICV"] = good_linear;
  what_it_is["ICV"] = "ICV: Information Control Value";
  which_good["ET"] = bad_log;
  what_it_is["ET"] = "embedding time [s]";
  which_good["RA"] = bad_linear;
  what_it_is["RA"] = "radius [absolute units]";
  which_good["RG"] = bad_linear;
  what_it_is["RG"] = "radius [grid tiles]";
  which_good["MAP"] = good_linear;
  what_it_is["MAP"] = "MAP: Mean Average Precision";
  which_good["MR"] = good_inverse;
  what_it_is["MR"] = "MR: MeanRank";
  which_good["GSR"] = good_linear;
  what_it_is["GSR"] = "GSR: Greedy Success Ratio";
  which_good["GSF"] = good_inverse;
  what_it_is["GSF"] = "GSF: Greedy Stretch Factor";
  which_good["GRE"] = good_linear;
  what_it_is["GRE"] = "GRE: Greedy Routing Efficiency";
  which_good["dMAP"] = good_linear;
  what_it_is["dMAP"] = "dMAP: discrete mAP";
  which_good["dMR"] = good_inverse;
  what_it_is["dMR"] = "dMR: discrete MeanRank";
  }

map<string, int> num_in_cat;

string last_cat = "";

void tableline(string title, const auto& f) {
  string cat = title.substr(0, title.find(" "));
  string nocat = title.substr(title.find(" ") + 1);

  if(compute_stats) {
    bool emptyline = true;
    for(auto g: graphs) {
      current_graph = g; if(f(g) != "") emptyline = false;
      }
    if(!emptyline) num_in_cat[cat]++;
    return;
    }

  bool emptyline = true;
  for(auto g: graphs) if(f(g) != "") emptyline = false;

  if(emptyline && colored) {
    printf("%% EMPTY LINE: %s\n", title.c_str());
    return;
    }

  if(nocat[0] == 'E' && multirows && nocat.find("50") != string::npos) printf("\\cline{2-%d}\n", int(graphs.size() - 1));

  if(multirows) {
    if(cat != last_cat)
      printf("\\multirow{%d}{*}{\\rotatebox{90}{%s}}\n", num_in_cat[cat], short_multirows ? cat.c_str() : what_it_is[cat].c_str());
    last_cat = cat;
    printf("&%-30s ", nocat.c_str());
    }
  else
    printf(" %-30s ", title.c_str());

  for(auto g: graphs) {
    current_graph = g; 
    string s = f(g);
    if(s == "" && colored) s = "\\hfill - \\hskip 0.75em \\rule{0cm}{0cm} \\hfill";
    printf(" &%7s", s.c_str());
    }
  printf(" \\\\\n");
  }

string its(int val) {
  if(val == 0) return "";
  char buf[100];
  sprintf(buf, "%d", val);
  return buf;  
  }

bool have_bars = false;

string fmtf(string fmt, double val, pair<double, double> bar) { 
  if(!val) return "";
  if(fmt == "") return its(val);
  char buf[100];
  sprintf(buf, fmt.c_str(), val);
  if(have_bars) {
    string ret1 = buf;
    sprintf(buf, fmt.c_str(), bar.first);
    string ret = "("; ret += buf;
    sprintf(buf, fmt.c_str(), bar.second);
    ret = ret + ", " + buf + ")";

    return "$\\underset{" + ret + "}{" + ret1 + "}$";
    }
  return buf;
  }

string fmtx(string category, string fmt, double val, pair<double, double> bar) { 

  if(compute_stats) { 
    if(!val) return "";
    if(!noncompete) stats[{current_graph, category}].push_back(val);
    return ".";
    }

  if(!val) return "";

  string content = fmtf(fmt, val, bar);
  if(!colored) return content;

  double maxval = -HUGE_VAL, minval = HUGE_VAL;
  int all = 0, good = 0;
  auto f = which_good[category];
  auto fval = f(val);
  for(auto x: stats[{current_graph, category}]) {
    auto fx = f(x);
    if(fx > maxval) maxval = fx;
    if(fx < minval) minval = fx;
    if(fval > fx) good++;
    if(fval >= fx) good++;
    all += 2;
    }

  good --; all -= 2;

  double rank = good * 1. / all;
  double qty = (fval - minval) / (maxval - minval);

  if(qty > 1) qty = 1;
  if(rank > 1) rank = 1;

  double compromise = (rank + qty) / 2;

  if(isnan(compromise) || isinf(compromise)) compromise = 0;

  int col = int((1 - pow(compromise, 0.7)) * 255 + 0.5);
  int colg = int((1 - compromise / 2) * 255 + 0.5);
 
  char buf[200];
  sprintf(buf, "\\cellcolor[HTML]{%02X%02X%02X}%s", col, colg, col, content.c_str());
  return buf;
  }

string measures = "tmausrlc";

int mode = 0;
bool undirected = false;

void gen(string s) {
  auto ex = [&] (char c) {
    bool incase = false;
    for(char d: s) if(c==d) return true; else if((c>>5) == (d>>5)) incase = true;
    if(!incase) return true;
    return false;
    };

  bool lc_ratio = true;

  auto ltf = [&] (string s, double v, pair<double, double> vb) {
    auto info = values[s]["entropy"]; return (lc_ratio && v) 
      ? fmtx("LL", "%.3f", 1 - v / info, pair<double, double>(1 - vb.second / info, 1 - vb.first / info))
      : fmtf("%.1f", v, vb); 
    };
  auto ctf = [&] (string s, double v, pair<double, double> vb) {
    auto info = values[s]["entropy"]; return (lc_ratio && v) 
      ? fmtx("ICV", "%.3f", info / (info + v), pair<double, double>(info / (info + vb.second), info / (info + vb.first)))
      : fmtf("%.1f", v, vb);
    };

  if(ex('n') && ex('N')) {
    tableline(" nodes", [&] (string s) { return its(values[s]["n"]); }); // log-bfkl-improve.txt    
    if(undirected)
      tableline(" edges (undirected)", [&] (string s) { return its(values[s]["m"]/2); }); // nothing
    else
      tableline(" edges", [&] (string s) { return its(values[s]["m"]); }); // nothing

    if(view_extra) tableline(" information", [&] (string s) { return its(values[s]["entropy"]); });
    }

  int no_time = 1, no_radius = 2, radius_tu = 4, alt_time = 8, no_routing = 16, no_stretch = 32, alt_map = 64, no_loglik = 128;

  auto fulleval = [&] (string showname, string paramname, int flags) {

    string altparamname = paramname.substr(0, paramname.size() - 3);

#define V(x) values[s][x], bars[s][x]

    if((flags & alt_time) && ex('t')) {
      tableline("ET " + showname, [&] (string s) { return fmtx("ET", "", V(altparamname + "_time")); });
      }
    else if(!(flags & no_time) && ex('t'))
      tableline("ET " + showname, [&] (string s) { return fmtx("ET", "", V(paramname + "_time")); });

    if((flags & radius_tu) && ex('r')) { if(view_extra) tableline("RT " + showname, [&] (string s) { return fmtf("%.3f", values[s][paramname+"_tu"] / 2, {0, 0}); }); }
    else if(!(flags & no_radius) && ex('r')) tableline("RA " + showname, [&] (string s) { return fmtx("RA", "%.3f", V(paramname+"_au")); });

    if(ex('m'))
      tableline("MR " + showname, [&] (string s) { return fmtx("MR", "%.1f", V(paramname+"_meanrank")); });

    if(ex('a'))
      tableline("MAP " + showname, [&] (string s) { return fmtx("MAP", "%.3f", V(paramname+"_map")); });

    if(ex('a') && (flags & alt_map))
      tableline("MAP orig " + showname, [&] (string s) { return fmtx("MAP", "%.3f", V(altparamname+"_map")); });

    if(ex('u') && mode != 2 && !(flags & no_routing))
      tableline("GSR " + showname, [&] (string s) { return fmtx("GSR", "%.3f", V(paramname+"_greedy_success")); });

    if(ex('s') && mode != 2 && !(flags & no_routing))
       tableline("GSF " + showname, [&] (string s) { return fmtx("GSF", "%.2f", V(paramname+"_greedy_stretch")); });

    if(ex('e') && mode != 2 && !(flags & no_routing))
       tableline("GRE " + showname, [&] (string s) { return fmtx("GRE", "%.3f", V(paramname+"_greedy_effect")); });

    if(ex('l') && !(flags & no_loglik))
      tableline("LL " + showname, [&] (string s) { return ltf(s, V(paramname+"_loglik")); });

    if(ex('c') && !(flags & no_loglik))
      tableline("ICV " + showname, [&] (string s) { return ctf(s, V(paramname+"_control")); });

    if(ex('d')) {
      tableline("dMR " + showname, [&] (string s) { return fmtx("dMR", "%.1f", V(paramname + "_dmeanrank")); }); // log-lorentz2-improve.txt
      tableline("dMR " + showname + " + DHRG", [&] (string s) { return fmtx("dMR", "%.1f", V(paramname + "_dmeanrank")); }); // log-lorentz2-improve.txt
      }

    if(ex('b')) {
      tableline("dMAP " + showname, [&] (string s) { return fmtx("dMAP", "%.3f", V(paramname + "_dmap")); }); // log-lorentz2-improve.txt
      tableline("dMAP " + showname + " + DHRG", [&] (string s) { return fmtx("dMAP", "%.3f", V(paramname + "_dmap")); }); // log-lorentz2-improve.txt
      }
    };

  if(ex('B')) {
    fulleval("BFKL", "bfkl", 0);
    fulleval("BFKL + DHRG", "bfkl_dhrg", 0);

    if(view_extra && ex('g')) tableline("RG BFKL", [&] (string s) { return fmtx("RG", "", V("bfkl_radius_pre_improve")); }); // log-bfkl-improve.txt
    }

  if(ex('P')) {
    fulleval("Poincare 2D", "poincare2", 0);
    fulleval("Poincare 2D + DHRG", "poincare2_dhrg", 0);
    fulleval("Poincare 3D", "poincare3", 0);
    fulleval("Poincare 5D", "poincare5", no_routing | no_loglik | no_time | no_radius);
    }

  if(ex('L')) {
    fulleval("Lorentz 2D", "lorentz2", 0);
    fulleval("Lorentz 2D + DHRG", "lorentz2_dhrg", 0);
    fulleval("Lorentz 3D", "lorentz3", 0);

    if(view_extra && ex('t')) tableline("VT Lorentz 2D", [&] (string s) { return fmtf("%.2f", V("lorentz2_eval_time")); }); // log-lorentz2-evaluate.txt
    if(view_extra && ex('g')) tableline("RG Lorentz 2D", [&] (string s) { return fmtx("RG", "", V("lorentz2_radius_pre_improve")); }); // log-lorentz2-improve.txt


    // if(ex('t')) tableline("Lorentz 3D eval time [s]", [&] (string s) { return fmtf("%.2f", V("lorentz3_eval_time")); }); // missing
    // if(ex('g')) tableline("Lorentz 3D radius grid", [&] (string s) { return its(V("lorentz3_radius_pre_improve")); }); // log-lorentz2-improve.txt
    }

  if(ex('Q')) fulleval("penalty", "bfkl_penalty2", no_time);

  if(ex('A')) {
    fulleval("anneal2", "anneal2", no_time);
    fulleval("anneal3", "anneal3", no_time);
    }

  if(ex('C')) {
    fulleval("coalescent2", "coalescent2", 0);
    fulleval("coalescent3", "coalescent3", 0);
    }

  if(ex('K')) fulleval("KVK", "kvk", 0);

  if(ex('V')) {
    fulleval("CLOVE", "clove", 0);
    fulleval("CLOVE + DHRG", "clove_dhrg", 0);
    }

  if(ex('X')) fulleval("LPCS", "lpcs", 0);

  if(ex('S')) {
    if(view_extra) fulleval("Landscape 50D", "landscape50", no_time | no_radius | no_routing | no_loglik);
    if(view_extra) fulleval("Landscape 200D", "landscape200", no_time | no_radius | no_routing | no_loglik);
    }

  if(ex('H') && mode != 1) {
    fulleval("HypViewer", "hypviewer", no_time | no_routing | no_loglik);
    if(view_extra) fulleval("RogueViz", "rogueviz", no_time | no_routing | no_loglik);
    }

  if(ex('M')) {
    fulleval("Mercator fast", "mfast", 0);
    fulleval("Mercator full", "mfull", 0);
    fulleval("d-Mercator", "mdim", 0);
    }

  if(ex('T')) {
    // if(ex('t')) tableline("ET TreeRep", [&] (string s) { return fmtx("ET", "", V("treerep_time")); }); <- no time because it generates a distance table anyway
    fulleval("TreeRep rec", "wrec_treerep_hr", radius_tu | no_time | alt_map | no_loglik);
    fulleval("TreeRep norec", "norec_treerep_hr", radius_tu | no_time | alt_map | no_loglik);
    }

  if(ex('G')) {
    fulleval("ltiling", "ltiling", no_loglik);
    }

  if(ex('E')) {
    noncompete = true;
    fulleval("Euclidean 50D", "euclidean50", no_time | no_radius | no_routing | no_loglik);
    fulleval("Euclidean 200D", "euclidean200", no_time | no_radius | no_routing | no_loglik);
    noncompete = false;
    }

  }

void gen_alt_layout(string arg) {

  int inrow = 0, rownum = 0;

  auto entry = [&] (string cap, string t) {
    if(inrow) printf("&");
    printf("%s", rownum ? t.c_str() : cap.c_str());
    inrow++;
    };

  auto row = [&] (string s, string title, string paramname) {
    inrow = 0;
    entry("embedder", title);
    for(char c: arg) if(c >= 'a') {
      if(c == 'a') entry("MAP", fmtx("MAP", "%.3f", V(paramname+"_map")));
      if(c == 'b') entry("dMAP", fmtx("dMAP", "%.3f", V(paramname+"_dmap")));
      if(c == 'm') entry("MR", fmtx("MR", "%.1f", V(paramname+"_meanrank")));
      if(c == 'd') entry("dMR", fmtx("dMR", "%.1f", V(paramname+"_dmeanrank")));
      }
    printf(" \\\\\n"); rownum++;
    };

  for(auto g: graphs) {
    rownum = 0; current_graph = g;
    row(g, g, ""); printf("\\hline\n");
    for(char c: arg) {
      if(c == 'B') row(g, "BFKL", "bfkl"), row(g, "BFKL + DHRG", "bfkl_dhrg");
      if(c == 'P') row(g, "Poincare 2D", "poincare2"), row(g, "Poincare 2D + DHRG", "poincare2_dhrg"), row(g, "Poincare 3D", "poincare3");
      if(c == 'L') row(g, "Lorentz 2D", "lorentz2"), row(g, "Lorentz 2D + DHRG", "lorentz2_dhrg"), row(g, "Lorentz 3D", "lorentz3");
      if(c == 'V') row(g, "CLOVE", "clove"), row(g, "CLOVE + DHRG", "clove_dhrg");
      }
    }
  }

int main(int argc, char ** argv) {

  set_good();

  string graph = "none";
  string s;

  if(argv[1][0] == '5') {
    have_bars = true;
    set<string> seen;
    while(true) {
      char graph[100], measure[100];
      double a, b, c;
      int num = scanf("%s%s%lf%lf%lf", graph, measure, &a, &b, &c);
      if(num != 5) break;
      if(!seen.count(graph)) { seen.insert(graph); graphs.push_back(graph); }
      values[graph][measure] = a;
      bars[graph][measure] = {b, c};
      }
    }

  else while(getline(cin, s)) {
    int p = -1;
    for(int i=0; i<int(s.size()); i++) if(s[i] == '=') p = i;
    if(p == -1) continue;
    string cat = s.substr(0, p);
    string vals = s.substr(p+1);
    if(cat == "GRAPH") { graph = vals; graphs.push_back(graph); continue; }
    double vald = atof(vals.c_str());
    if(isnan(vald)) vald = 0;
    values[graph][cat] = vald;
    }

  string argvs = argc > 1 ? argv[1] : "";

  for(auto cs: {true, false}) {
    compute_stats = cs;
    string line = "";

    if(alt_layout) { gen_alt_layout(argvs); return 0; }

    if(!cs) tableline(" graph name", [&] (string s) { 
      if(s.substr(0, 11) == "connectome/") s = s.substr(11, 5) + s.back();
      if(s.substr(3, 11) == "connectome/") s = s.substr(0, 3) + s.substr(14, 2) + s.back();
      if(s.substr(0, 12) == "simulated/3d") {
        int n, r, t, id;
        sscanf(s.c_str(), "simulated/3d/%d-%d-%d/%d", &n, &r, &t, &id);
        char head[20];
        sprintf(head, "S%d%d%d-%d", n/1000, r, t, id);
        s = head;
        }
      if(s.substr(0, 12) == "simulated/co") {
        int n, r, t, id;
        sscanf(s.c_str(), "simulated/converted/%d-T%d/%d", &n, &t, &id);
        char head[20];
        sprintf(head, "U%d-%d-%d", n/500, t, id);
        s = head;
        }

      return s.substr(0, 6);
      });

    for(char ch: argvs) {
      if(ch == '3')
        colored = true, multirows = true; 
      else if(ch == '5')
        short_multirows = true;
      else if(ch == '4')
        undirected = true;
      else if(ch == '5') ;
      else if(ch == '6') alt_layout = true;
      else if(ch == '1' || ch == '2')
        mode = ch - '0';
      else if(ch == ',') gen(line), line = "";
      else if(ch == ';') {
        if(line != "") gen(line), line = "";
        if(!cs) printf("\\hline\n");
        }
      else {
        line += ch;
        }
      }

    gen(line);
    }
  
  return 0;
  }
