#!/bin/bash
# usage: bash process/sh [steps] [graph name]
# needs graphs/[graph name]/graph.txt and graphs/[graph name]/graph.csv

# extra env variables:
# set SEED to change the RNG seed
# set PROCESS_DRY to just output the commands, do not execute them

GRAPH=$2
IDIR=graphs/$GRAPH/
DIR=results/$GRAPH/
DHRG="hyperrogue/hyper -c - -fixx 42 -nogui -s -"
DHRGSEED="hyperrogue/hyper -c - -fixx 42$SEED -nogui -s -"

if [ -n "$SEED" ]
then DIR=results/repeated/$GRAPH$SEED/
GRAPH=r$SEED/$GRAPH
fi
echo "#" DIR is \'$DIR\', SEED is \'$SEED\'

function log {
  echo 1>&2
  echo "######" $GRAPH " : " $1 1>&2
  echo 1>&2
  echo `TZ=Universal date` : $1 >> $DIR/log.txt
  }

function logdo {
  if [ -n "$PROCESS_DRY" ]
  then echo "#" log message: $1
  shift
  echo "$@"
  else  
  LOGMSG="$1"
  shift
  log "$LOGMSG"
  "$@"
  log "$LOGMSG: finished"
  fi
  }

function indir {
  pushd . >/dev/null
  cd $1
  shift
  "$@"
  popd >/dev/null
  }

function poincare {
  indir poincare-embeddings python3 "$@"
  }

mkdir -p $DIR

if [[ $1 == *a* ]]
then logdo "embed using BFKL" hyperbolic-embedder/embedder -seed 123456$SEED -input $IDIR/graph.txt -embed $DIR/bfkl | tee $DIR/log-bfkl.txt
fi

if [[ $1 == *b* ]]
then logdo "embed using Lorentz 2D" poincare embed.py \
  -checkpoint ../$DIR/lorentz2.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold lorentz \
  -dim 2 | tee $DIR/log-lorentz2.txt
fi

if [[ $1 == *c* ]]
then logdo "evaluate Lorentz 2D" poincare kx-evaluate.py ../$DIR/lorentz2.bin.best -rank 1 | tee $DIR/log-lorentz2-evaluate.txt
fi

if [[ $1 == *d* ]]
then logdo "convert Lorentz 2D" poincare kx-evaluate.py ../$DIR/lorentz2.bin.best -output ../$DIR/lorentz2-coordinates.txt | tee $DIR/log-lorentz2-convert.txt
cp $DIR/bfkl-links.txt $DIR/lorentz2-links.txt
fi

if [[ $1 == *e* ]]
then logdo "improve Lorentz 2D" $DHRG -dhrg $DIR/lorentz2 -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-lorentz2-improve.txt
fi

if [[ $1 == *f* ]]
then logdo "improve BFKL" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -iterate 10 -discrete-rank -iterate 100 -discrete-rank -esave | tee $DIR/log-bfkl-improve.txt
fi

if [[ $1 == *g* ]]
then logdo "landscape 200D" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -landscape-rank 200 | tee $DIR/log-landscape-200.txt
fi

if [[ $1 == *h* ]]
then logdo "landscape 50D" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -discrete-rank -landscape-rank 50 | tee $DIR/log-landscape-50.txt
fi

if [[ $1 == *j* ]]
then logdo "evaluate bfkl fulleval" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-bfkl-fulleval.txt
fi

if [[ $1 == *k* ]]
then logdo "evaluate lorentz2 fulleval" \
$DHRG -dhrg $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-lorentz2-fulleval.txt
fi

if [[ $1 == *l* ]]
then logdo "evaluate dhrg fulleval" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -routing-qdist | tee $DIR/log-dhrg-greedy.txt
fi

if [[ $1 == *o* ]]
then logdo "evaluate ddhrg bfkl fulleval" \
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -cellcoords -eval-hdist -dhrg-to-polar $DIR/dhrg-polar.txt | tee $DIR/log-ddhrg-bfkl-fulleval.txt
fi

if [[ $1 == *p* ]]
then logdo "evaluate ddhrg lorentz2 fulleval" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -cellcoords -eval-hdist | tee $DIR/log-ddhrg-lorentz2-fulleval.txt
fi

if [[ $1 == *q* ]]
then logdo "embed using Poincare 2D" poincare embed.py \
  -checkpoint ../$DIR/poincare2.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 2 | tee $DIR/log-poincare2.txt
fi

if [[ $1 == *r* ]]
then logdo "embed using Euclidean 50D" poincare embed.py \
  -checkpoint ../$DIR/euclidean50.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold euclidean \
  -dim 50 | tee $DIR/log-euclidean50.txt
fi

if [[ $1 == *s* ]]
then logdo "embed using Euclidean 200D" poincare embed.py \
  -checkpoint ../$DIR/euclidean200.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 1 \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold euclidean \
  -dim 200 | tee $DIR/log-euclidean200.txt
fi

if [[ $1 == *t* ]]
then logdo "embed using Poincare 3D" poincare embed.py \
  -checkpoint ../$DIR/poincare3.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 3 | tee $DIR/log-poincare3.txt
fi

if [[ $1 == *u* ]]
then logdo "embed using Poincare 5D" poincare embed.py \
  -checkpoint ../$DIR/poincare5.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold poincare \
  -dim 5 | tee $DIR/log-poincare5.txt
fi

if [[ $1 == *v* ]]
then logdo "evaluate Poincare 2D" poincare kx-evaluate.py ../$DIR/poincare2.bin.best -rank 1 | tee $DIR/log-poincare2-evaluate.txt
fi

if [[ $1 == *w* ]]
then logdo "evaluate Euclidean 50D" poincare kx-evaluate.py ../$DIR/euclidean50.bin.best -rank 1 | tee $DIR/log-euclidean50-evaluate.txt
fi

if [[ $1 == *x* ]]
then logdo "evaluate Euclidean 200D" poincare kx-evaluate.py ../$DIR/euclidean200.bin.best -rank 1 | tee $DIR/log-euclidean200-evaluate.txt
fi

if [[ $1 == *y* ]]
then logdo "evaluate Poincare 3D" poincare kx-evaluate.py ../$DIR/poincare3.bin.best -rank 1 | tee $DIR/log-poincare3-evaluate.txt
fi

if [[ $1 == *z* ]]
then logdo "evaluate Poincare 5D" poincare kx-evaluate.py ../$DIR/poincare5.bin.best -rank 1 | tee $DIR/log-poincare5-evaluate.txt
fi

if [[ $1 == *A* ]]
then logdo "convert Poincare 3D" poincare kx-evaluate.py ../$DIR/poincare3.bin.best -output3 ../$DIR/poincare3.txt | tee $DIR/log-poincare3-convert.txt
fi

if [[ $1 == *B* ]]
then logdo "evaluate poincare3 fulleval" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare3 $DIR/poincare3.txt | tee $DIR/log-poincare3-fulleval.txt
fi

function gentree {
  hypviewer/gentree < $IDIR/graph.txt > $DIR/hypviewer.txt 2> $DIR/log-hypviewer.txt
  }

if [[ $1 == *C* ]]
then logdo "apply HypViewer" gentree
fi

if [[ $1 == *D* ]]
then logdo "evaluate HypViewer" poincare kx-evaluate.py ../$DIR/poincare3.bin.best -input3 ../$DIR/hypviewer.txt | tee $DIR/log-evaluate-hypviewer.txt
fi

if [[ $1 == *E* ]]
then logdo "evaluate HypViewer using dhrg" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare3 $DIR/hypviewer.txt | tee $DIR/log-hypviewer-fulleval.txt
fi

if [[ $1 == *F* ]]
then logdo "embed using Mercator fast" mercator/mercator -f -s 123456$SEED -o $DIR/mercator-fast $IDIR/graph.txt
bin/mercator-to-bfkl < $DIR/mercator-fast.inf_coord > $DIR/mercator-fast-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/mercator-fast-links.txt
fi

if [[ $1 == *G* ]]
then logdo "embed using Mercator full" mercator/mercator -s 123456$SEED -o $DIR/mercator-full $IDIR/graph.txt
bin/mercator-to-bfkl < $DIR/mercator-full.inf_coord > $DIR/mercator-full-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/mercator-full-links.txt
fi

if [[ $1 == *H* ]]
then logdo "embed using Lorentz 3D" poincare embed.py \
  -checkpoint ../$DIR/lorentz3.bin \
  -dset ../$IDIR/graph.csv \
  -epochs 1500 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -fresh \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -lr 0.5 -no-maxnorm \
  -train_threads 1 \
  -dampening 1.0 \
  -batchsize 50 \
  -gpu 0 \
  -manifold lorentz \
  -dim 3 | tee $DIR/log-lorentz3.txt
fi

if [[ $1 == *I* ]]
then logdo "convert Lorentz 3D" poincare kx-evaluate.py ../$DIR/lorentz3.bin.best -output3 ../$DIR/lorentz3.txt | tee $DIR/log-lorentz3-convert.txt
fi

if [[ $1 == *J* ]]
then bin/mercator-to-bfkl < $DIR/mercator-fast.inf_coord > $DIR/mercator-fast-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/mercator-fast-links.txt
logdo "evaluate Mercator fast fulleval" \
$DHRG -dhrg $DIR/mercator-fast -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-mercator-fast-fulleval.txt
fi

if [[ $1 == *K* ]]
then bin/mercator-to-bfkl < $DIR/mercator-full.inf_coord > $DIR/mercator-full-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/mercator-full-links.txt
logdo "evaluate Mercator full fulleval" \
$DHRG -dhrg $DIR/mercator-full -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-mercator-full-fulleval.txt
fi

if [[ $1 == *L* ]]
then log "embed using RogueViz"
hypviewer/gentree < $IDIR/graph.txt > /dev/null
$DHRG -tree /tmp/treedata-tol.txt -write-in-dhrg-format $DIR/rogueviz-coordinates.txt | tee $DIR/log-rogueviz.txt
cp $DIR/bfkl-links.txt $DIR/rogueviz-links.txt
log "embed using RogueViz: finished"
fi

if [[ $1 == *M* ]]
then log "evaluate RogueViz mAP"
hypviewer/gentree < $IDIR/graph.txt > /dev/null
$DHRG -dhrg $DIR/rogueviz -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-rogueviz-fulleval.txt
log "evaluate RogueViz mAP: finished"
fi

if [[ $1 == *N* ]]
then logdo "embed using TreeRep and evaluate" julia -t 8 TreeRep/experiment.jl $IDIR/graph.csv $DIR/treerep-edgelist.txt $DIR/treerep-norec.txt $DIR/treerep-wrec.txt | tee $DIR/log-treerep.txt
fi

if [[ $1 == *O* ]]
then logdo "evaluate TreeReps with other measures" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-disttable $DIR/treerep-edgelist.txt $DIR/treerep-norec.txt -eval-disttable $DIR/treerep-edgelist.txt $DIR/treerep-wrec.txt | tee $DIR/log-treerep-eval.txt
fi

if [[ $1 == *P* ]]
then logdo "embed using LTiling_rsgd" indir HyperbolicTiling_Learning python3 embed.py \
  -dset ../$IDIR/graph.csv \
  -epochs 1000 \
  -negs 50 \
  -burnin 20 \
  -dampening 0.75 \
  -ndproc 4 \
  -eval_each 100 \
  -sparse \
  -burnin_multiplier 0.01 \
  -neg_multiplier 0.1 \
  -lr_type constant \
  -train_threads 2 \
  -dampening 1.0 \
  -batchsize 50 \
  -manifold LTiling_rsgd \
  -dim 2 \
  -com_n 1 \
  -lr 0.3 -no-maxnorm \
  -keys ../$DIR/ltiling-keys.txt \
  -disttable ../$DIR/ltiling-disttable.txt
# epochs=1000, train_threads=2
#       -checkpoint ../$DIR/ltiling.bin \
fi

if [[ $1 == *Q* ]]
then logdo "evaluate LTiling_rsgd using other measures" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-disttable $DIR/ltiling-keys.txt $DIR/ltiling-disttable.txt | tee $DIR/log-ltiling-eval.txt
fi

if [[ $1 == *R* ]]
then logdo "convert Poincare 2D" poincare kx-evaluate.py ../$DIR/poincare2.bin.best -output3 ../$DIR/poincare2.txt | tee $DIR/log-poincare2-convert.txt
fi

if [[ $1 == *S* ]]
then logdo "evaluate poincare2 fulleval" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-poincare2 $DIR/poincare2.txt | tee $DIR/log-poincare2-fulleval.txt
fi

if [[ $1 == *T* ]]
then logdo "embed using d-Mercator" \
d-mercator/mercator -f -s 123456$SEED -o $DIR/d-mercator $IDIR/graph.txt
fi

if [[ $1 == *U* ]]
then logdo "evaluate d-mercator" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-mercator3 $DIR/d-mercator.inf_coord | tee $DIR/log-d-mercator-fulleval.txt
fi

if [[ $1 == *V* ]]
then logdo "penalty test" \
$DHRG -dhrg $DIR/bfkl -rdi $IDIR/graph.txt -penalty 2 -iterate 100 -cellcoords -eval-hdist | tee $DIR/log-penalty2-fulleval.txt
fi

if [[ $1 == *W* ]]
then logdo "evaluate lorentz3 fulleval" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-lorentz3 $DIR/lorentz3.txt | tee $DIR/log-lorentz3-fulleval.txt
fi

if [[ $1 == *X* ]]
then logdo "evaluate annealing2 fulleval" \
$DHRGSEED stamplen=9 -sag-creq 20000 -sag_gdist_load /tmp/graph-g711 -sagformat 1 -sag $IDIR/graph.txt -sag-bridging -rdi $IDIR/graph.txt -sag-fulleval -exit | tee $DIR/log-anneal2-fulleval.txt
fi

if [[ $1 == *Y* ]]
then logdo "evaluate annealing3 fulleval" \
$DHRGSEED stamplen=9 -geo 435h -bch-subcubes 2 -sag-creq 20000 -sag_gdist_load /tmp/graph-g435b2 -sagformat 1 -sag $IDIR/graph.txt -sag-bridging -rdi $IDIR/graph.txt -sag-fulleval -exit | tee $DIR/log-anneal3-fulleval.txt
fi

if [[ $1 == *Z* ]]
then log "evaluate euclid50 fulleval"
poincare kx-evaluate.py ../$DIR/euclidean50.bin.best -output3 ../$DIR/euclidean50.txt | tee $DIR/log-euclidean50-convert.txt
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-euclid $DIR/euclidean50.txt 50 | tee $DIR/log-euclidean50-fulleval.txt
log "evaluate euclid50 fulleval: finished"
fi

if [[ $1 == *i* ]]
then log "evaluate euclid200 fulleval"
poincare kx-evaluate.py ../$DIR/euclidean200.bin.best -output3 ../$DIR/euclidean200.txt | tee $DIR/log-euclidean200-convert.txt
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-euclid $DIR/euclidean200.txt 200 | tee $DIR/log-euclidean200-fulleval.txt
log "evaluate euclid200 fulleval: finished"
fi

if [[ $1 == *m* ]]
then log "evaluate poincare5 fulleval"
poincare kx-evaluate.py ../$DIR/poincare5.bin.best -output3 ../$DIR/poincare5.txt | tee $DIR/log-poincare5-convert.txt
$DHRG -eload $DIR/bfkl -rdi $IDIR/graph.txt -eval-poincare-any $DIR/poincare5.txt 5 | tee $DIR/log-poincare5-fulleval.txt
log "evaluate poincare5 fulleval: finished"
fi

if [[ $1 == *1* ]]
then log "embed using coalescent2"
bin/graph-to-matlab 2 < $IDIR/graph.txt > $DIR/coalescent2.m
indir coalescent_embedding/usage_example octave ../../$DIR/coalescent2.m | grep -v "Python communication" > $DIR/coalescent2-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/coalescent2-links.txt
log "embed using coalescent2: finished"
fi

if [[ $1 == *2* ]]
then logdo "evaluate coalescent2 fulleval" \
$DHRG -dhrg $DIR/coalescent2 -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-coalescent2-fulleval.txt
fi

if [[ $1 == *3* ]]
then log "embed using coalescent3"
bin/graph-to-matlab 3 < $IDIR/graph.txt > $DIR/coalescent3.m
indir coalescent_embedding/usage_example octave ../../$DIR/coalescent3.m | grep -v "Python communication" > $DIR/coalescent3-coordinates.txt
cp $DIR/bfkl-links.txt $DIR/coalescent3-links.txt
log "embed using coalescent3: finished"
fi

if [[ $1 == *4* ]]
then logdo "evaluate coalescent3 fulleval" \
$DHRG -eload $DIR/lorentz2 -rdi $IDIR/graph.txt -eval-lorentz3 $DIR/coalescent3-coordinates.txt | tee $DIR/log-coalescent3-fulleval.txt
fi

if [[ $1 == *5* ]]
then logdo "embed using kvk" \
sh kvk/bridging-hyperlink.sh $IDIR/graph.txt $DIR/bfkl-coordinates.txt $DIR/kvk-coordinates.txt $SEED | tee $DIR/log-kvk.txt
cp $DIR/bfkl-links.txt $DIR/kvk-links.txt
fi

if [[ $1 == *7* ]]
then logdo "embed using kvk knowing ground truth" \
sh kvk/bridging-hyperlink.sh $IDIR/graph.txt $IDIR/groundtruth-coordinates.txt $DIR/kvk-coordinates.txt $SEED | tee $DIR/log-kvk.txt
cp $DIR/bfkl-links.txt $DIR/kvk-links.txt
fi

if [[ $1 == *6* ]]
then logdo "evaluate kvk fulleval" \
$DHRG -dhrg $DIR/kvk -rdi $IDIR/graph.txt -eval-hdist | tee $DIR/log-kvk-fulleval.txt
fi

function timer {
  start=`grep "$1" $DIR/log.txt | grep -v finished | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  end=`grep "$1" $DIR/log.txt | grep finished | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  echo START=$start END=$end >> tmp
  if [ -n "$start" ]
  then if [ "$start" = "$end" ]
    then echo 1
    else echo $((`date "+%s" -d "$end"`-`date "+%s" -d "$start"`))
    fi
  else echo missing >> tmp
  fi
  }

# a part of experiments had to be redone in Sep 1X, 2025 due to an earlier bug
# but they had be done on a faster setup, so we still use the old time for the statistics
function timer-old {
  start=`grep "$1" $DIR/log.txt | grep -v finished | grep -v "Sep 1" | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  end=`grep "$1" $DIR/log.txt | grep finished | grep -v "Sep 1" | tail -n 1 | sed "s- : .*--" | sed "s-universal--"`
  echo START=$start END=$end >> tmp
  if [ -n "$start" ]
  then if [ "$start" = "$end" ]
    then echo 1
    else echo $((`date "+%s" -d "$end"`-`date "+%s" -d "$start"`))
    fi
  else echo missing >> tmp
  fi
  }

function stat {

  echo GRAPH=$GRAPH

  echo :: graph $GRAPH 1>&2

  echo n=`cat $DIR/log-bfkl-improve.txt | grep "Vertices by distance" | head -n 1 | sed "s-.*N = --" | sed "s-):--"`
  echo m=`wc -l $IDIR/graph.txt | sed "s- .*--"`
  echo entropy=`cat $DIR/log-bfkl-fulleval.txt | grep "entropy =" | head -n 1 | sed "s-.*entropy = --"`

  function fullevalp {
    if [ -f $2 ]
    then echo $1_au=`cat $2 | grep "entropy" | grep "radius =" | tail -n 1 | sed "s-.*radius = --" | sed "s- .*--"`
    echo $1_greedy_success=`cat $2 | grep "greedy routing:" | sed "s-.*success = --" | sed "s- .*--"`
    echo $1_greedy_stretch=`cat $2 | grep "greedy routing:" | sed "s-.*stretch = --" | sed "s- .*--"`
    echo $1_loglik=`cat $2 | grep loglikelihood | sed "s-.*loglikelihood = --"`
    echo $1_control=`cat $2 | grep "control" | tail -n 1 |  sed "s-.*control = --"`
    else echo "fulleval file missing: $2" 1>&2
    fi
    }

  function fulleval {
    if [ -f $2 ]
    then fullevalp $1 $2
    echo $1_meanrank=`cat $2 | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
    echo $1_map=`cat $2 | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`
    #echo mdim_lcont=`cat $DIR/log-d-mercator-fulleval.txt | grep "lcont" | sed "s-(R=.*--" | cut -c 17-`    
    else echo "fulleval file missing: $2" 1>&2
    fi
    }

  echo lorentz2_time=`timer "embed using Lorentz 2D"`
  echo lorentz2_meanrank=`cat $DIR/log-lorentz2-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo lorentz2_map=`cat $DIR/log-lorentz2-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
  echo lorentz2_eval_time=`cat $DIR/log-lorentz2-evaluate.txt | grep rank | tail -n 1 | sed "s-.*time: --"`
  fullevalp lorentz2 $DIR/log-lorentz2-fulleval.txt

  echo poincare3_time=`timer "embed using Poincare 3D"`
  echo poincare3_meanrank=`cat $DIR/log-poincare3-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo poincare3_map=`cat $DIR/log-poincare3-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
  fullevalp poincare3 $DIR/log-poincare3-fulleval.txt

  echo lorentz3_time=`timer "embed using Lorentz 3D"`
  fulleval lorentz3 $DIR/log-lorentz3-fulleval.txt

  echo poincare2_time=`timer "embed using Poincare 2D"`
  echo poincare2_meanrank=`cat $DIR/log-poincare2-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo poincare2_map=`cat $DIR/log-poincare2-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
  fullevalp poincare2 $DIR/log-poincare2-fulleval.txt

  echo mercator_fast_time=`timer "embed using Mercator fast"`
  fulleval mfast $DIR/log-mercator-fast-fulleval.txt

  echo mercator_full_time=`timer "embed using Mercator full"`
  fulleval mfull $DIR/log-mercator-full-fulleval.txt

  echo coalescent2_time=`timer "embed using coalescent2"`
  fulleval coalescent2 $DIR/log-coalescent2-fulleval.txt

  echo coalescent3_time=`timer "embed using coalescent3"`
  fulleval coalescent3 $DIR/log-coalescent3-fulleval.txt

  echo kvk_time=`timer-old "embed using kvk"`
  fulleval kvk $DIR/log-kvk-fulleval.txt

  echo mdim_time=`timer "embed using d-Mercator"`
  fulleval mdim $DIR/log-d-mercator-fulleval.txt

  echo bfkl_time=`timer "embed using BFKL"`
  fulleval bfkl $DIR/log-bfkl-fulleval.txt

  echo improve_bfkl_time=`timer "improve BFKL"`
  fulleval bfkl_ddhrg $DIR/log-ddhrg-bfkl-fulleval.txt
  echo bfkl_radius_pre_improve=`cat $DIR/log-bfkl-improve.txt | grep "Vertices by distance" -A 1 | head -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
  echo bfkl_radius_post_improve=`cat $DIR/log-bfkl-improve.txt | grep "Vertices by distance" -A 1 | tail -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
  echo bfkl_pre_meanrank=`cat $DIR/log-bfkl-improve.txt | grep meanrank | head -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
  echo bfkl_pre_map=`cat $DIR/log-bfkl-improve.txt | grep meanrank | head -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  #echo bfkl_pre_improve=`cat $DIR/log-bfkl-improve.txt | grep meanrank | head -n 1`
  echo bfkl_post_meanrank=`cat $DIR/log-bfkl-improve.txt | grep meanrank | tail -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
  echo bfkl_post_map=`cat $DIR/log-bfkl-improve.txt | grep meanrank | tail -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  #echo bfkl_post_improve=`cat $DIR/log-bfkl-improve.txt | grep meanrank | tail -n 1`

  echo dhrg_greedy_success=`cat $DIR/log-dhrg-greedy.txt | grep "greedy routing:" | sed "s-.*success = --" | sed "s- .*--"`
  echo dhrg_greedy_stretch=`cat $DIR/log-dhrg-greedy.txt | grep "greedy routing:" | sed "s-.*stretch = --" | sed "s- .*--"`

  echo improve_lorentz_time=`timer "improve Lorentz"`
  fulleval lorentz2_ddhrg $DIR/log-ddhrg-lorentz2-fulleval.txt
  echo lorentz2_radius_pre_improve=`cat $DIR/log-lorentz2-improve.txt | grep "Vertices by distance" -A 1 | head -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
  echo lorentz2_radius_post_improve=`cat $DIR/log-lorentz2-improve.txt | grep "Vertices by distance" -A 1 | tail -n 2 | tail -n 1 | sed "s-.* --" | sed "s-:.*--"`
  #echo lorentz2_result=`cat $DIR/log-lorentz2-evaluate.txt | grep rank | tail -n 1`
  echo lorentz2_pre_meanrank=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | head -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
  echo lorentz2_pre_map=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | head -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  #echo lorentz2_pre_improve=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | head -n 1`
  echo lorentz2_post_meanrank=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | tail -n 1 | sed "s-.*meanrank = --" | sed "s- .*--"`
  echo lorentz2_post_map=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | tail -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  #echo lorentz2_post_improve=`cat $DIR/log-lorentz2-improve.txt | grep meanrank | tail -n 1`

  fulleval penalty2 $DIR/log-penalty2-fulleval.txt

  echo : visualizations 1>&2

  echo hypviewer_au=`cat $DIR/log-hypviewer.txt | grep radius | cut -d "=" -f 2`
  echo hypviewer_meanrank=`cat $DIR/log-evaluate-hypviewer.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo hypviewer_map=`cat $DIR/log-evaluate-hypviewer.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`

  echo rogueviz_au=`cat $DIR/rogueviz-coordinates.txt | tail -n +3 | cut -d " " -f 2 | sort -g | tail -n 1`
  echo rogueviz_meanrank=`cat $DIR/log-rogueviz-fulleval.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo rogueviz_map=`cat $DIR/log-rogueviz-fulleval.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`

  echo mdim_au=`cat $DIR/log-d-mercator-fulleval.txt | grep radius | sed "s-.*radius = --" | sed "s- .*--"`

  echo : treerep 1>&2

  echo wrec_treerep_map=`cat $DIR/log-treerep.txt | grep "wrec mAP" | sed "s-.* = --"`
  echo norec_treerep_map=`cat $DIR/log-treerep.txt | grep "norec mAP" | sed "s-.* = --"`

  echo norec_treerep_hr_meanrank=`cat $DIR/log-treerep-eval.txt | grep MeanRank | head -n 1 | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo norec_treerep_hr_map=`cat $DIR/log-treerep-eval.txt | grep MeanRank | head -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  echo norec_treerep_hr_greedy_success=`cat $DIR/log-treerep-eval.txt | grep "greedy routing:" | head -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo norec_treerep_hr_greedy_stretch=`cat $DIR/log-treerep-eval.txt | grep "greedy routing:" | head -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`
  echo norec_treerep_tu=`cat $DIR/log-treerep-eval.txt | grep "maximum distance" | head -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--"`

  echo wrec_treerep_hr_meanrank=`cat $DIR/log-treerep-eval.txt | grep MeanRank | tail -n 1 | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo wrec_treerep_hr_map=`cat $DIR/log-treerep-eval.txt | grep MeanRank | tail -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  echo wrec_treerep_hr_greedy_success=`cat $DIR/log-treerep-eval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo wrec_treerep_hr_greedy_stretch=`cat $DIR/log-treerep-eval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`
  echo wrec_treerep_tu=`cat $DIR/log-treerep-eval.txt | grep "maximum distance" | tail -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--"`

  echo ltiling_time=`timer "embed using LTiling_rsgd"`
  echo ltiling_meanrank=`cat $DIR/log-ltiling-eval.txt | grep MeanRank | tail -n 1 | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo ltiling_map=`cat $DIR/log-ltiling-eval.txt | grep MeanRank | tail -n 1 | sed "s-.*MAP = --" | sed "s- .*--"`
  echo ltiling_greedy_success=`cat $DIR/log-ltiling-eval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo ltiling_greedy_stretch=`cat $DIR/log-ltiling-eval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`
  echo ltiling_diam=`cat $DIR/log-ltiling-eval.txt | grep "maximum distance" | tail -n 1 | sed "s-.*maximum distance = --" | sed "s- .*--"`

  echo : Euclidean/landscape/Poincare5 1>&2

  echo landscape200_meanrank=`cat $DIR/log-landscape-200.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo landscape200_map=`cat $DIR/log-landscape-200.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`
  #echo landscape199_meanrank=`cat $DIR/log-landscape-199b.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  #echo landscape199_map=`cat $DIR/log-landscape-199b.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`
  echo landscape50_meanrank=`cat $DIR/log-landscape-50.txt | grep MeanRank | sed "s-.*MeanRank = --" | sed "s- .*--"`
  echo landscape50_map=`cat $DIR/log-landscape-50.txt | grep MeanRank | sed "s-.*MAP = --" | sed "s- .*--"`

  echo poincare5_meanrank=`cat $DIR/log-poincare5-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo poincare5_map=`cat $DIR/log-poincare5-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
  echo euclidean50_meanrank=`cat $DIR/log-euclidean50-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo euclidean50_map=`cat $DIR/log-euclidean50-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`
  echo euclidean200_meanrank=`cat $DIR/log-euclidean200-evaluate.txt | grep rank | tail -n 1 | sed "s-Mean rank: --" | sed "s-, mAP.*--"`
  echo euclidean200_map=`cat $DIR/log-euclidean200-evaluate.txt | grep rank | tail -n 1 | sed "s-.*mAP rank: --" | sed "s-, time.*--"`

  echo euclidean50_greedy_success=`cat $DIR/log-euclidean50-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo euclidean50_greedy_stretch=`cat $DIR/log-euclidean50-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`
  echo euclidean200_greedy_success=`cat $DIR/log-euclidean200-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo euclidean200_greedy_stretch=`cat $DIR/log-euclidean200-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`
  echo poincare5_greedy_success=`cat $DIR/log-poincare5-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*success = --" | sed "s- .*--"`
  echo poincare5_greedy_stretch=`cat $DIR/log-poincare5-fulleval.txt | grep "greedy routing:" | tail -n 1 | sed "s-.*stretch = --" | sed "s- .*--"`

  fulleval anneal2 $DIR/log-anneal2-fulleval.txt
  fulleval anneal3 $DIR/log-anneal3-fulleval.txt

  echo : done 1>&2

  #echo landscape199=`cat $DIR/log-landscape-199b.txt | grep MeanRank`
  #echo landscape50=`cat $DIR/log-landscape-50.txt | grep MeanRank`
  #echo BFKL embed time: `grep "embed using BFKL" $DIR/log.txt`
  }

if [ -n "$PROCESS_DRY" ]
then
echo "#" done
else
stat > $DIR/summary.txt
fi

if [[ $1 == *Z* ]]
then log "experiment complete"
fi

