def get_similarity(rep_3d_src: dict[str, (str, str)], rep_3d_dst_list: list[dict], sim_type: str):
    """
    Calculate the maximum similarity between source 3D representation and destination representations.
    
    Args:
        rep_3d_src: Source 3D representation (Dict[face, (orientation, color)])
        rep_3d_dst_list: List of destination 3D representations to compare against
        sim_type: Type of similarity to calculate - 'scene' (3 faces) or 'rep' (6 faces)
    
    Returns:
        similarities: Maximum similarity scores for [complete, color, orientation] metrics
                     [0] = complete match (orientation + color)
                     [1] = color similarity only
                     [2] = orientation similarity only
    
    Raises:
        ValueError: If sim_type is not 'scene' or 'rep'
    """
    # Define which faces to compare based on similarity type
    if sim_type == 'scene':
        face_list = ['front', 'top', 'right']  # 3D scene faces only
    elif sim_type == 'rep':
        face_list = ['front', 'top', 'right', 'back', 'left', 'bottom']  # All 6 faces
    else:
        raise ValueError('sim_type must be either "scene" or "rep"')
    
    similarities = [0, 0, 0] # [complete, color, orientation]
    for rep_3d_dst in rep_3d_dst_list:
        current_sim = [0, 0, 0]
        for face in face_list:
            if rep_3d_src[face][0] == rep_3d_dst[face][0]: # orientation similarity
                current_sim[2] += 1
            if rep_3d_src[face][1] == rep_3d_dst[face][1]: # color similarity
                current_sim[1] += 1
                if rep_3d_src[face][0] == rep_3d_dst[face][0]:
                    current_sim[0] += 1
        
        for i in range(3):
            similarities[i] = max(similarities[i], current_sim[i])
    
    return similarities
