
##### 3D #####
## pattern representation (color + orientation)
FACE_NAMES = ('front', 'top',  'back', 'bottom', 'left', 'right')
FACE_IDX = {name: idx for idx, name in enumerate(FACE_NAMES)}
FACE_ADJACENCY = {
    'front': ['top', 'right', 'bottom', 'left'],
    'top': ['back', 'right', 'front', 'left'],
    'back': ['bottom', 'right', 'top', 'left'],
    'bottom': ['front', 'right', 'back', 'left'],
    'left': ['top', 'front', 'bottom', 'back'],
    'right': ['top', 'back', 'bottom', 'front'],
}

def get_3d_rep(ori_idx: list[int], colors: list[str]): # 3d representation (cube view representation)
    assert len(ori_idx) == 6
    assert len(colors) == 6

    ori_3d = [
        FACE_ADJACENCY[FACE_NAMES[i]][idx]
        for i, idx in enumerate(ori_idx)
    ]
    cube_view = {
        FACE_NAMES[i]: (ori_3d[i], colors[i]) 
        for i in range(6)
    }
    return cube_view

def from_3d_rep(cube_view: dict):
    assert len(cube_view) == 6

    ori_idx = []
    colors = []
    for face in FACE_NAMES:
        face_adj = cube_view[face][0]
        face_adj_list = FACE_ADJACENCY[face]
        ori_idx.append(face_adj_list.index(face_adj))
        colors.append(cube_view[face][1])

    return ori_idx, colors


# example of pettern representation
# 'front', 'top',  'back', 'bottom', 'left', 'right'
# (('red','top'), ('blue','front'), ('green','bottom'), ('brown','back'), ('purple','top'), ('cyan','top'))

## group representation
# representation of the cube rotation group (following chiral octahedral symmetry)
def gen_group_elements(face_names, init_element = (0,1,2,3,4,5)): 
    assert face_names == ('front', 'top',  'back', 'bottom', 'left', 'right')

    # 2 generators of the cube rotation group
    def rot_a(ele): # front-center pivot, clockwise 90 rotation
        # top -> right, right -> bottom, bottom -> left, left -> top
        return (ele[0], ele[4], ele[2], ele[5], ele[3], ele[1])
    def rot_b(ele): # front-top pivot, 180 rotation
        # left <-> right, top <-> front, bottom <-> back
        return (ele[1], ele[0], ele[3], ele[2], ele[5], ele[4])

    element_set = set()
    st = [init_element] # stack for BFS
    while st: # generate all the 24 possible rotations
        ele = st.pop()
        element_set.add(ele)

        ele_a = rot_a(ele)
        if ele_a not in element_set:
            element_set.add(ele_a)
            st.append(ele_a)

        ele_b = rot_b(ele)
        if ele_b not in element_set:
            element_set.add(ele_b)
            st.append(ele_b)
    return element_set

group_elements = gen_group_elements(FACE_NAMES)

def rotate_cube(cube_view, group_ele):
    new_cube_view = dict()
    for i in range(6):
        src_face = FACE_NAMES[group_ele[i]]
        dst_face = FACE_NAMES[i]

        src_ori_idx = FACE_IDX[cube_view[src_face][0]] # oriented to a face labeled with idx
        dst_color = cube_view[src_face][1]
        for dst_ori_idx in range(6):
            if group_ele[dst_ori_idx] == src_ori_idx: # find the new position of the face
                dst_ori = FACE_NAMES[dst_ori_idx]
                new_cube_view[dst_face] = (dst_ori, dst_color)
                break

    return new_cube_view

## 2D representation
NET_ORIENTATIONS = {
    0: 'up',
    1: 'right',
    2: 'down',
    3: 'left',
}

NET_COORDINATES = {
    '141-00': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 0], [2, 0]],
    '141-01': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 0], [2, 1]],
    '141-02': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 0], [2, 2]],
    '141-03': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 0], [2, 3]],
    '141-11': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 1], [2, 1]],
    '141-12': [[1, 0], [1, 1], [1, 2], [1, 3], [0, 1], [2, 2]],
    '132-1' : [[1, 1], [1, 2], [1, 3], [2, 0], [0, 1], [2, 1]],
    '132-2' : [[1, 1], [1, 2], [1, 3], [2, 0], [0, 2], [2, 1]],
    '132-3' : [[1, 1], [1, 2], [1, 3], [2, 0], [0, 3], [2, 1]],
    '222'   : [[1, 1], [1, 2], [0, 3], [2, 0], [0, 2], [2, 1]],
    '33'    : [[0, 2], [0, 3], [0, 4], [1, 1], [1, 0], [1, 2]],
}

NET_TYPES = list(NET_COORDINATES.keys())

NET_MATCH = {
    '141-00': [0,0,0,0,0,0],
    '141-01': [0,0,0,0,0,3], # rotate 90 deg clockwise
    '141-02': [0,0,0,0,0,2], # rotate 180 deg clockwise
    '141-03': [0,0,0,0,0,1], # rotate 270 deg clockwise
    '141-11': [0,0,0,0,1,3],
    '141-12': [0,0,0,0,1,2],
    '132-1' : [0,0,0,3,0,0],
    '132-2' : [0,0,0,3,1,0],
    '132-3' : [0,0,0,3,2,0],
    '222'   : [0,0,3,3,1,0],
    '33'    : [0,0,0,3,2,0],
}


def get_2d_rep(ori_idx: list[int], colors: list[str], net_name: str): # 2d representation (net view representation)
    assert len(ori_idx) == 6
    assert len(colors) == 6

    net = NET_COORDINATES[net_name]
    net_rot = NET_MATCH[net_name] # pattern rotation
    x_min = min(coor[0] for coor in net)
    y_min = min(coor[1] for coor in net)

    net_view = dict()
    for face_idx, face_coor in enumerate(net):
        x, y = face_coor
        x = x - x_min
        y = y - y_min
        ori_idx_rot = ori_idx[face_idx] + net_rot[face_idx]
        ori_idx_rot %= 4 # circular
        ori = NET_ORIENTATIONS[ori_idx_rot]
        color = colors[face_idx]

        net_view[f'{x}_{y}'] = (ori, color)
    return net_view

