#!/usr/bin/env python
# coding=utf-8
# Copyright 2024 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and

import argparse
# import logging
import math
import os
import random
import shutil
from pathlib import Path
import memory_management
import numpy as np
import PIL
import safetensors
import safetensors.torch as sf
from safetensors.torch import load_file
import torch
import torch.nn.functional as F
import torch.utils.checkpoint
import transformers
from accelerate import Accelerator
from accelerate.logging import get_logger
from accelerate.utils import ProjectConfiguration, set_seed
from huggingface_hub import create_repo, upload_folder
from packaging import version
from PIL import Image
from torch.utils.data import Dataset
from torchvision import transforms
from tqdm.auto import tqdm
from transformers import CLIPTextModel, CLIPTextModelWithProjection, CLIPTokenizer

import diffusers
from diffusers import (
  AutoencoderKL,
  DDPMScheduler,
  DiffusionPipeline,
  DPMSolverMultistepScheduler,
  UNet2DConditionModel,
)
from diffusers import StableDiffusionPipeline, DPMSolverMultistepScheduler

from diffusers.optimization import get_scheduler
from diffusers.utils import check_min_version, is_wandb_available
from diffusers.utils.hub_utils import load_or_create_model_card, populate_model_card
from diffusers.utils.import_utils import is_xformers_available

from lib_layerdiffuse.vae import TransparentVAEDecoder, TransparentVAEEncoder
from lib_layerdiffuse.utils import download_model
from diffusers_kdiffusion_sdxl import KDiffusionStableDiffusionXLPipeline
from diffusers import StableDiffusionPipeline, DPMSolverMultistepScheduler

from diffusers_kdiffusion_sdxl import *
import logging
import json

if is_wandb_available():
  import wandb

if version.parse(version.parse(PIL.__version__).base_version) >= version.parse("9.1.0"):
  PIL_INTERPOLATION = {
    "linear": PIL.Image.Resampling.BILINEAR,
    "bilinear": PIL.Image.Resampling.BILINEAR,
    "bicubic": PIL.Image.Resampling.BICUBIC,
    "lanczos": PIL.Image.Resampling.LANCZOS,
    "nearest": PIL.Image.Resampling.NEAREST,
  }
else:
  PIL_INTERPOLATION = {
    "linear": PIL.Image.LINEAR,
    "bilinear": PIL.Image.BILINEAR,
    "bicubic": PIL.Image.BICUBIC,
    "lanczos": PIL.Image.LANCZOS,
    "nearest": PIL.Image.NEAREST,
  }
# ------------------------------------------------------------------------------


# Will error if the minimal version of diffusers is not installed. Remove at your own risks.
check_min_version("0.28.0.dev0")

logger = get_logger(__name__)


def save_progress(text_encoder_1, placeholder_token_ids_1, text_encoder_2, placeholder_token_ids_2, accelerator, args,
                  save_path, safe_serialization=True):
  logger.info("Saving embeddings")
  learned_embeds_1 = (
    accelerator.unwrap_model(text_encoder_1)
    .get_input_embeddings()
    .weight[min(placeholder_token_ids_1): max(placeholder_token_ids_1) + 1]
  )
  learned_embeds_2 = (
    accelerator.unwrap_model(text_encoder_2)
    .get_input_embeddings()
    .weight[min(placeholder_token_ids_2): max(placeholder_token_ids_2) + 1]
  )
  # learned_embeds_dict = {}
  learned_embeds_dict_1 = {args.placeholder_token: learned_embeds_1.detach().cpu()}
  learned_embeds_dict_2 = {args.placeholder_token: learned_embeds_2.detach().cpu()}

  # Get the basename of the original save path
  base_name = os.path.basename(save_path)
  new_base_name_1 = os.path.splitext(base_name)[0] + '_1' + os.path.splitext(base_name)[1]
  new_save_path_1 = os.path.join(os.path.dirname(save_path), new_base_name_1)
  new_base_name_2 = os.path.splitext(base_name)[0] + '_2' + os.path.splitext(base_name)[1]
  new_save_path_2 = os.path.join(os.path.dirname(save_path), new_base_name_2)

  if safe_serialization:
    safetensors.torch.save_file(learned_embeds_dict_1, new_save_path_1, metadata={"format": "pt"})
    safetensors.torch.save_file(learned_embeds_dict_2, new_save_path_2, metadata={"format": "pt"})
  else:
    torch.save(learned_embeds_dict_1, new_save_path_1)
    torch.save(learned_embeds_dict_2, new_save_path_2)


def parse_args():
  parser = argparse.ArgumentParser(description="Simple example of a training script.")
  parser.add_argument(
    "--save_steps",
    type=int,
    default=100,
    help="Save learned_embeds.bin every X updates steps.",
  )
  parser.add_argument(
    "--save_as_full_pipeline",
    action="store_true",
    help="Save the complete stable diffusion pipeline.",
  )
  parser.add_argument(
    "--num_vectors",
    type=int,
    default=2,
    help="How many textual inversion vectors shall be used to learn the concept.",
  )

  parser.add_argument(
    "--strength",
    type=float,
    default=0.7,
    help="",
  )

  parser.add_argument(
    "--pretrained_model_name_or_path",
    type=str,
    default="SG161222/RealVisXL_V4.0",
    help="Path to pretrained model or model identifier from huggingface.co/models.",
  )
  parser.add_argument(
    "--revision",
    type=str,
    default=None,
    required=False,
    help="Revision of pretrained model identifier from huggingface.co/models.",
  )
  parser.add_argument(
    "--variant",
    type=str,
    default=None,
    help="Variant of the model files of the pretrained model identifier from huggingface.co/models, 'e.g.' fp16",
  )
  #   parser.add_argument(
  #     "--train_data_dir", type=str, default=None, required=True, help="A folder containing the training data."
  #   )

  parser.add_argument(
    "--placeholder_token",
    type=str,
    default="<embed>",
    help="A token to use as a placeholder for the concept.",
  )
  parser.add_argument(
    "--initializer_token", type=str, default=None, required=True, help="A token to use as initializer word."
  )
  parser.add_argument("--learnable_property", type=str, default="object", help="Choose between 'object' and 'style'")
  parser.add_argument("--repeats", type=int, default=100, help="How many times to repeat the training data.")
  parser.add_argument(
    "--output_dir",
    type=str,
    default="text-inversion-model-steps1000-lr1e-7/n_25_s_0.7",
    help="The output directory where the model predictions and checkpoints will be written.",
  )
  parser.add_argument("--seed", type=int, default=None, help="A seed for reproducible training.")
  parser.add_argument(
    "--resolution",
    type=int,
    default=512,
    help=(
      "The resolution for input images, all the images in the train/validation dataset will be resized to this"
      " resolution"
    ),
  )
  parser.add_argument(
    "--center_crop", action="store_true", help="Whether to center crop images before resizing to resolution."
  )
  parser.add_argument(
    "--train_batch_size", type=int, default=1, help="Batch size (per device) for the training dataloader."
  )
  parser.add_argument("--num_train_epochs", type=int, default=100)
  parser.add_argument(
    "--max_train_steps",
    type=int,
    default=1000,
    help="Total number of training steps to perform.  If provided, overrides num_train_epochs.",
  )
  parser.add_argument(
    "--gradient_accumulation_steps",
    type=int,
    default=1,
    help="Number of updates steps to accumulate before performing a backward/update pass.",
  )
  parser.add_argument(
    "--gradient_checkpointing",
    action="store_true",
    help="Whether or not to use gradient checkpointing to save memory at the expense of slower backward pass.",
  )
  parser.add_argument(
    "--learning_rate",
    type=float,
    default=1e-7,
    help="Initial learning rate (after the potential warmup period) to use.",
  )
  parser.add_argument(
    "--scale_lr",
    action="store_true",
    default=False,
    help="Scale the learning rate by the number of GPUs, gradient accumulation steps, and batch size.",
  )
  parser.add_argument(
    "--lr_scheduler",
    type=str,
    default="constant",
    help=(
      'The scheduler type to use. Choose between ["linear", "cosine", "cosine_with_restarts", "polynomial",'
      ' "constant", "constant_with_warmup"]'
    ),
  )
  parser.add_argument(
    "--lr_warmup_steps", type=int, default=500, help="Number of steps for the warmup in the lr scheduler."
  )
  parser.add_argument(
    "--lr_num_cycles",
    type=int,
    default=1,
    help="Number of hard resets of the lr in cosine_with_restarts scheduler.",
  )
  parser.add_argument(
    "--dataloader_num_workers",
    type=int,
    default=0,
    help=(
      "Number of subprocesses to use for data loading. 0 means that the data will be loaded in the main process."
    ),
  )
  parser.add_argument(
    "--use_8bit_adam", action="store_true", help="Whether or not to use 8-bit Adam from bitsandbytes."
  )
  parser.add_argument("--adam_beta1", type=float, default=0.9, help="The beta1 parameter for the Adam optimizer.")
  parser.add_argument("--adam_beta2", type=float, default=0.999, help="The beta2 parameter for the Adam optimizer.")
  parser.add_argument("--adam_weight_decay", type=float, default=1e-2, help="Weight decay to use.")
  parser.add_argument("--adam_epsilon", type=float, default=1e-08, help="Epsilon value for the Adam optimizer")
  parser.add_argument("--push_to_hub", action="store_true", help="Whether or not to push the model to the Hub.",
                      default="false")
  parser.add_argument("--hub_token", type=str, default=None, help="The token to use to push to the Model Hub.")
  parser.add_argument(
    "--hub_model_id",
    type=str,
    default=None,
    help="The name of the repository to keep in sync with the local `output_dir`.",
  )
  parser.add_argument(
    "--logging_dir",
    type=str,
    default="logs",
    help=(
      "[TensorBoard](https://www.tensorflow.org/tensorboard) log directory. Will default to"
      " *output_dir/runs/**CURRENT_DATETIME_HOSTNAME***."
    ),
  )
  parser.add_argument(
    "--mixed_precision",
    type=str,
    default="no",
    choices=["no", "fp16", "bf16"],
    help=(
      "Whether to use mixed precision. Choose"
      "between fp16 and bf16 (bfloat16). Bf16 requires PyTorch >= 1.10."
      "and an Nvidia Ampere GPU."
    ),
  )
  parser.add_argument(
    "--allow_tf32",
    action="store_true",
    help=(
      "Whether or not to allow TF32 on Ampere GPUs. Can be used to speed up training. For more information, see"
      " https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices"
    ),
  )
  parser.add_argument(
    "--report_to",
    type=str,
    default='wandb',
    help=(
      'The integration to report the results and logs to. Supported platforms are `"tensorboard"`'
      ' (default), `"wandb"` and `"comet_ml"`. Use `"all"` to report to all integrations.'
    ),
  )
  parser.add_argument(
    "--validation_prompt",
    type=str,
    default=None,
    help="A prompt that is used during validation to verify that the model is learning.",
  )
  parser.add_argument(
    "--num_validation_images",
    type=int,
    default=4,
    help="Number of images that should be generated during validation with `validation_prompt`.",
  )
  parser.add_argument(
    "--validation_steps",
    type=int,
    default=None,
    help=(
      "Run validation every X steps. Validation consists of running the prompt"
      " `args.validation_prompt` multiple times: `args.num_validation_images`"
      " and logging the images."
    ),
  )
  parser.add_argument("--local_rank", type=int, default=-1, help="For distributed training: local_rank")
  parser.add_argument(
    "--checkpointing_steps",
    type=int,
    default=500,
    help=(
      "Save a checkpoint of the training state every X updates. These checkpoints are only suitable for resuming"
      " training using `--resume_from_checkpoint`."
    ),
  )
  parser.add_argument(
    "--checkpoints_total_limit",
    type=int,
    default=None,
    help=("Max number of checkpoints to store."),
  )
  parser.add_argument(
    "--resume_from_checkpoint",
    type=str,
    default=None,
    help=(
      "Whether training should be resumed from a previous checkpoint. Use a path saved by"
      ' `--checkpointing_steps`, or `"latest"` to automatically select the last available checkpoint.'
    ),
  )
  parser.add_argument(
    "--enable_xformers_memory_efficient_attention", action="store_true", help="Whether or not to use xformers."
  )

  parser.add_argument(
    "--train_dir_list", type=str, default=None, help="The input text file for dataset path."
  )

  parser.add_argument(
    "--reverse",
    action="store_true"
  )

  args = parser.parse_args()
  env_local_rank = int(os.environ.get("LOCAL_RANK", -1))
  if env_local_rank != -1 and env_local_rank != args.local_rank:
    args.local_rank = env_local_rank

  return args


imagenet_templates_small = [
  "a photo of a {}",
  "a rendering of a {}",
  "a cropped photo of the {}",
  "the photo of a {}",
  "a photo of a clean {}",
  "a photo of a dirty {}",
  "a dark photo of the {}",
  "a photo of my {}",
  "a photo of the cool {}",
  "a close-up photo of a {}",
  "a bright photo of the {}",
  "a cropped photo of a {}",
  "a photo of the {}",
  "a good photo of the {}",
  "a photo of one {}",
  "a close-up photo of the {}",
  "a rendition of the {}",
  "a photo of the clean {}",
  "a rendition of a {}",
  "a photo of a nice {}",
  "a good photo of a {}",
  "a photo of the nice {}",
  "a photo of the small {}",
  "a photo of the weird {}",
  "a photo of the large {}",
  "a photo of a cool {}",
  "a photo of a small {}",
]


class TextualInversionDataset(Dataset):
  def __init__(
    self,
    data_root,
    tokenizer_1,
    tokenizer_2,
    learnable_property="object",  # [object, style]
    size=512,
    repeats=100,
    interpolation="bicubic",
    flip_p=0.5,
    set="train",
    placeholder_token="*",
    center_crop=False,
  ):
    self.data_root = data_root
    self.tokenizer_1 = tokenizer_1
    self.tokenizer_2 = tokenizer_2
    self.learnable_property = learnable_property
    self.size = size
    self.placeholder_token = placeholder_token
    self.center_crop = center_crop
    self.flip_p = flip_p
    self.image_paths = []
    
    image_extensions = ('.png', '.jpg', '.jpeg', '.bmp', '.gif', '.tiff')
    for file_path in os.listdir(self.data_root):
        # Check if the file has a valid image extension
        if file_path.lower().endswith(image_extensions):
            full_path = os.path.join(self.data_root, file_path)
            try:
                with Image.open(full_path) as img:
                    # Check if the image mode is RGBA
                    if img.mode == "RGBA":
                        self.image_paths.append(full_path)
            except IOError:
                print(f"Failed to open {full_path}")
    print(f"{len(self.image_paths)} RGBA images are loaded in {self.data_root}")

    self.num_images = len(self.image_paths)
    self._length = self.num_images

    if set == "train":
      self._length = self.num_images * repeats

    self.interpolation = {
      "linear": PIL_INTERPOLATION["linear"],
      "bilinear": PIL_INTERPOLATION["bilinear"],
      "bicubic": PIL_INTERPOLATION["bicubic"],
      "lanczos": PIL_INTERPOLATION["lanczos"],
    }[interpolation]

    self.templates = imagenet_style_templates_small if learnable_property == "style" else imagenet_templates_small
    self.flip_transform = transforms.RandomHorizontalFlip(p=self.flip_p)
    self.crop = transforms.CenterCrop(size) if center_crop else transforms.RandomCrop(size)

  def __len__(self):
    return self._length

  def __getitem__(self, i):
    example = {}
    image = Image.open(self.image_paths[i % self.num_images])
    placeholder_string = self.placeholder_token
    text = random.choice(self.templates).format(placeholder_string)
    example["original_size"] = (image.height, image.width)
    image = image.resize((self.size, self.size), resample=Image.Resampling.LANCZOS)

    image = self.flip_transform(image)

    if self.center_crop:
      y1 = max(0, int(round((image.height - self.size) / 2.0)))
      x1 = max(0, int(round((image.width - self.size) / 2.0)))
      image = self.crop(image)
    else:
      y1, x1, h, w = self.crop.get_params(image, (self.size, self.size))
      image = transforms.functional.crop(image, y1, x1, h, w)

    example["crop_top_left"] = (y1, x1)

    example["input_ids_1"] = self.tokenizer_1(
      text,
      padding="max_length",
      truncation=True,
      max_length=self.tokenizer_1.model_max_length,
      return_tensors="pt",
    ).input_ids[0]

    example["input_ids_2"] = self.tokenizer_2(
      text,
      padding="max_length",
      truncation=True,
      max_length=self.tokenizer_2.model_max_length,
      return_tensors="pt",
    ).input_ids[0]

    # default to score-sde preprocessing
    img = np.array(image).astype(np.uint8)
    # image = Image.fromarray(img)
    # image = self.flip_transform(image)
    # image = np.array(image).astype(np.uint8)
    # image = (image / 127.5 - 1.0).astype(np.float32)
    # example["pixel_values"] = torch.from_numpy(image).permute(2, 0, 1)
    example["pixel_values"] = img
    return example


from types import SimpleNamespace

class Sigma_Scheduler:
  def __init__(self, timesteps=1000, linear_start=0.00085, linear_end=0.012):
    self.device = "cpu"
    betas = torch.linspace(linear_start ** 0.5, linear_end ** 0.5, timesteps, dtype=torch.float64) ** 2
    alphas = 1. - betas
    alphas_cumprod = torch.tensor(np.cumprod(alphas, axis=0), dtype=torch.float32).clone().detach()
    self.sigmas = ((1 - alphas_cumprod) / alphas_cumprod) ** 0.5
    self.log_sigmas = self.sigmas.log()
    self.sigma_data = 1.0

  @property
  def sigma_min(self):
    return self.sigmas[0]

  @property
  def sigma_max(self):
    return self.sigmas[-1]

  def get_sigmas_karras(self, n, rho=7.):
    ramp = torch.linspace(0, 1, n)
    min_inv_rho = self.sigma_min ** (1 / rho)
    max_inv_rho = self.sigma_max ** (1 / rho)
    sigmas = (max_inv_rho + ramp * (min_inv_rho - max_inv_rho)) ** rho
    return torch.cat([sigmas, sigmas.new_zeros([1])])

  def timestep(self, sigma):
    log_sigma = sigma.log()
    dists = log_sigma.to(self.log_sigmas.device) - self.log_sigmas[:, None]
    return dists.abs().argmin(dim=0).view(sigma.shape).to(sigma.device)


def main():
  args = parse_args()
  if args.report_to == "wandb" and args.hub_token is not None:
    raise ValueError(
      "You cannot use both --report_to=wandb and --hub_token due to a security risk of exposing your token."
      " Please use `huggingface-cli login` to authenticate with the Hub."
    )
  print(args.output_dir, args.logging_dir)
  logging_dir = os.path.join(args.output_dir, args.logging_dir)
  accelerator_project_config = ProjectConfiguration(project_dir=args.output_dir, logging_dir=logging_dir)
  accelerator = Accelerator(
    gradient_accumulation_steps=args.gradient_accumulation_steps,
    mixed_precision=args.mixed_precision,
    log_with=args.report_to,
    project_config=accelerator_project_config,
  )

  # Disable AMP for MPS.
  if torch.backends.mps.is_available():
    accelerator.native_amp = False

  if args.report_to == "wandb":
    if not is_wandb_available():
      raise ImportError("Make sure to install wandb if you want to use it for logging during training.")

  # Make one log on every process with the configuration for debugging.
  logging.basicConfig(
    format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
    datefmt="%m/%d/%Y %H:%M:%S",
    level=logging.INFO,
  )
  logger.info(accelerator.state, main_process_only=False)
  if accelerator.is_local_main_process:
    transformers.utils.logging.set_verbosity_warning()
    diffusers.utils.logging.set_verbosity_info()
  else:
    transformers.utils.logging.set_verbosity_error()
    diffusers.utils.logging.set_verbosity_error()

  # If passed along, set the training seed now.
  if args.seed is not None:
    set_seed(args.seed)

  # Handle the repository creation
  if accelerator.is_main_process:
    if args.output_dir is not None:
      os.makedirs(args.output_dir, exist_ok=True)

  # Load tokenizer
  tokenizer_1 = CLIPTokenizer.from_pretrained(args.pretrained_model_name_or_path, subfolder="tokenizer")
  tokenizer_2 = CLIPTokenizer.from_pretrained(args.pretrained_model_name_or_path, subfolder="tokenizer_2")

  text_encoder_1 = CLIPTextModel.from_pretrained(
    args.pretrained_model_name_or_path, subfolder="text_encoder", revision=args.revision
  )
  text_encoder_2 = CLIPTextModelWithProjection.from_pretrained(
    args.pretrained_model_name_or_path, subfolder="text_encoder_2", revision=args.revision
  )
  vae = AutoencoderKL.from_pretrained(
    args.pretrained_model_name_or_path, subfolder="vae", revision=args.revision, variant=args.variant
  )
  unet = UNet2DConditionModel.from_pretrained(
    args.pretrained_model_name_or_path, subfolder="unet", revision=args.revision, variant=args.variant
  )

  # Modify unet attention
  path_ld_diffusers_sdxl_attn = download_model(
    url='https://huggingface.co/lllyasviel/LayerDiffuse_Diffusers/resolve/main/ld_diffusers_sdxl_attn.safetensors',
    local_path='path/to/icml24/LayerDiffuse_DiffusersCLI/models/ld_diffusers_sdxl_attn.safetensors'
  )

  sd_offset = sf.load_file(path_ld_diffusers_sdxl_attn)
  sd_origin = unet.state_dict()
  keys = sd_origin.keys()
  sd_merged = {}
  for k in sd_origin.keys():
    if k in sd_offset:
      sd_merged[k] = sd_origin[k] + sd_offset[k]
    else:
      sd_merged[k] = sd_origin[k]
  unet.load_state_dict(sd_merged, strict=True)
  del sd_offset, sd_origin, sd_merged, keys, k

  # Add the placeholder token in tokenizer_1
  placeholder_tokens = [args.placeholder_token]

  if args.num_vectors < 1:
    raise ValueError(f"--num_vectors has to be larger or equal to 1, but is {args.num_vectors}")

  # add dummy tokens for multi-vector
  additional_tokens = []
  for i in range(1, args.num_vectors):
    additional_tokens.append(f"{args.placeholder_token}_{i}")
  placeholder_tokens += additional_tokens

  # add placeholder token to tokenizer_1
  num_added_tokens = tokenizer_1.add_tokens(placeholder_tokens)
  if num_added_tokens != args.num_vectors:
    raise ValueError(
      f"The tokenizer_1 already contains the token {args.placeholder_token}. Please pass a different"
      " `placeholder_token` that is not already in the tokenizer."
    )
    # Convert the initializer_token, placeholder_token to ids
  token_ids_1 = tokenizer_1.encode(args.initializer_token, add_special_tokens=False)
  # Check if initializer_token is a single token or a sequence of tokens
  if len(token_ids_1) > 1:
    raise ValueError("The initializer token must be a single token.")
  initializer_token_id_1 = token_ids_1[0]
  placeholder_token_ids_1 = tokenizer_1.convert_tokens_to_ids(placeholder_tokens)
  # Resize the token embeddings as we are adding new special tokens to the tokenizer
  text_encoder_1.resize_token_embeddings(len(tokenizer_1))
  # Initialise the newly added placeholder token with the embeddings of the initializer token
  token_embeds_1 = text_encoder_1.get_input_embeddings().weight.data
  with torch.no_grad():
    for token_id in placeholder_token_ids_1:
      token_embeds_1[token_id] = token_embeds_1[initializer_token_id_1].clone()

  # add placeholder token to tokenizer_2
  num_added_tokens = tokenizer_2.add_tokens(placeholder_tokens)
  if num_added_tokens != args.num_vectors:
    raise ValueError(
      f"The tokenizer_2 already contains the token {args.placeholder_token}. Please pass a different"
      " `placeholder_token` that is not already in the tokenizer."
    )
    # Convert the initializer_token, placeholder_token to ids
  token_ids_2 = tokenizer_2.encode(args.initializer_token, add_special_tokens=False)
  # Check if initializer_token is a single token or a sequence of tokens
  if len(token_ids_2) > 1:
    raise ValueError("The initializer token must be a single token.")
  initializer_token_id_2 = token_ids_2[0]
  placeholder_token_ids_2 = tokenizer_2.convert_tokens_to_ids(placeholder_tokens)
  # Resize the token embeddings as we are adding new special tokens to the tokenizer
  text_encoder_2.resize_token_embeddings(len(tokenizer_2))
  # Initialise the newly added placeholder token with the embeddings of the initializer token
  token_embeds_2 = text_encoder_2.get_input_embeddings().weight.data
  with torch.no_grad():
    for token_id in placeholder_token_ids_2:
      token_embeds_2[token_id] = token_embeds_2[initializer_token_id_2].clone()

  # Freeze vae and unet
  vae.requires_grad_(False)
  unet.requires_grad_(False)
  # text_encoder_2.requires_grad_(False)

  # Freeze all parameters except for the token embeddings in text encoder
  text_encoder_1.text_model.encoder.requires_grad_(False)
  text_encoder_1.text_model.final_layer_norm.requires_grad_(False)
  text_encoder_1.text_model.embeddings.position_embedding.requires_grad_(False)

  # Freeze all parameters except for the token embeddings in text encoder2
  text_encoder_2.text_model.encoder.requires_grad_(False)
  text_encoder_2.text_model.final_layer_norm.requires_grad_(False)
  text_encoder_2.text_model.embeddings.position_embedding.requires_grad_(False)

  path_list = []
  # 读取args.train_dir_list,其是提个.txt文件，每行为一个数据集文件夹的绝对路径
  with open(args.train_dir_list, 'r') as f:
    lines = f.readlines()
  for line in lines:
    line = line.strip()
    if not os.path.exists(line):
      raise ValueError(f"Dataset path {line} does not exist.")
    # 把它存到path_list中
    path_list.append(line)

  weight_dtype = torch.float32
  if accelerator.mixed_precision == "fp16":
    weight_dtype = torch.float16
  elif accelerator.mixed_precision == "bf16":
    weight_dtype = torch.bfloat16
  unet.to(accelerator.device, dtype=weight_dtype)
  vae.to(accelerator.device, dtype=weight_dtype)

  # Move vae and unet and text_encoder_2 to device and cast to weight_dtype
  # text_encoder_2.to(accelerator.device, dtype=weight_dtype)
  path_ld_diffusers_sdxl_vae_transparent_encoder = download_model(
    url='https://huggingface.co/lllyasviel/LayerDiffuse_Diffusers/resolve/main/ld_diffusers_sdxl_vae_transparent_encoder.safetensors',
    local_path='path/to/icml24/LayerDiffuse_DiffusersCLI/models/ld_diffusers_sdxl_vae_transparent_encoder.safetensors'
  )
  # load transparent encoder
  transparent_encoder = TransparentVAEEncoder(path_ld_diffusers_sdxl_vae_transparent_encoder,
                                              device=accelerator.device, dtype=weight_dtype)

  orig_embeds_params_1 = accelerator.unwrap_model(text_encoder_1).get_input_embeddings().weight.data.clone().to(accelerator.device)
  # orig_embeds_params_1 = orig_embeds_params_1.to(accelerator.device, dtype=weight_dtype)
  orig_embeds_params_2 = accelerator.unwrap_model(text_encoder_2).get_input_embeddings().weight.data.clone().to(accelerator.device)
  # orig_embeds_params_2 = orig_embeds_params_2.to(accelerator.device, dtype=weight_dtype)
  
  placeholder_token = " ".join(tokenizer_1.convert_ids_to_tokens(placeholder_token_ids_1))
  print(f"placeholder_token = {placeholder_token}")
  
  if args.enable_xformers_memory_efficient_attention:
    if is_xformers_available():
      import xformers

      xformers_version = version.parse(xformers.__version__)
      if xformers_version == version.parse("0.0.16"):
        logger.warning(
          "xFormers 0.0.16 cannot be used for training in some GPUs. If you observe problems during training, please update xFormers to at least 0.0.17. See https://huggingface.co/docs/diffusers/main/en/optimization/xformers for more details."
        )
      unet.enable_xformers_memory_efficient_attention()
    else:
      raise ValueError("xformers is not available. Make sure it is installed correctly")
  # Enable TF32 for faster training on Ampere GPUs,
  # cf https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices
  if args.allow_tf32:
    torch.backends.cuda.matmul.allow_tf32 = True
  if args.scale_lr:
    args.learning_rate = (
      args.learning_rate * args.gradient_accumulation_steps * args.train_batch_size * accelerator.num_processes
    )
  # Initialize the optimizer
  if args.use_8bit_adam:
    try:
      import bitsandbytes as bnb
    except ImportError:
      raise ImportError(
        "To use 8-bit Adam, please install the bitsandbytes library: `pip install bitsandbytes`."
      )

    optimizer_class = bnb.optim.AdamW8bit
  else:
    optimizer_class = torch.optim.AdamW
  optimizer = optimizer_class(
    list(text_encoder_1.get_input_embeddings().parameters()) + list(
      text_encoder_2.get_input_embeddings().parameters()),  # only optimize the embeddings
    lr=args.learning_rate,
    betas=(args.adam_beta1, args.adam_beta2),
    weight_decay=args.adam_weight_decay,
    eps=args.adam_epsilon,
  )

  # Scheduler and math around the number of training steps.
  
  lr_scheduler = get_scheduler(
    args.lr_scheduler,
    optimizer=optimizer,
    num_warmup_steps=args.lr_warmup_steps * accelerator.num_processes,
    num_training_steps=args.max_train_steps * accelerator.num_processes,
    num_cycles=args.lr_num_cycles,
  )
  text_encoder_1.train()
  text_encoder_2.train()
  # Prepare everything with our `accelerator`.
  text_encoder_1, text_encoder_2, optimizer, lr_scheduler = accelerator.prepare(
    text_encoder_1, text_encoder_2, optimizer, lr_scheduler
  )
  
  scheduler = Sigma_Scheduler()
  
  def adaptive_init_timestep(scheduler, num_inference_steps, num_train, num_total, max_strength, max_train_steps):
    """
    Calculate the initial timestep and adaptive training steps for a scheduler based on training progress.
    
    Parameters:
    - scheduler: The scheduler object that has methods `get_sigmas_karras` and `timestep`.
    - num_inference_steps: The number of inference steps to be performed.
    - num_train: The number of training steps completed.
    - num_total: The total number of training steps.
    - max_strength: The maximum strength parameter that influences the number of sigmas.
    - max_train_steps: The maximum number of training steps.
    
    Returns:
    - init_timestep: The initial timestep calculated from the sigmas.
    - train_steps: The adjusted number of training steps based on the initial timestep.
    """
    
    # Get the sigmas for the maximum strength
    sigmas = scheduler.get_sigmas_karras(int(num_inference_steps / max_strength))  # Length = n/s
    sigmas = sigmas[-(num_inference_steps + 1):]  # Length n+1
    max_init_timestep = scheduler.timestep(sigmas[0]).item()
    
    # Calculate the ratio of training steps completed to the total training steps
    ratio = num_train / num_total
    
    # Determine the strength based on the ratio
    # if ratio <= 1/3:
    #     strength = 0
    #     return 0, 0
    # elif ratio <= 1/2:
    #     strength = max_strength / 2
    # else: 
    strength = max_strength
    
    # Get the sigmas using the calculated strength
    sigmas = scheduler.get_sigmas_karras(int(num_inference_steps / strength))  # Length = n/s
    sigmas = sigmas[-(num_inference_steps + 1):]  # Length n+1
    
    # Calculate the initial timestep from the first sigma
    init_timestep = scheduler.timestep(sigmas[0]).item()
    
    # Print the calculated initial timestep and strength for debugging purposes
    print(f'ratio={num_train}/{num_total} strength = {strength} / {max_strength}: init_timestep: {init_timestep}')
    
    # Calculate the adjusted number of training steps
    train_steps = (init_timestep / max_init_timestep) * max_train_steps
    
    return init_timestep, max_train_steps

  # Assuming args is a parsed argument object that includes a reverse attribute
  if args.reverse:
    path_list = path_list[::-1]

  for path in path_list:
    cluster_name = os.path.basename(path)
    class_name = os.path.basename(os.path.dirname(path))
    output_dir = os.path.join(args.output_dir, class_name, cluster_name)
    embed1_save_path = os.path.join(output_dir, 'learned_embeds_1.safetensors')
    embed2_save_path = os.path.join(output_dir, 'learned_embeds_2.safetensors')
    
    # Check if the embedding files already exist
    if os.path.exists(embed1_save_path) and os.path.exists(embed2_save_path):
        continue
    
    os.makedirs(output_dir, exist_ok=True)
    print(f"Start Inverse {class_name} {cluster_name} to {embed1_save_path}")
    
    # Create the training dataset
    train_dataset = TextualInversionDataset(
        data_root=path,
        tokenizer_1=tokenizer_1,
        tokenizer_2=tokenizer_2,
        size=args.resolution,
        placeholder_token=placeholder_token,
        repeats=args.repeats,
        learnable_property=args.learnable_property,
        center_crop=args.center_crop,
        set="train",
    )
    
    train_dataloader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.train_batch_size, shuffle=True, num_workers=args.dataloader_num_workers
    )
    train_dataloader = accelerator.prepare(train_dataloader)
    
    # Calculate the initial timestep
    init_timestep, train_steps = adaptive_init_timestep(
        scheduler=scheduler, 
        num_inference_steps=25, 
        num_train=train_dataset.num_images, 
        num_total=30, 
        max_strength=args.strength,
        max_train_steps = args.max_train_steps
    )
    if train_steps ==0:
      continue
    
    # Start the training process
    training(
        output_dir, init_timestep, accelerator, args, 
        placeholder_token_ids_1, placeholder_token_ids_2, text_encoder_1,
        text_encoder_2, tokenizer_1, tokenizer_2, unet, vae, 
        train_dataloader, transparent_encoder, scheduler, weight_dtype, 
        orig_embeds_params_1, orig_embeds_params_2, optimizer, lr_scheduler, train_steps
    )

  accelerator.end_training()

def training(output_dir, init_timestep, accelerator, args, placeholder_token_ids_1,
             placeholder_token_ids_2, text_encoder_1, text_encoder_2,
             tokenizer_1, tokenizer_2, unet, vae, train_dataloader, transparent_encoder,scheduler,weight_dtype,orig_embeds_params_1,orig_embeds_params_2,optimizer,lr_scheduler, train_steps):

  # We need to recalculate our total training steps as the size of the training dataloader may have changed.
  num_update_steps_per_epoch = math.ceil(len(train_dataloader) / args.gradient_accumulation_steps)
  overrode_max_train_steps = False
  # Afterwards we recalculate our number of training epochs
  args.num_train_epochs = math.ceil(train_steps / num_update_steps_per_epoch)
  # We need to initialize the trackers we use, and also store our configuration.
  # The trackers initializes automatically on the main process.
  # if accelerator.is_main_process:
  #     accelerator.init_trackers("textual_inversion", config=vars(args))
  # Train!
  total_batch_size = args.train_batch_size * accelerator.num_processes * args.gradient_accumulation_steps
  logger.info("***** Running training *****")
  logger.info(f"  Num Epochs = {args.num_train_epochs}")
  logger.info(f"  Instantaneous batch size per device = {args.train_batch_size}")
  logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_batch_size}")
  logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
  logger.info(f"  Total optimization steps = {train_steps}")
  global_step = 0
  first_epoch = 0

  initial_global_step = 0
  progress_bar = tqdm(
    range(0, train_steps),
    initial=initial_global_step,
    desc="Steps",
    # Only show the progress bar once on each machine.
    disable=not accelerator.is_local_main_process,
  )

  for epoch in range(first_epoch, args.num_train_epochs):
    text_encoder_1.train()
    text_encoder_2.train()
    for step, batch in enumerate(train_dataloader):
      with accelerator.accumulate(text_encoder_1, text_encoder_2):
        # Convert images to latent space
        latents = transparent_encoder(vae, batch["pixel_values"].cpu().numpy())
        latents = latents.detach()
        latents = latents * vae.config.scaling_factor
        # Sample noise that we'll add to the latents
        noise = torch.randn_like(latents)
        bsz = latents.shape[0]

        # Sample a random timestep for each image
        timesteps = torch.randint(0, init_timestep, (bsz,), device=latents.device)
        timesteps = timesteps.long()
        sigma = scheduler.sigmas[timesteps.to(scheduler.sigmas.device)].to(noise)
        sigma = sigma.view(bsz, 1, 1, 1)
        noisy_latents = latents + noise * sigma
        noisy_latents = noisy_latents / (sigma ** 2 + 1.0) ** 0.5

        # Get the text embedding for conditioning
        encoder_hidden_states_1 = (
          text_encoder_1(batch["input_ids_1"], output_hidden_states=True)
          .hidden_states[-2]
          .to(dtype=weight_dtype)
        )

        encoder_output_2 = text_encoder_2(
          batch["input_ids_2"].reshape(batch["input_ids_1"].shape[0], -1), output_hidden_states=True
        )
        encoder_hidden_states_2 = encoder_output_2.hidden_states[-2].to(dtype=weight_dtype)

        original_size = [
          (batch["original_size"][0][i].item(), batch["original_size"][1][i].item())
          for i in range(args.train_batch_size)
        ]
        crop_top_left = [
          (batch["crop_top_left"][0][i].item(), batch["crop_top_left"][1][i].item())
          for i in range(args.train_batch_size)
        ]
        target_size = (args.resolution, args.resolution)
        add_time_ids = torch.cat(
          [
            torch.tensor(original_size[i] + crop_top_left[i] + target_size)
            for i in range(args.train_batch_size)
          ]
        ).to(accelerator.device, dtype=weight_dtype)
        added_cond_kwargs = {"text_embeds": encoder_output_2[0], "time_ids": add_time_ids}
        encoder_hidden_states = torch.cat([encoder_hidden_states_1, encoder_hidden_states_2], dim=-1)

        # Predict the noise residual
        model_pred = unet(
          noisy_latents, timesteps, encoder_hidden_states, added_cond_kwargs=added_cond_kwargs
        ).sample

        loss = F.mse_loss(model_pred.float(), noise.float(), reduction="mean")

        accelerator.backward(loss)

        optimizer.step()
        lr_scheduler.step()
        optimizer.zero_grad()

        # Let's make sure we don't update any embedding weights besides the newly added token
        index_no_updates_encoder1 = torch.ones((len(tokenizer_1),), dtype=torch.bool)
        index_no_updates_encoder1[min(placeholder_token_ids_1): max(placeholder_token_ids_1) + 1] = False

        index_no_updates_encoder2 = torch.ones((len(tokenizer_2),), dtype=torch.bool)
        index_no_updates_encoder2[min(placeholder_token_ids_2): max(placeholder_token_ids_2) + 1] = False

        with torch.no_grad():
          accelerator.unwrap_model(text_encoder_1).get_input_embeddings().weight[
            index_no_updates_encoder1
          ] = orig_embeds_params_1[index_no_updates_encoder1]

          accelerator.unwrap_model(text_encoder_2).get_input_embeddings().weight[
            index_no_updates_encoder2
          ] = orig_embeds_params_2[index_no_updates_encoder2]

      # Checks if the accelerator has performed an optimization step behind the scenes
      if accelerator.sync_gradients:
        images = []
        progress_bar.update(1)
        global_step += 1
        if global_step % args.save_steps == 0:
          weight_name = f"learned_embeds-steps-{global_step}.safetensors"
          save_path = os.path.join(output_dir, weight_name)
          save_progress(
            text_encoder_1,
            placeholder_token_ids_1,
            text_encoder_2,
            placeholder_token_ids_2,
            accelerator,
            args,
            save_path,
            safe_serialization=True,
          )

      logs = {"loss": loss.detach().item(), "lr": lr_scheduler.get_last_lr()[0]}
      progress_bar.set_postfix(**logs)
      accelerator.log(logs, step=global_step)

      if global_step >= train_steps:
        break

  index_no_updates = torch.ones((len(tokenizer_1),), dtype=torch.bool)
  index_no_updates[min(placeholder_token_ids_1): max(placeholder_token_ids_1) + 1] = False

  index_no_updates = torch.ones((len(tokenizer_2),), dtype=torch.bool)
  index_no_updates[min(placeholder_token_ids_2): max(placeholder_token_ids_2) + 1] = False

  with torch.no_grad():
  # Using XOR operator instead of subtraction for boolean tensors
    index_updates_encoder1 = torch.ones((len(tokenizer_1),), dtype=torch.bool) ^ index_no_updates_encoder1
    accelerator.unwrap_model(text_encoder_1).get_input_embeddings().weight[
      index_updates_encoder1
    ] = orig_embeds_params_1[index_updates_encoder1]

    index_updates_encoder2 = torch.ones((len(tokenizer_2),), dtype=torch.bool) ^ index_no_updates_encoder2
    accelerator.unwrap_model(text_encoder_2).get_input_embeddings().weight[
      index_updates_encoder2
    ] = orig_embeds_params_2[index_updates_encoder2]
      
  # Create the pipeline using the trained modules and save it.
  accelerator.wait_for_everyone()
  if accelerator.is_main_process:
    weight_name = "learned_embeds.safetensors"
    save_path = os.path.join(output_dir, weight_name)
    save_progress(
      text_encoder_1,
      placeholder_token_ids_1,
      text_encoder_2,
      placeholder_token_ids_2,
      accelerator,
      args,
      save_path,
      safe_serialization=True,
    ) 
 
if __name__ == "__main__":
  main()
