import os
from datasets import load_dataset
import torch
import json
from transformers import AutoTokenizer, LlamaTokenizer, LlamaForCausalLM, AutoModelForCausalLM
from tqdm import tqdm
import numpy as np
import random
import argparse
import time
import torch.distributed as dist
import torch.multiprocessing as mp
from models.modeling_llama_linear import LlamaAttention_SLAKE, LlamaModel_SLAKE_forward
from models.modeling_qwen_linear import Qwen2_SLAKE_forward, Qwen2Attention_SLAKE
from models.modeling_mistral_linear import Mistral_SLAKE_forward, MistralAttention_SLAKE
from transformers.models.llama.modeling_llama import LlamaAttention, LlamaModel, LlamaDecoderLayer



def parse_args(args=None):
    parser = argparse.ArgumentParser()
    parser.add_argument('--model', type=str, default=None, choices=["llama2-7b-chat-4k", "llama2-13b-chat-4k", "llama-3.1-8B-instruct", "qwen2.5-7b-instruct", "mistral-0.3-7b-32k"])
    parser.add_argument('--e', action='store_true', help="Evaluate on LongBench-E")
    parser.add_argument('--cache_type', type=str, default="", choices=["SLAKE", "FullKV"])
    parser.add_argument("--cache_size", type=int, default=128, help="")
    parser.add_argument("--max_capacity_prompts_ratio", type=float, default=-1, help="")
    parser.add_argument("--window_sizes", type=int, default=96, help="")
    parser.add_argument("--recent_size", type=int, default=32, help="")
    parser.add_argument("--linear_cache_size", type=int, default=0, help="")
    parser.add_argument("--alpha", type=float, default=0.6, help="alpha param of score")
    parser.add_argument("--beta", type=float, default=0.5, help="beta param of score")
    parser.add_argument("--use_fast_tokenizer", type=bool, default=True, help="")
    parser.add_argument("--use_cache", type=bool, default=True, help="")
    parser.add_argument("--data_parallel", type=bool, default=False, help="")
    parser.add_argument("--attn_implementation", type=str, default="flash_attention_2",
                        choices=["flash_attention_2", "sdpa", "eager"])
    parser.add_argument("--pruning_ratio", type=float, default=0.4, help="pruning ratio of Key Cache")
    parser.add_argument("--steps", type=int, default=-1, help="maximum number of examples to evaluate per task.")
    parser.add_argument("--merge", type=str, default=None, help="kv merge method(look-m)")
    parser.add_argument('--floor', type=float, default=0.2, help='hyper-parameter used in AdaKV')
    parser.add_argument('--head_path', type=str,
                        default='./data/heads_score/Meta-Llama-3-8B-Instruct_retrieval_reasoning_heads.json',
                        help='Path to head score (HeadKV)')
    parser.add_argument('--head_beta', type=float, default=1.01, help='hyper-parameter used on HeadKV')
    parser.add_argument("--addr", type=str, default='12499', help="process address")


    parser.add_argument(
        "--use_chat_format",
        action="store_true",
        help="If given, we will use the chat format for the prompts."
    )
    parser.add_argument(
        "--chat_formatting_function",
        type=str,
        default="eval.templates.create_prompt_with_tulu_chat_format",
        help="The function to use to create the chat format. This function will be dynamically imported. Please see examples in `eval/templates.py`."
    )
    return parser.parse_args(args)

# This is the customized building prompt for chat models
def build_chat(tokenizer, prompt, model_name):

    if "llama3" in model_name:
        #print("======== llama3 build chat ========")
        prompt = f"<|begin_of_text|><|start_header_id|>user<|end_header_id|>\n\n{prompt}<|eot_id|><|start_header_id|>assistant<|end_header_id|>\n\n"
    elif "llama2" in model_name:
        prompt = f"[INST]{prompt}[/INST]"
    elif "mistral" in model_name:
        #print("======== mistral build chat ========")
        prompt = f'<s>[INST] {prompt} [/INST]'
    elif "qwen" in model_name:
        #print("======== qwen build chat ========")
        messages = [
            {"role": "system", "content": "You are Qwen, created by Alibaba Cloud. You are a helpful assistant."},
            {"role": "user", "content": prompt}
        ]
        prompt = tokenizer.apply_chat_template(
            messages,
            tokenize=False,
            add_generation_prompt=True
        )

    return prompt

def post_process(response, model_name):
    if "xgen" in model_name:
        response = response.strip().replace("Assistant:", "")
    elif "internlm" in model_name:
        response = response.split("<eoa>")[0]
    return response

def get_pred(rank, world_size, data, max_length, max_gen, prompt_format, dataset, device, model_name, model2path, out_path, args):
    #if world_size > 1:

    os.environ['MASTER_ADDR'] = os.environ.get('MASTER_ADDR', 'localhost')
    os.environ['MASTER_PORT'] = os.environ.get('MASTER_PORT', args.addr)
    dist.init_process_group(backend="nccl", rank=rank, world_size=world_size)
    device = torch.device(f'cuda:{rank}')
    if args.data_parallel:
        device_map ={"": rank}
    else:
        device_map = "auto"

    if "mamba" in model_name:
        tokenizer = AutoTokenizer.from_pretrained("tiiuae/falcon-mamba-7b")
        model = AutoModelForCausalLM.from_pretrained("tiiuae/falcon-mamba-7b", device_map=device_map,)
    elif "aug-linear" in model_name:
        tokenizer = AutoTokenizer.from_pretrained("LinearizedLLM/llama-2-7b-aug-linear")
        model = AutoModelForCausalLM.from_pretrained("LinearizedLLM/llama-2-7b-aug-linear", device_map=device_map,)
    else:
        model, tokenizer = load_model_and_tokenizer(model2path[model_name], model_name, device, args, rank)
    for json_obj in tqdm(data):
        prompt = prompt_format.format(**json_obj)
        # truncate to fit max_length (we suggest truncate in the middle, since the left and right side may contain crucial instructions)
        tokenized_prompt = tokenizer(prompt, truncation=False, return_tensors="pt").input_ids[0]
        if "chatglm3" in model_name:
            tokenized_prompt = tokenizer(prompt, truncation=False, return_tensors="pt", add_special_tokens=False).input_ids[0]
        if len(tokenized_prompt) > max_length:
            half = int(max_length/2)
            prompt = tokenizer.decode(tokenized_prompt[:half], skip_special_tokens=True)+tokenizer.decode(tokenized_prompt[-half:], skip_special_tokens=True)
        if dataset not in ["trec", "triviaqa", "samsum", "lsht", "lcc", "repobench-p"]: # chat models are better off without build prompts on these tasks
            prompt = build_chat(tokenizer, prompt, model_name)
        if "chatglm3" in model_name:
            if dataset in ["trec", "triviaqa", "samsum", "lsht", "lcc", "repobench-p"]:
                input = tokenizer(prompt, truncation=False, return_tensors="pt").to(device)
            else:
                input = prompt.to(device)
        else:
            input = tokenizer(prompt, truncation=False, return_tensors="pt").to(device)
            # attention_mask = input["attention_mask"].to(device)

        context_length = input.input_ids.shape[-1]
        if dataset == "samsum": # prevent illegal output on samsum (model endlessly repeat "\nDialogue"), might be a prompting issue
            output = model.generate(
                **input,
                max_new_tokens=max_gen,
                num_beams=1,
                do_sample=False,
                temperature=1.0,
                min_length=context_length+1,
                eos_token_id=[tokenizer.eos_token_id, tokenizer.encode("\n", add_special_tokens=False)[-1]],
                top_p=None,
                top_k=None,
            )[0]
        else:
            output = model.generate(
                **input,
                max_new_tokens=max_gen,
                num_beams=1,
                do_sample=False,
                temperature=1.0,
                top_p=None,
                top_k=None,
                # attention_mask=attention_mask,
            )[0]
        pred = tokenizer.decode(output[context_length:], skip_special_tokens=True)
        pred = post_process(pred, model_name)
        with open(out_path, "a", encoding="utf-8") as f:
            json.dump({"pred": pred, "answers": json_obj["answers"], "all_classes": json_obj["all_classes"], "length": json_obj["length"]}, f, ensure_ascii=False)
            f.write('\n')
    #dist.destroy_process_group()
    if dist.is_initialized():
        dist.destroy_process_group()

def seed_everything(seed):
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    np.random.seed(seed)
    random.seed(seed)
    torch.backends.cudnn.benchmark = False
    torch.backends.cudnn.deterministic = True
    torch.cuda.manual_seed_all(seed)

def load_model_and_tokenizer(path, model_name, device, args, rank):
    cache_type = args.cache_type
    from pyramidkv.monkeypatch import replace_llama, replace_mistral
    replace_llama(cache_type.lower())
    replace_mistral(cache_type.lower())
    if args.data_parallel:
        device_map ={"": rank}
    else:
        device_map = "auto"

    tokenizer = AutoTokenizer.from_pretrained(
        path,
        use_fast=args.use_fast_tokenizer,
        padding_side="left",
    )

    model = AutoModelForCausalLM.from_pretrained(
        path,
        torch_dtype=torch.bfloat16,
        low_cpu_mem_usage=True,
        device_map=device_map,
        use_cache=args.use_cache,
        attn_implementation=args.attn_implementation,
    )

    # --- 추가 ---
    if tokenizer.pad_token_id is None:
        tokenizer.pad_token = tokenizer.eos_token
        tokenizer.pad_token_id = tokenizer.eos_token_id

    # 모델 쪽에도 동일하게 반영
    model.config.pad_token_id = tokenizer.pad_token_id
    model.generation_config.pad_token_id = tokenizer.pad_token_id
    # (선택) eos도 명시
    model.config.eos_token_id = tokenizer.eos_token_id
    model.generation_config.eos_token_id = tokenizer.eos_token_id

    tokenizer.padding_side = "left"


    #print(model)

    cache_size = args.cache_size
    if args.cache_type == "FullKV":
        print("using Full KV")
    elif args.cache_type =="SLAKE":
        print("Using SLAKE cache")
        if "llama" in model_name:
            target_dtype = next(model.model.parameters()).dtype
            target_device = next(model.model.parameters()).device
            model.model.forward = LlamaModel_SLAKE_forward.__get__(model.model, LlamaModel)
            #print("LA attention")
            # replace llama attention
            for i, layer in enumerate(model.model.layers):
                ref_attn = layer.self_attn
                custom_attn = LlamaAttention_SLAKE(model.config, i, args.cache_size, args.window_sizes, args.linear_cache_size, args.alpha, args.beta)
                custom_attn.load_state_dict(ref_attn.state_dict(), strict=False)
                custom_attn = custom_attn.to(dtype=target_dtype, device=target_device)
                layer.self_attn = custom_attn
        elif "qwen" in model_name:
            target_dtype = next(model.model.parameters()).dtype
            target_device = next(model.model.parameters()).device

            model.model.forward = Qwen2_SLAKE_forward.__get__(model.model, LlamaModel)
            #print("LA attention")
            # replace llama attention
            for i, layer in enumerate(model.model.layers):
                ref_attn = layer.self_attn
                custom_attn = Qwen2Attention_SLAKE(model.config, i, args.cache_size, args.window_sizes, args.linear_cache_size, args.alpha, args.beta)
                custom_attn.load_state_dict(ref_attn.state_dict(), strict=False)
                custom_attn = custom_attn.to(dtype=target_dtype, device=target_device)
                layer.self_attn = custom_attn
        elif "mistral" in model_name:
            print("using LAKE based mistral")
            target_dtype = next(model.model.parameters()).dtype
            target_device = next(model.model.parameters()).device
            model.model.forward = Mistral_SLAKE_forward.__get__(model.model, LlamaModel)
            #print("LA attention")
            # replace llama attention
            for i, layer in enumerate(model.model.layers):
                ref_attn = layer.self_attn
                custom_attn = MistralAttention_SLAKE(model.config, i, args.cache_size, args.window_sizes, args.linear_cache_size, args.alpha, args.beta)
                custom_attn.load_state_dict(ref_attn.state_dict(), strict=False)
                custom_attn = custom_attn.to(dtype=target_dtype, device=target_device)
                layer.self_attn = custom_attn

    model.eval()
    return model, tokenizer

if __name__ == '__main__':
    seed_everything(42)
    torch.cuda.synchronize()
    start_time = time.time()

    args = parse_args()
    data_parallel = args.data_parallel
    if data_parallel:
        print("using data_parallel")
        world_size = torch.cuda.device_count()
    else:
        print("data_parallel disabled")
        world_size = 1
    mp.set_start_method('spawn', force=True)
    cache_type = args.cache_type
    window_size = args.window_sizes
    cache_size = args.cache_size
    linear_cache_size = args.linear_cache_size
    model2path = json.load(open("config/model2path.json", "r"))
    model2maxlen = json.load(open("config/model2maxlen.json", "r"))
    device = torch.device('cuda' if torch.cuda.is_available() else 'cpu')
    model_name = args.model
    # define your model
    max_length = model2maxlen[model_name]
    if args.e:
        datasets = ["qasper", "multifieldqa_en", "hotpotqa", "2wikimqa", "gov_report", "multi_news", \
            "trec", "triviaqa", "samsum", "passage_count", "passage_retrieval_en", "lcc", "repobench-p"]
    else:

        datasets = ["narrativeqa", "qasper", "multifieldqa_en", "hotpotqa", "2wikimqa", "musique", \
                    "gov_report", "qmsum", "multi_news",  "trec", "triviaqa", "samsum", \
                    "passage_count", "passage_retrieval_en", "lcc", "repobench-p"]


    # we design specific prompt format and max generation length for each task, feel free to modify them to optimize model output
    dataset2prompt = json.load(open("config/dataset2prompt.json", "r"))
    dataset2maxlen = json.load(open("config/dataset2maxlen.json", "r"))
    # predict on each dataset
    if not os.path.exists("pred"):
        os.makedirs("pred")
    if not os.path.exists("pred_e"):
        os.makedirs("pred_e")
    for dataset in datasets:
        if args.e:
            data = load_dataset('THUDM/LongBench', f"{dataset}_e", split='test')
            if not os.path.exists(f"pred_e/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}"):
                os.makedirs(f"pred_e/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}")
            out_path = f"pred_e/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}/{dataset}.jsonl"
        else:
            data = load_dataset('THUDM/LongBench', dataset, split='test')
            if not os.path.exists(f"pred/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}"):
                os.makedirs(f"pred/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}")
            out_path = f"pred/{model_name}_{cache_type}_{cache_size}_{window_size}_{linear_cache_size}_{args.alpha}_{args.beta}/{dataset}.jsonl"
        prompt_format = dataset2prompt[dataset]
        max_gen = dataset2maxlen[dataset]
        data_all = [data_sample for data_sample in data]

        # start = 95
        # data_all = data_all[start:]



        data_subsets = [data_all[i::world_size] for i in range(world_size)]
        processes = []
        for rank in range(world_size):
            p = mp.Process(target=get_pred, args=(rank, world_size, data_subsets[rank], max_length, \
                        max_gen, prompt_format, dataset, device, model_name, model2path, out_path, args))
            p.start()
            processes.append(p)
        for p in processes:
            p.join()
    torch.cuda.synchronize()
    end_time = time.time()
    elapsed_sec = end_time - start_time
    m, s = divmod(elapsed_sec, 60)
    h, m = divmod(m, 60)
    print(f"total evaluation time: {int(h):02d}:{int(m):02d} (hh:mm)")