import random

def split_indices(num_indices=1000, train_ratio=0.8, val_ratio=0.1, seed=42):
    """
    Splits a range of indices into training, validation, and test sets.

    Args:
        num_indices (int): The total number of indices to split (e.g., 1000 for 0-999).
        train_ratio (float): The proportion of indices for the training set.
        val_ratio (float): The proportion of indices for the validation set.
        seed (int): A random seed for reproducibility.

    Returns:
        tuple: A tuple containing three lists of indices:
               (train_indices, val_indices, test_indices).
    """
    # The test ratio is inferred from the other two
    test_ratio = 1.0 - train_ratio - val_ratio
    if not (train_ratio + val_ratio + test_ratio) == 1.0:
        raise ValueError("Ratios must sum to 1.0")

    # Create a list of indices [0, 1, ..., 999]
    indices = list(range(num_indices))
    
    # Set the seed for random shuffling to make the split reproducible
    random.seed(seed)
    # Shuffle the indices randomly in-place
    random.shuffle(indices)
    
    # Calculate the split points
    train_split_point = int(train_ratio * num_indices)
    val_split_point = int((train_ratio + val_ratio) * num_indices)
    
    # Slice the shuffled list to create the three sets
    train_indices = indices[:train_split_point]
    val_indices = indices[train_split_point:val_split_point]
    test_indices = indices[val_split_point:]
    
    return train_indices, val_indices, test_indices

# --- Example Usage ---
if __name__ == "__main__":
    # Define the 8:1:1 ratio
    TRAIN_RATIO = 0.8
    VAL_RATIO = 0.1
    
    # Get the split indices
    train_set, val_set, test_set = split_indices(
        num_indices=1000,
        train_ratio=TRAIN_RATIO,
        val_ratio=VAL_RATIO,
        seed=42 # Using a seed ensures you get the same split every time
    )
    
    # --- Store the splits as files ---
    with open('train.txt', 'w') as f:
        for index in train_set:
            f.write(f'{index}\n')

    with open('val.txt', 'w') as f:
        for index in val_set:
            f.write(f'{index}\n')

    with open('test.txt', 'w') as f:
        for index in test_set:
            f.write(f'{index}\n')
            
    print("Successfully stored splits in train.txt, val.txt, and test.txt")