from __future__ import annotations

from dataclasses import dataclass
from typing import Callable, Literal, Optional


def curvature_schedule(c0: float, alpha: float, speed: float, beta: float, uncertainty: float) -> float:
    return c0 * (1.0 + alpha * speed + beta * uncertainty)


@dataclass
class DynamicCurvature:
    base: float = 1.0
    alpha: float = 0.0
    beta: float = 0.0
    clamp_min: float = 0.0
    clamp_max: Optional[float] = None

    def __call__(self, speed: float, uncertainty: float) -> float:
        c = curvature_schedule(self.base, self.alpha, speed, self.beta, uncertainty)
        if self.clamp_max is not None:
            c = min(c, float(self.clamp_max))
        if self.clamp_min is not None:
            c = max(c, float(self.clamp_min))
        return c


def hysteresis_schedule(prev: float, low: float, high: float, speed: float, gain: float = 1.0) -> float:
    if speed > high:
        return prev * (1.0 + gain)
    if speed < low:
        return prev * (1.0 - gain)
    return prev


def smooth_update(prev: float, target: float, tau: float) -> float:
    return (1 - tau) * prev + tau * target


def bounded_update(prev: float, delta: float, lo: float, hi: float) -> float:
    return float(min(max(prev + delta, lo), hi))


def decay(value: float, factor: float) -> float:
    return float(value * factor)


def _demo():
    dc = DynamicCurvature(base=1.0, alpha=2.0, beta=1.0, clamp_min=0.5, clamp_max=3.0)
    print(dc(0.2, 0.1))


if __name__ == "__main__":
    _demo()
                
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
