from __future__ import annotations

from dataclasses import dataclass
from typing import Dict, Sequence, Tuple
import numpy as np


def summarize_returns_costs(returns: Sequence[float], costs: Sequence[float]) -> Dict[str, float]:
    return {
        "sum/return_mean": float(np.mean(returns) if returns else 0.0),
        "sum/return_std": float(np.std(returns) if returns else 0.0),
        "sum/cost_mean": float(np.mean(costs) if costs else 0.0),
    }


def risk_calibration(kappas: Sequence[float], failures: Sequence[int], bins: int = 10) -> Dict[str, float]:
    if len(kappas) == 0:
        return {"cal/auc": 0.0}
    kap = np.array(kappas)
    fail = np.array(failures)
    qs = np.quantile(kap, np.linspace(0, 1, bins + 1))
    auc = 0.0
    for i in range(bins):
        mask = (kap >= qs[i]) & (kap < qs[i + 1])
        if mask.sum() == 0:
            continue
        auc += fail[mask].mean()
    return {"cal/auc": float(auc / bins)}


def episode_stats(returns: Sequence[float], lengths: Sequence[int], costs: Sequence[float]) -> Dict[str, float]:
    return {
        "ep/return_mean": float(np.mean(returns) if returns else 0.0),
        "ep/length_mean": float(np.mean(lengths) if lengths else 0.0),
        "ep/cost_mean": float(np.mean(costs) if costs else 0.0),
    }


def success_rate(success: Sequence[int]) -> Dict[str, float]:
    if not success:
        return {"success/rate": 0.0}
    return {"success/rate": float(np.mean(success))}


def _demo():
    print(summarize_returns_costs([1.0, 2.0], [0.0, 1.0]))


if __name__ == "__main__":
    _demo()
                
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
