from __future__ import annotations

from dataclasses import dataclass
from typing import Optional, Tuple
import numpy as np


@dataclass
class EMA:
    beta: float = 0.9
    value: float = 0.0
    init: bool = False

    def update(self, x: float) -> float:
        if not self.init:
            self.value = x
            self.init = True
        else:
            self.value = (1 - self.beta) * x + self.beta * self.value
        return self.value


class BoundaryVelocityEstimator:
    def __init__(self, ema_beta: float = 0.9):
        self.prev_center: Optional[np.ndarray] = None
        self.speed_ema = EMA(beta=ema_beta)
        self.unc_ema = EMA(beta=ema_beta)

    def update(self, center: np.ndarray) -> Tuple[float, float]:
        if self.prev_center is None:
            self.prev_center = center.astype(np.float32)
            return 0.0, 0.0
        v = float(np.linalg.norm(center - self.prev_center))
        self.prev_center = center.astype(np.float32)
        v_s = float(self.speed_ema.update(v))
        unc = float(self.unc_ema.update(min(1.0, v)))
        return v_s, unc


def _demo():
    est = BoundaryVelocityEstimator(ema_beta=0.8)
    c = np.array([0.0, 0.0], dtype=np.float32)
    for t in range(5):
        c = c + np.array([0.1, -0.05], dtype=np.float32)
        print(est.update(c))


if __name__ == "__main__":
    _demo()
                
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
     
