from __future__ import annotations

import argparse
import csv
import glob
import os
import numpy as np


def parse_metrics(path: str):
    rets, costs = [], []
    with open(path) as f:
        r = csv.reader(f)
        for row in r:
            if not row:
                continue
            kv = dict(zip(row[1::2], row[2::2]))
            if "ep/return" in kv:
                try:
                    rets.append(float(kv["ep/return"]))
                    costs.append(float(kv.get("ep/cost", 0.0)))
                except Exception:
                    pass
    return rets, costs


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--runs", type=str, default="runs/gdc")
    parser.add_argument("--out", type=str, default="summary.csv")
    args = parser.parse_args()

    rows = [("agent", "run", "return_mean", "cost_mean", "n_eps")]
    for run in glob.glob(os.path.join(args.runs, "*")):
        m = os.path.join(run, "metrics.csv")
        if os.path.isfile(m):
            R, C = parse_metrics(m)
            if R:
                rows.append((os.path.basename(run).split("_")[0], run, np.mean(R[-10:]), np.mean(C[-10:]), len(R)))

    with open(args.out, "w", newline="") as f:
        w = csv.writer(f)
        w.writerows(rows)
    print(f"Wrote {len(rows)-1} rows to {args.out}")


if __name__ == "__main__":
    main()

