import os
from typing import Optional, Dict, Any

try:
    from torch.utils.tensorboard import SummaryWriter                
except Exception:                    
    SummaryWriter = None                


class Logger:
    def __init__(self, log_dir: Optional[str] = None, csv_name: Optional[str] = "metrics.csv"):
        self.step = 0
        self.episode = 0
        self.tb: Optional[SummaryWriter] = None
        self.csv_fp = None
        if log_dir:
            os.makedirs(log_dir, exist_ok=True)
            if SummaryWriter is not None:
                self.tb = SummaryWriter(log_dir)
            else:
                self.tb = None
            if csv_name:
                self.csv_fp = open(os.path.join(log_dir, csv_name), "a")

    def log(self, scalars: Dict[str, Any], step: Optional[int] = None):
        if step is None:
            step = self.step
        if self.tb:
            for k, v in scalars.items():
                try:
                    self.tb.add_scalar(k, float(v), global_step=step)
                except Exception:
                                                   
                    pass
                                
        kv = " ".join([f"{k}={v}" for k, v in scalars.items()])
        print(f"[step {step}] {kv}")
             
        if self.csv_fp:
                                  
            row = [str(step)]
            for k, v in scalars.items():
                row.append(k)
                row.append(str(v))
            self.csv_fp.write(",".join(row) + "\n")
            self.csv_fp.flush()

    def increment_step(self, n: int = 1):
        self.step += n

    def increment_episode(self, n: int = 1):
        self.episode += n

    def close(self):
        if self.tb:
            self.tb.flush()
            self.tb.close()
        if self.csv_fp:
            self.csv_fp.flush()
            self.csv_fp.close()
