import json
from typing import List
from evaluation.dataset_parser import DatasetParser
import random
import os

random.seed(89)


class IconDQAParser(DatasetParser):

    def load_data(self, sample_file: str):
        self.data = []
        with open(self.dataset_file, 'r') as file:
            self.data = json.load(file)

        self.sample_file = sample_file
        self.sample_ids = self.sample_questions() if sample_file else None
        self.data_iterator = self.create_iterator(self.data)

    def create_iterator(self, data):
        for diagram_id, question_obj in data.items():
            if not self.sample_file or diagram_id in self.sample_ids:
                picked_q_type = question_obj["random_pick"]
                picked_question_obj = question_obj[picked_q_type]
                question = picked_question_obj["question"]
                answer_choices = picked_question_obj["choices"]
                correct_answer_text = picked_question_obj["answer"]
                image_name = f"{diagram_id}.png"
                image_path = os.path.join(
                    self.image_folder, image_name)
                yield {"diagram_id": diagram_id,
                       "question": question,
                       "answer_choices": answer_choices,
                       "correct_answer_text": correct_answer_text,
                       "image_path": image_path,
                       "question_type": picked_q_type}

    def next(self):
        # Return the next item from the iterator, or raise StopIteration if there are no more items
        return next(self.data_iterator)

    def sample_questions(self):
        samples = []
        with open(self.sample_file, 'r') as file:
            for line in file:
                samples.append(line.strip())
        return samples

    @staticmethod
    def format_qa(question: str, answer_choices: List[str], answer: str) -> str:
        question_with_answer_choices = "{}\n{}".format(question,
                                                       "\n".join("{}) {}".format(chr(65 + i), choice)
                                                                 for i, choice in enumerate(answer_choices)))

        answer_letter = chr(65 + answer_choices.index(answer))
        return question_with_answer_choices, answer_letter
