import torch
import torch.nn as nn
from SOCONV_Class import SOCONV
import math


def conv3x3(in_planes, out_planes,alpha, stride=1, dilation=1):
    """3x3 convolution with padding"""
    return SOCONV(in_planes, out_planes, Kernel_size=3, stride=stride,
                     padding=dilation,alpha=alpha,sigma=int(math.sqrt(out_planes)),decay_rate=-3)


def conv1x1(in_planes,out_planes,alpha, stride=1):
    """1x1 convolution"""
    return SOCONV(in_planes, out_planes, Kernel_size=1, stride=stride,alpha=alpha,sigma=int(math.sqrt(out_planes)),decay_rate=-3,padding=0)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes,alpha, stride=1, downsample=None,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")

        self.soconv1 = conv3x3(inplanes, planes,alpha,stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.soconv2 = conv3x3(planes, planes,alpha)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.soconv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.soconv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out    
    
class ResNet(nn.Module):

    def __init__(self, block, num_layers, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, alpha_high = 0.01,alpha_low = 0.008,replace_stride_with_dilation=None,
                 norm_layer=None):
        super(ResNet, self).__init__()
        assert num_layers in [18, 34, 50, 101, 152], f'ResNet{num_layers}: Unknown architecture! Number of layers has ' \
                                                     f'to be 18, 34, 50, 101, or 152 '
        super(ResNet, self).__init__()
        if num_layers < 50:
            self.expansion = 1
        else:
            self.expansion = 4
        if num_layers == 18:
            layers = [2, 2, 2, 2]
        elif num_layers == 34 or num_layers == 50:
            layers = [3, 4, 6, 3]
        elif num_layers == 101:
            layers = [3, 4, 23, 3]
        else:
            layers = [3, 8, 36, 3]
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.soconv1 = SOCONV(3, self.inplanes, Kernel_size=7, stride=2, padding=3,
                               alpha=alpha_low,sigma=8,decay_rate=-3)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0],alpha = alpha_low)
        self.layer2 = self._make_layer(block, 144, layers[1],alpha=alpha_low, stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2],alpha=alpha_low, stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 529, layers[3],alpha=alpha_high, stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(529 * block.expansion, num_classes)
    
        for m in self.modules():
            if isinstance(m, SOCONV):
                
                nn.init.kaiming_normal_(m.conv1.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
    
        

    def _make_layer(self, block, planes, blocks,alpha, stride=1, dilate=False):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion,alpha,stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes,alpha, stride, downsample,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes,alpha,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    def _forward_impl(self, x):
       
        x = self.soconv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward(self, x):
        return self._forward_impl(x)    


    def self_organize(self,current_iter, max_iter):
      for param in self.children():
        if hasattr(param,'self_organizing'):
          param.self_organizing(current_iter, max_iter)
                                

    def self_organize_old(self,current_iter, max_iter):
            """ calling self-organize """ 
            for param in ResNet.children(self):
                if hasattr(param,'self_organizing'):
                    param.self_organizing(current_iter, max_iter)
                elif hasattr(param,'0'):
                    for i in range (2):
                        for layer in param[i].children():
                            if hasattr(layer,'self_organizing'):
                                layer.self_organizing(current_iter,max_iter)
                            elif hasattr(layer,'0'):
                                if hasattr(layer[0],'self_organizing'):
                                    layer[0].self_organizing(current_iter,max_iter)