# main file for our agent
import os, sys, json, re, random, copy, signal
import multiprocessing
random.seed(98)
from abc import ABC, abstractmethod 
import numpy as np
import time
from agent.llm.llm_actions import *
from utils.logic.fol import *
from utils.logic.nli import NLI
from utils.logger import logger
from utils.logic.queue import PriorityQueue, Node
from utils.retriever import ST_Retriever, GPT3_Retriever
from utils.logic.prioritizer import GD_prioritizer, Pure_prioritizer
from utils.wikidata_types import *
from utils.logic.datalog import *

class Agent(ABC):
    def __init__(self):
        pass 
    @abstractmethod
    def __call__(self):
        pass 
    
    def _negate_preds(self, clause):
        if isinstance(clause, Predicate):
            clause_neg = Not(clause)
        elif isinstance(clause, Not):
            clause_neg = clause.arg
        return clause_neg

    

    @abstractmethod
    def _prove(self):
        pass 


class Typed_Resolution(Agent):
    def __init__(self, retriever, dataset_name):
        super().__init__()
        if retriever == 'gpt3':
            self.retriever = GPT3_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        elif retriever == 'st':
            self.retriever = ST_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        self.prioiritizer = GD_prioritizer()
        self.nli = NLI()
    
    def __call__(self, query, kb, llm_name, log, max_steps):
        answer = self._prove(query, kb,llm_name,  log=log, max_steps=max_steps)
        return str(answer)
    
    def _prove(self, query, kb, llm_name, log=None, max_steps=10):

        pos_query = query
        neg_query = self._negate_preds(pos_query)
        log("negated_query", neg_query)

        pos_proof, pos_scores = self._gd_resolution(pos_query, kb, max_steps)


        if pos_proof is None:
            log("pos_proof", "None"); log("pos_scores", "None")
            answer = False
        else:
            log("pos_proof", pos_proof); log("pos_scores", pos_scores)
            answer = True

        return answer
    
    def _gd_resolution(self, query, kb, max_steps):

            datalog = Datalog()
            q = PriorityQueue()
            root_priority = (-1, -1, 0)
            root = Node(None, None, query, root_priority, kb, {})
            q.push(root, root_priority)

            while not q.is_empty() and max_steps > 0:
                max_steps -= 1
                cur_node = q.pop()
                print(cur_node.goal)
                if cur_node.goal is None:
                    return self._construct_proof(cur_node), (cur_node.priority[0], cur_node.priority[1])
                new_nodes = self._expand_node(cur_node, datalog)

                for node in new_nodes:
                    q.push(node, node.priority)
            return None, 0
        
    def _expand_node(self, node, datalog):
        new_nodes = []
        node_goals = flatten_and(node.goal)
        theta = node.theta
        
        for goal in node_goals:
            other_goals = [g for g in node_goals if str(g) != str(goal)]
            top_k, original_clauses, theta_clauses = self._rank_clauses(goal, node.kb, node.theta)

            
            for i, (clause, score, child_types) in enumerate(top_k):  
                if isinstance(clause, Predicate) or isinstance(clause, Not):
                    if other_goals:
                        updated_goals = []
                        for goal in other_goals:
                            updated_goal = substitute(theta_clauses[i], goal)
                            updated_goals.append(updated_goal)
                        new_goal = and_list(updated_goals)
                    else:
                        new_goal = None 

                elif isinstance(clause, ForAll):
                    datalog_answer = False
                    if all(isinstance(arg, Constant) for arg in clause.body.rhs.args):
                        datalog_answer = datalog(original_clauses, clause.body.rhs)
                    if datalog_answer:
                        new_goal = None
                    else:
                        new_goal = clause.body.lhs
                        new_goal = self._apply_types(new_goal, child_types)
                        if other_goals:
                            new_goal = and_list([new_goal]+other_goals)

                new_node_priority = (node.priority[0]*score[0], node.priority[1]*score[1], node.priority[2]+1)
                new_node_kb = copy.deepcopy(node.kb)
                new_node_kb = self._remove_clause(original_clauses[i], new_node_kb)
                new_node = Node(node, clause, new_goal, new_node_priority, new_node_kb, theta_clauses[i])
                new_nodes.append(new_node)
        return new_nodes
    
    def _apply_types(self, goal, child_types):

        if isinstance(goal, Predicate):
            for arg in goal.args:
                if arg.name in child_types:
                    arg.type = child_types[arg.name]
        elif isinstance(goal, Not):
            for arg in goal.arg.args:
                if arg.name in child_types:
                    arg.type = child_types[arg.name]

        elif isinstance(goal, And):
            typed_goals = []
            for clause in flatten_and(goal):
                typed_goals.append(self._apply_types(clause, child_types))
            goal = and_list(typed_goals)
        elif isinstance(goal, Or):
            typed_goals = []
            for clause in flatten_or(goal):
                typed_goals.append(self._apply_types(clause, child_types))
            goal = or_list(typed_goals)
            
        return goal
        
                
    
    def _remove_clause(self, clause, kb):
        matching_clauses = [c for c in kb if str(c) == str(clause)]
        if matching_clauses:
            kb.remove(matching_clauses[0])
        return kb
    
    def _rank_clauses(self, goal, kb, theta, k=15):
        """
        rank the clauses in kb to backchain from node.goal
        return a list where each item is (clause, score)
        """
        # 1. semantic search of kb clause RHSs with the goal to get rid of red-herrings
        kb = self.retriever(goal, kb, k)
        # 2. top k clauses (exact matches, entailment scores)
        
        topk_clauses, original_clauses, theta_clauses = self.prioiritizer(kb, goal, theta, 2*k)
        # topk_clauses is a list of tuples: (clause, (type_score, pred_score), child_types)
        # original clauses is a list of the original clauses in kb

        return topk_clauses, original_clauses, theta_clauses
    def _construct_proof(self, node):
        """
        repeatedly traverse up parent_node to reconstruct proof
        """
        used_clauses = []
        while node.parent_node is not None:
            used_clauses.append(node.parent_clause)
            node = node.parent_node

        return used_clauses

    def _get_const_types(self, const_name):
        const_types = get_wikidata_types(const_name)
        return const_types
    






    # the main backchaining algorithm based on recursion

    
    def fol_bc_ask(self, kb, query):
        return self._fol_bc_or(kb, query, {})

    def _fol_bc_or(self, kb, goal , theta):
        fetched_rules = self._fetch_rules(kb, goal)
        for fetched_rule in fetched_rules:
            kb.remove(fetched_rule)
            if isinstance(fetched_rule, Predicate):
                lhs, rhs = [], fetched_rule
                lhss = []
            elif isinstance(fetched_rule, ForAll):
                lhs, rhs = fetched_rule.body.lhs, fetched_rule.body.rhs
                lhss = flatten_and(lhs)
            #return self._fol_bc_and(kb, lhss, self._unify(rhs, goal, theta))
            for theta_prime in self._fol_bc_and(kb, lhss, self._unify(rhs, goal, theta)):
                yield theta_prime

    def _fol_bc_and(self, kb, goals, theta):
        if theta is None:
            return None
        elif len(goals) == 0:
            yield theta
        else:
            first, rest = goals[0], goals[1:]
            theta_prime = self._fol_bc_or(kb, self._substitute(theta, first), theta)
            theta_prime_prime = self._fol_bc_and(kb, rest, theta_prime)
            #return theta_prime_prime
            for theta_prime in self._fol_bc_or(kb, self._substitute(theta, first), theta):
                #return self._fol_bc_and(kb, rest, theta_prime)
                for theta_prime_prime in self._fol_bc_and(kb, rest, theta_prime):
                    yield theta_prime_prime

    def _fetch_rules(self, kb, goal):
        fetched_rules = []
        for clause in kb:
            if isinstance(clause, Predicate) and clause.name == goal.name:
                if len(clause.args) != len(goal.args):
                    continue
            
                
                selected = True
                for i, arg in enumerate(clause.args):
                    if isinstance(goal.args[i], Constant):
                        if isinstance(arg, Constant) and arg.name != goal.args[i].name:
                            selected = False
                            break

                if selected:
                    fetched_rules.append(clause)
        
            elif isinstance(clause, ForAll):
                for rule in flatten_or(clause.body.rhs):
                    if isinstance(rule, Predicate) and rule.name == goal.name:
                        if len(rule.args) != len(goal.args):
                            continue
                        selected = True
                        for i, arg in enumerate(rule.args):
                            if isinstance(arg, Constant) and arg.name != goal.args[i].name:
                                selected = False
                                break
                        if selected:
                            fetched_rules.append(clause)

        return fetched_rules

    def _substitute(self, theta, clause):
        if theta is None:
            return None
        elif isinstance(clause, Predicate):
            substituted_args = [self._substitute(theta, arg) for arg in clause.args]
            return Predicate(clause.name, *substituted_args)
        elif isinstance(clause, Variable):
            for key, value in theta.items():
                if key == clause.name:
                    return Constant(value)
            return clause
        else:
            return clause
        
    

    def _unify(self, x, y, theta={}):
        """
        returns a substitution to make x and y identical
        x , a variable, constant, list, or compound expression
        y , a variable, constant, list, or compound expression
        theta, the substitution built up so far (optional, default = {})
        """
        if theta is None:
            return None
        elif str(x) == str(y):
            return theta
        elif isinstance(x, Constant) and isinstance(y, str):
            if x.name == y:
                return theta
        elif isinstance(y, Constant) and isinstance(x, str):
            if y.name == x:
                return theta
        elif isinstance(x, Variable):
            return self._unify_var(x, y, theta)
        elif isinstance(y, Variable):
            return self._unify_var(y, x, theta)
        elif isinstance(x, Predicate) and isinstance(y, Predicate):
            return self._unify(x.args, y.args, self._unify(x.name, y.name, theta))
        elif isinstance(x, tuple) and isinstance(y, tuple):
            return self._unify(x[1:], y[1:], self._unify(x[0], y[0], theta))
        else:
            return None
        
    def _unify_var(self, var, x, theta):
        """
        returns a substitution to make var and x identical
        """
        # instead of var, should be var.name

        if var.name in theta:
            return self._unify(theta[var.name], x, theta)

        elif x.name in theta:
            return self._unify(var, theta[x], theta)
        else:
            theta[var.name] = x.name
            return theta




class ZSCoT(Agent):
    def __init__(self, retriever, dataset_name):
        super().__init__()
        if retriever == 'gpt3':
            self.retriever = GPT3_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        elif retriever == 'st':
            self.retriever = ST_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        self.dataset_name = dataset_name


    def __call__(self, query, kb, llm_name, log, max_steps):
        answer = self._prove(query, llm_name, log=log, max_steps=max_steps)
        return str(answer)
    
    def _prove(self, query, llm_name, log=None, max_steps=10):
        answer, proof = GetMonolithicProof(llm_name, self.dataset_name, 'agent/llm/llm_prompts')(QUERY=query)
        log("proof", proof)
        return answer

class FSCoT(Agent):
    def __init__(self, retriever, dataset_name):
        super().__init__()
        if retriever == 'gpt3':
            self.retriever = GPT3_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        elif retriever == 'st':
            self.retriever = ST_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        self.dataset_name = dataset_name

    def __call__(self, query, kb, llm_name, log, max_steps):
        answer = self._prove(query, llm_name, kb, log=log, max_steps=max_steps)
        return str(answer)
    
    def _prove(self, query, llm_name, kb, log=None, max_steps=10):

        answer, proof = GetFewshotProof(llm_name, self.dataset_name, 'agent/llm/llm_prompts')(QUERY=query)
        log("proof", proof)
        return answer

class ZSRAG(Agent):
    def __init__(self, retriever, dataset_name):
        super().__init__()
        if retriever == 'gpt3':
            self.retriever = GPT3_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        elif retriever == 'st':
            self.retriever = ST_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        self.dataset_name = dataset_name
    def __call__(self, query, kb, llm_name, log, max_steps):
        time.sleep(5)
        kb = self.retriever(query, kb, 15)
        answer = self._prove(query, llm_name, kb, log=log, max_steps=max_steps)
        return str(answer)
    
    def _prove(self, query, llm_name, kb, log=None, max_steps=10):
        answer, proof = GetMonolithicProofRAG(llm_name, self.dataset_name, 'agent/llm/llm_prompts')(QUERY=query, KB=kb)
        log("proof", proof)
        return answer

class FSRAG(Agent):
    def __init__(self, retriever, dataset_name):
        super().__init__()
        if retriever == 'gpt3':
            self.retriever = GPT3_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        elif retriever == 'st':
            self.retriever = ST_Retriever(dataset_name, os.path.join(os.getcwd(), 'models'))
        self.dataset_name = dataset_name
    def __call__(self, query, kb, llm_name, log, max_steps):
        kb = self.retriever(query, kb, 15)
        answer = self._prove(query, llm_name, kb, log=log, max_steps=max_steps)
        return str(answer)
    
    def _prove(self, query, llm_name, kb, log=None, max_steps=10):

        answer, proof = GetFewshotProofRAG(llm_name, self.dataset_name, 'agent/llm/llm_prompts')(QUERY=query, KB=kb)
        log("proof", proof)
        return answer



# class Typed_Resolution_synthetic(Agent):
#     def __init__(self):
#         super().__init__()
#         self.nli = NLI()
        
#     def __call__(self, query, llm_name, log, max_steps, typed_constants='True'):

#         answer = self.prove(query,llm_name,  log=log, max_steps=max_steps, typed_constants=typed_constants)
#         return str(answer)
    

#     def prove(self, query, llm_name, log=None, max_steps=10, typed_constants='True'):

#         # # convert query to FOL
#         query_fol = self._query2fol(query, llm_name, log)
#         if typed_constants == 'True':
#             self._set_const_types(query_fol)

#         pos_query = query_fol
#         neg_query = self._negate_preds(pos_query)
#         log("negated_query", neg_query)




#         typed_axiom = self._get_type_axiom(query, query_fol, llm_name, log)

#         ##typed_axiom = Predicate('can_hold', [Variable('x', 'physical container'), Variable('y', 'physical object')])
#         typed_axiom_neg = self._negate_types(typed_axiom)

#         ##neg_query = Not(Predicate('can_hold', [Constant('basket'), Constant('bing')]))
    
#         substitutions, scores = self._unify(typed_axiom, neg_query, typed_constants)
#         log("substitutions", substitutions)

#         substitutions_neg, scores_neg = self._unify(typed_axiom_neg, neg_query, typed_constants)
#         log("substitutions_neg", substitutions_neg)

#         answer = all(a >= b for a, b in zip(scores, scores_neg))

#         return answer

    
    
#     def _negate_types(self, clause):
        
#         if isinstance(clause, Predicate):
#             neg_args = []
#             for arg in clause.args[0]:
#                 if isinstance(arg, Variable):
#                     if arg.type is not None:
#                         neg_args.append(Variable(arg.name, 'not '+arg.type))
#                 elif isinstance(arg, Constant):
#                     neg_args.append(Constant('not ' + arg.name))
#             clause_neg_typed = Predicate(clause.name, neg_args)
#         elif isinstance(clause, Not):
#             inner_clause = clause.arg
#             neg_args = []
#             for arg in inner_clause.args[0]:
#                 if isinstance(arg, Variable):
#                     if arg.type is not None:
#                         neg_args.append(Variable(arg.name, 'not '+arg.type))
#                 elif isinstance(arg, Constant):
#                     neg_args.append(Constant('not ' + arg.name))
#             clause_neg_typed = Not(Predicate(inner_clause.name, neg_args))

#         return clause_neg_typed

                    

    
#     def _unify(self, pos, neg, typed_constants):

#         # if isinstance(literal1, Predicate) and isinstance(literal2, Predicate) or isinstance(literal1, Not) and isinstance(literal2, Not):
#         #     return None, None
#         # elif isinstance(literal1, Predicate) and isinstance(literal2, Not):
#         #     pos = literal1; neg = literal2.arg
#         # elif isinstance(literal1, Not) and isinstance(literal2, Predicate):
#         #     pos = literal2; neg = literal1.arg


#         # the dict of substitutions with the corresponding scores and types
#         substitutions = {}
#         if len(pos.args[0]) != len(neg.arg.args[0]):
#             return None

#         for i in range(len(pos.args[0])):
#             # Case 1: unifying constant with constant
#             # TODO: for now just exact sting matches, but need to allow co-references
#             if isinstance(pos.args[0][i], Constant) and isinstance(neg.arg.args[0][i], Constant):
#                 if pos.args[0][i].name != neg.arg.args[0][i].name:
#                     return None
#                 else:
#                     substitutions[pos.args[0][i]] = (neg.arg.args[0][i], neg.arg.args[0][i], 1)
#                     substitutions[neg.arg.args[0][i]] = (pos.args[0][i], pos.args[0][i], 1)
                
#             # Case 2: unifying variable with constant    
#             elif isinstance(pos.args[0][i], Constant) and isinstance(neg.arg.args[0][i], Variable):
#                 neg_type = neg.arg.args[0][i].type
#                 if typed_constants == 'True':
#                     pos_types = pos.args[0][i].type
#                     type_score = 0
#                     for pos_type in pos_types:
#                         type_score1 = self.nli(pos_type, neg_type)['scores'][0]; type_score2 = self.nli(neg_type, pos_type)['scores'][0]
#                         if type_score1 > type_score2 and type_score1 > type_score:
#                             substituted_type = pos_type
#                             type_score = type_score1
#                         elif type_score2 > type_score1 and type_score2 > type_score:
#                             substituted_type = neg_type
#                             type_score = type_score2
                
#                 else:
#                     pos_type = pos.args[0][i].name
                
#                     neg_type = neg.arg.args[0][i].type
#                     type_score1 = self.nli(pos_type, neg_type)['scores'][0]; type_score2 = self.nli(neg_type, pos_type)['scores'][0]
#                     if type_score1 > type_score2:
#                         substituted_type = pos_type
#                         type_score = type_score1
#                     else:
#                         substituted_type = neg_type
#                         type_score = type_score2
#                 substitutions[neg.arg.args[0][i]] = (pos.args[0][i], substituted_type, type_score)
#                 substitutions[pos.args[0][i]] = (pos.args[0][i], substituted_type, type_score)
#             elif isinstance(pos.args[0][i], Variable) and isinstance(neg.arg.args[0][i], Constant):
#                 pos_type = pos.args[0][i].type
#                 if typed_constants == 'True':
#                     neg_types = neg.arg.args[0][i].type
#                     type_score = 0
#                     for neg_type in neg_types:
#                         type_score1 = self.nli(pos_type, neg_type)['scores'][0]; type_score2 = self.nli(neg_type, pos_type)['scores'][0]
#                         if type_score1 >= type_score2 and type_score1 > type_score:
#                             substituted_type = pos_type
#                             type_score = type_score1
#                         elif type_score2 > type_score1 and type_score2 > type_score:
#                             substituted_type = neg_type
#                             type_score = type_score2
                
#                 else:
#                     neg_type = neg.arg.args[0][i].name
#                     type_score1 = self.nli(pos_type, neg_type)['scores'][0]; type_score2 = self.nli(neg_type, pos_type)['scores'][0]
#                     # variable being the more specific type
#                     if type_score1 > type_score2:
#                         substituted_type = pos_type
#                         type_score = type_score1
#                     else:
#                         substituted_type = neg_type
#                         type_score = type_score2

#                 substitutions[pos.args[0][i]] = (neg.arg.args[0][i], substituted_type, type_score)
#                 substitutions[neg.arg.args[0][i]] = (neg.arg.args[0][i] , substituted_type, type_score)
            

#             # Case 3: unifying variable with variable
#             elif isinstance(pos.args[0][i], Variable) and isinstance(neg.arg.args[0][i], Variable):
#                 pos_type = pos.args[0][i].type
#                 neg_type = neg.arg.args[0][i].type
#                 type_score1 = self.nli(pos_type, neg_type)['scores'][0]; type_score2 = self.nli(neg_type, pos_type)['scores'][0]
#                 if type_score1 > type_score2:
#                     substituted_type = pos_type
#                     type_score = type_score1
#                 else:
#                     substituted_type = neg_type
#                     type_score = type_score2
#                 substitutions[pos.args[0][i]] = (neg.arg.args[0][i], substituted_type, type_score)
#                 substitutions[neg.arg.args[0][i]] = (pos.args[0][i], substituted_type, type_score)

            
            

#         scores = [substitution[2] for substitution in substitutions.values()]
#         return substitutions, scores


#     def _set_const_types(self, query_fol):
#         if isinstance(query_fol, Predicate):
#             for const in query_fol.args[0]:
#                 const_name = const.name
#                 const_types = get_wikidata_types(const_name)
#                 if const_types:
#                     const.type = const_types
#                 else:
#                     const.type = const.name
            
#         elif isinstance(query_fol, Not):
#             inner_clause = query_fol.arg
#             for const in inner_clause.args[0]:
#                 const_name = const.name
#                 const_types = get_wikidata_types(const_name)
#                 if const_types:
#                     const.type = const_types
#                 else:
#                     const.type = const.name

    


# def _query2fol(self, question, llm_name, log):

#         # assuming an untyped predicate as query
#         query_fol_str = "At(Presseum, South Korea)"
        
#         predicate = query_fol_str.split('(')[0].strip()
#         arg1, arg2 = query_fol_str.split('(')[1].split(')')[0].split(',')

#         #query_fol = Predicate(predicate, Constant(arg1.strip(), get_wikidata_types(arg1.strip())), Constant(arg2.strip(), get_wikidata_types(arg2.strip())))
#         query_fol = Predicate(predicate, Constant(arg1.strip(), get_wikidata_types(arg1.strip())), Constant(arg2.strip(), get_wikidata_types(arg2.strip())))

#         # convert query to FOL 
#         # if llm_name == 'gpt3.5':
#         #     prompts_dir = 'agent/llm/llm_prompts/chat/Query2FOL.yaml'
#         # elif llm_name == 'gemini':
#         #     prompts_dir = 'agent/llm/llm_prompts/completion/Query2FOL.yaml'
#         # query_fol_str, response = Query2FOL(prompts_dir, llm_name)(QUESTION=question)
#         # log("response", response)
#         # predicate = query_fol_str.split('(')[0].strip()
#         # args = query_fol_str.split('(')[1].split(')')[0].split(',')
#         # query_fol = Predicate(predicate, [Constant(arg.strip()) for arg in args])

    
#         return query_fol
    
#     def _get_type_axiom(self,question, query_fol, llm_name, log):
#         query_fol_str = str(query_fol)

#         # get the typed query
#         if llm_name == 'gpt3.5':
#             prompts_dir = 'agent/llm/llm_prompts/chat/GetTypeAxiom.yaml'
#         elif llm_name == 'gemini':
#             prompts_dir = 'agent/llm/llm_prompts/completion/GetTypeAxiom.yaml'
#         typed_query_str, response = GetTypeAxiom(prompts_dir, llm_name)(QUESTION=question, FOL_QUESTION=query_fol_str)
        
#         #typed_query_str = "requires-electricity(x) | x: electronic device"
#         #log("response", response)
#         typed_query = parse_predicate_string(typed_query_str)
#         return typed_query
       # #typed_axiom = self._get_type_axiom(query, query_fol, llm_name, log)
        # _, typed_axiom1 = parse_fol("FOR_ALL ?A:place or location ?B:city ?C:country, PartOf(?A:place or location, ?B:city) && PartOf(?B:city, ?C:country) => PartOf(?A:place or location, ?C:country)")
        # _, typed_axiom2 = parse_fol("FOR_ALL ?A:country ?B:country, ShareLandBorders(?A:country, ?B:country) => LandConnected(?A:country, ?B:country)")
        # _, typed_axiom3 = parse_fol("FOR_ALL ?A:place or location ?B:place or location ?C:country ?D:country, PartOf(?A:place or location, ?C:country) && PartOf(?B:place or location, ?D:country) && LandConnected(?C:country, ?D:country) => CanDriveBetween(?A:place or location, ?B:place or location)")      
        # #kb = [typed_axiom, Predicate('PartOf', Constant('Milad Tower', 'location'), Constant('Tehran', 'location')), Predicate('PartOf', Constant('Tehran', 'location'), Constant('Iran', 'location'))]
        # kb = [typed_axiom1, #typed_axiom2,
        #         typed_axiom3, 
        #         Predicate('PartOf', Constant('Sayḩ Rab‘', 'desert'), Constant('United Arab Emirates', 'country')),
        #         Predicate('PartOf', Constant('United Arab Emirates', 'country'), Constant('Asia', 'continent')), 
        #         #Predicate('PartOf', Constant('Milad Tower', 'tourist attraction'), Constant('Iran', 'country')),
        #         Predicate('LandConnected', Constant('France', 'country'), Constant('Italy', 'country')),
        #         #Predicate('PartOf', Constant('Arabian Desert', 'desert'), Constant('Iraq', 'country')),
               
        #        Predicate('PartOf', Constant('Agnano', 'Mountain'), Constant('Naples', 'city')),
        #        Predicate('PartOf', Constant('Naples', 'city'), Constant('Italy', 'country'))]
        
        #pos_query = Predicate('CanDriveBetween', Constant('Agnano', 'Mountain'), Constant('Eiffel Tower', 'tourist attraction'))]