import re
from collections import defaultdict

import numpy as np

from gym.utils import seeding
from gym.envs import registration
from minihack import MiniHackNavigation, LevelGenerator, RewardManager
from minihack.tiles.rendering import get_des_file_rendering
from minihack.base import MH_NETHACKOPTIONS

from envs.minihack import minihack_grid 
from nle import nethack
from nle.nethack import Command, CompassDirection
from nle.env.base import FULL_ACTIONS

from util import cprint


_ACTIONS = \
     list(CompassDirection) + \
     [Command.KICK]

_ACTION2INDEX = {a:i for i,a in enumerate(_ACTIONS)}

_FULL_ACTION2INDEX = {a:i for i,a in enumerate(FULL_ACTIONS)}


DEBUG = False


class MiniHackFireDungeon(MiniHackNavigation):
    """Environment for "empty" task."""
    def __init__(
        self,
        *args,
        grid_h=3,
        grid_w=3,
        room_size=4,
        max_difficulty=0,
        sparse_rewards=False,
        fully_observable=False,
        p=[1.0, 0.0, 0.0],
        full_maze=False,
        randomize_monsters=False,
        reward_kill=True,
        reward_armor=True,
        reward_dist='uniform',
        n_lava=0,
        n_wall=0,
        n_monster=3,
        generator_dist='constant',
        max_episode_steps=250,
        lit=True,
        autopickup=False,
        penalty_step=0.0,
        seed=0,
        obl_correction=False,
        use_learned_beliefs=False,
        goal_hint_p=0.0,
        fixed_environment=False,
        **kwargs
    ):  
        kwargs["character"] = "bar-hum-law-mal" # Tourist by default
        kwargs["max_episode_steps"] = max_episode_steps
        kwargs["autopickup"] = kwargs.pop("autopickup", autopickup)
        kwargs["actions"] = kwargs.pop("actions", _ACTIONS)

        kwargs["allow_all_yn_questions"] = kwargs.pop(
            "allow_all_yn_questions", True
        )

        self.fully_observable = fully_observable
        if fully_observable:
            kwargs["wizard"] = True
        
        custom_options = ["nudist", "role:bar"]
        if not autopickup:
            custom_options.append('!autopickup')
        kwargs["options"] = tuple(list(MH_NETHACKOPTIONS) + custom_options)

        # grid_h and grid_w are in room units, e.g. 3x3 == 3 rooms by 3 rooms
        self.grid_size = (grid_h, grid_w)
        self.room_size = room_size
        self.max_difficulty = max_difficulty
        self.difficulty = 0
        self.sparse_rewards = sparse_rewards

        self.generator_dist = generator_dist # How random entity counts are sampled
        self.n_wall = n_wall
        self.n_monster = n_monster

        self.n_lava = n_lava # Too crazy for now

        self.reward_armor = reward_armor
        self.reward_kill = reward_kill
        self.randomize_monsters = randomize_monsters
        
        self.p = p
        self.obl_correction = obl_correction
        self.use_learned_beliefs = use_learned_beliefs
        self.item_info = {
            '0': {'name':'red dragon scale mail', 'symbol':'['},
            '1': {'name':'leather jacket', 'symbol':'['},
            '2': {'name':'leather armor', 'symbol':'['},
        }

        if self.obl_correction and self.use_learned_beliefs:
            self.belief_dist = {
                'item_present_index': np.ones(len(self.item_info))/len(self.item_info)
            }
        else: # assume ground truth beliefs
            belief_dist = np.array(p)
            self.belief_dist = {
                'item_present_index': belief_dist/belief_dist.sum()
            }

        self.goal_hint_p = goal_hint_p
        self.reward_dist = reward_dist
        self.item_present_index = None

        self.last_message = ''

        self._init_empty_grid(seed=seed)

        self.lvl_gen = LevelGenerator(w=1, h=1, lit=lit)

        super().__init__(*args, 
            des_file=self.lvl_gen.get_des(), **kwargs)

        self.penalty_step = penalty_step
        kwargs["penalty_step"] = penalty_step

        self.seed(seed) # sets np_random seed

        self.unseeded_np_random,_ = seeding.np_random()

        self._regenerate_level()

    def _reset_episodic_counts(self):
        # We track these for each episode
        self.episodic_counts = {
            'wear_armor': 0,
            'wear_fireproof': 0,
            'monster_kills': 0,
            'goal_reached': 0,
            'solved_room_count': 0,
            'solved_fire_room': 0,
            'solved_safe_room': 0,
        }

    def _room_coord2index(self, coord):
        """
        Room idx is room_y*room_grid_width + room_x
        Coord is x,y, but grid_size is y,x
        """
        return coord[1]*self.grid_size[0] + coord[0]

    def _room_index2coord(self, idx):
        return idx % self.grid_size[1], idx // self.grid_size[1]

    def _place_room(self, room_idx):
        (tl_x, tl_y), (br_x, br_y) = self.room_coords[room_idx]

        for y in range(tl_y,br_y+1):
            self.grid.set_char('-', loc=(tl_x,y))
            self.grid.set_char('-', loc=(br_x,y))

        for x in range(tl_x,br_x+1):
            self.grid.set_char('-', loc=(x,tl_y))
            self.grid.set_char('-', loc=(x,br_y))

        for y in range(tl_y+1,br_y):
            for x in range(tl_x+1, br_x):
                self.grid.set_char('.', loc=(x,y))

    def _sample_neighbors(self, room_idx, num_neighbors=1):
        """
        Given a room coord, samples two joint neighbors.
        """
        x,y = self._room_index2coord(room_idx)
        height,width = self.grid_size

        assert 0 <= x < width and 0 <= y < height

        dirs = ['N','E','S','W']

        if x == 0:
            dirs.remove('W')
        elif x == self.grid_size[1] - 1:
            dirs.remove('E')

        if y == 0:
            dirs.remove('N')
        elif y == self.grid_size[0] - 1:
            dirs.remove('S')

        neighbor_mask = np.zeros_like(self.free_room_mask)
        for d in dirs:
            if d == 'N':
                neighbor_mask[y-1][x] = True
            elif d == 'E':
                neighbor_mask[y][x+1] = True
            elif d == 'S':
                neighbor_mask[y+1][x] = True
            else:
                neighbor_mask[y][x-1] = True

        free_neighbor_mask = np.logical_and(
            self.free_room_mask, neighbor_mask)

        p = free_neighbor_mask.flatten().astype(np.float32)
        z = p.sum()

        sample_size = min(round(z), num_neighbors)

        if not sample_size > 0:
            return [], [] # No free neighbors

        neighbors_idx = \
            self.np_random.choice(
                range(len(p)), 
                sample_size, 
                p=p/z, 
                replace=False).tolist()

        door_locs = []
        for idx in neighbors_idx:
            room_x,room_y = self._room_index2coord(idx)
            self.free_room_mask[room_y][room_x] = False

            door_loc = self._door_loc_from_neighbor(
                room_idx=room_idx,
                neighbor_coord=(room_x,room_y))
            door_locs.append(door_loc)

        return neighbors_idx, door_locs

    def _door_loc_from_neighbor(self, room_idx, neighbor_coord):
        room_x,room_y = self._room_index2coord(room_idx)
        offset_x = neighbor_coord[0] - room_x
        offset_y = neighbor_coord[1] - room_y
        room_dim = self.room_size

        (x,y),_ = self.room_coords[room_idx]

        if offset_x == 0 and offset_y == 0:
            return None

        if offset_x != 0:
            if offset_x < 0:
                door_x = x
            else:
                door_x = x + (room_dim + 1)

            door_y = y + self.np_random.randint(1,self.room_size+1)
        else:
            if offset_y < 0:
                door_y = y
            else:
                door_y = y + (room_dim + 1)

            door_x = x + self.np_random.randint(1,self.room_size+1)

        return door_x,door_y

    def _init_empty_grid(self, seed=None):
        """
        Grid is initially the empty room layout.
        """
        height = self.grid_size[0]*(self.room_size+1) + 1
        width = self.grid_size[1]*(self.room_size+1) + 1

        self.free_room_mask = np.ones(self.grid_size, dtype=np.bool)

        empty_grid_str = ((' '*width + '\n')*height)
        self.grid = minihack_grid.Grid(
            height=height,
            width=width, 
            fixed_grid_str=empty_grid_str,
            seed=seed)

        custom_objects = {
            # Armors
            k: {'name':info['name'], 'symbol':info['symbol']} 
                for k, info in self.item_info.items()}
        custom_objects.update({
                # Goal indicators
                'r': {'name':'ruby', 'symbol':'*'},
                'b': {'name':'sapphire', 'symbol':'*'},
            })

        self.grid.add_custom_objects(custom_objects)

        self.minion_name = 'pyrolisk'
        self.minion_symbol = 'c'

        self.grid.add_custom_monsters({
            'j': {'name': self.minion_name, 'symbol': self.minion_symbol}
        })

        h,w = self.grid.map.shape

        self._generate_rooms()

    def _generate_rooms(self):
        # Generate room coordinates and masks
        self.room_coords = []
        self.room_masks = []
        for y in range(self.grid_size[0]):
            for x in range(self.grid_size[1]):
                # Top left corner of free space in room
                top_left = (x*(self.room_size+1), y*(self.room_size+1))
                bottom_right = (
                    x*(self.room_size+1) + self.room_size + 1, 
                    y*(self.room_size+1) + self.room_size + 1)

                self.room_coords.append((top_left, bottom_right))
                
                free_top_left = top_left[0]+1,top_left[1]+1
                free_bottom_right = bottom_right[0]-1,bottom_right[1]-1

                room_mask = self.grid.mask_rect(free_top_left,free_bottom_right)
                self.room_masks.append(room_mask)

    def print_mask(self, mask):
        print('\n'.join([''.join([str(int(x)) for x in row]) for row in mask]))

    def _regenerate_level(self):
        """
        Generate n_clutter walls and randomly place the agent and goals (an apple and banana)

        Note the goals are two items, with auto-pickup on.
        """
        self._reset_episodic_counts()
        self._last_agent_loc = None
        self.episode_reward = 0

        # === Generate topology ===
        self.grid.clear()
        self.free_room_mask.fill(True)

        h,w = self.grid.map.shape
        
        # Sample difficulty (number of additional rooms to traverse)
        if self.max_difficulty > 0:
            if self.np_random.rand() < 1/(self.max_difficulty+1):
                difficulty = 0
            else:
                difficulty = self.np_random.randint(1,self.max_difficulty+1)
        else:
            difficulty = 0
        self.difficulty = difficulty

        # Sample start room
        start_room_x = self.np_random.choice(range(self.grid_size[1]))
        start_room_y = self.np_random.choice(range(self.grid_size[0]))

        self.start_room_idx = self._room_coord2index((start_room_x, start_room_y))
        start_room_x,start_room_y = self._room_index2coord(self.start_room_idx)
        self.free_room_mask[start_room_y][start_room_x] = False

        self.door_locs = []

        # Determine fork point
        self.fork_room_idx = self.start_room_idx
        forked_rooms, doors = self._sample_neighbors(self.start_room_idx, num_neighbors=2)

        self.door_locs += doors

        # Grow start room based on difficulty
        path_rooms = [self.start_room_idx]
        for _ in range(difficulty):
            neighbor, doors = self._sample_neighbors(
                self.start_room_idx, num_neighbors=1)
            if len(neighbor) == 1:
                self.start_room_idx = neighbor[0]
                path_rooms.append(self.start_room_idx)

                self.door_locs += doors
            else:
                break

        self.rooms = path_rooms + forked_rooms

        # Index into tile space to get room masks
        self.room_mask = np.zeros((h,w), dtype=np.bool)

        for room_idx in self.rooms:
            # Update global room mask
            room_mask = self.grid.mask_rect(*self.room_coords[room_idx])
            self.room_mask = np.logical_or(self.room_mask, room_mask)

            # Draw room boundaries
            self._place_room(room_idx)

        # # Place walls
        # for i in range(self.n_wall):
        #     self.grid.set_char('-', loc='random_free', mask=~self.wall_mask)

        # Place monsters in right room
        for i in range(self.n_monster):
            self.grid.set_char('j', 
                loc='random_free',
                mask=~self.room_masks[forked_rooms[1]])

        # Place room markers
        fork_room_mask = self.room_masks[self.fork_room_idx]

        safe_door_mask = np.logical_or(
            fork_room_mask,
            self.grid.mask_neighbors(loc=[self.door_locs[0]]))

        danger_door_mask = np.logical_or(
            fork_room_mask,
            self.grid.mask_neighbors(loc=[self.door_locs[1]]))

        # Place agent in first room
        self.grid.set_char('b', loc='random_free', mask=~safe_door_mask)
        self.grid.set_char('r', loc='random_free', mask=~danger_door_mask)

        # Place agent in first room
        self.grid.set_char(
            '<', 
            loc='random_free', 
            mask=~self.room_masks[self.start_room_idx])

        # Place goals
        self.goal_locs = []
        for room_idx in forked_rooms:
            goal_loc = self.grid.set_char(
                '>', 
                loc='random_free', 
                mask=~self.room_masks[room_idx])

            self.goal_locs.append(goal_loc)
        self.safe_goal_loc = self.goal_locs[0]
        self.fire_goal_loc = self.goal_locs[1]

        # Place doors
        for (x,y) in self.door_locs:
            self.grid.set_char('+', loc=(x,y))

        # Generate armor with probability p
        # Choose an item index
        item_dist = self.p
        item_options = range(len(item_dist))
        if self.obl_correction:
            if self.use_learned_beliefs:
                item_dist = self.belief_dist['item_present_index']
            self.item_present_index = self.unseeded_np_random.choice(item_options, p=item_dist)

            # print('sampled armor from dist', item_dist)
        else:
            self.item_present_index = self.np_random.choice(item_options, p=item_dist)

        item_char = list(self.item_info)[self.item_present_index]

        self.grid.set_char(
            item_char, 
            loc='random_free', 
            mask=~self.room_masks[self.fork_room_idx])

        # === Compute shortest path metrics ===
        clutter_chars = []
        if self.n_lava > 0:
            clutter_chars.append('L')
        if self.n_wall > 0:
            clutter_chars.append('-')
        if self.n_monster > 0:
            clutter_chars.append('j')
        grid_info = \
            self.grid.get_metrics(
                goal_chars=['>',] + [k for k in self.item_info],
                clutter_chars=clutter_chars,
                aliases={
                    '>': 'goal',
                    '-': 'wall',
                    'L': 'lava',
                    'j': 'monster',
                    **{k:'armor' for k in self.item_info.keys()}
                })
        self.passable = grid_info['passable']
        self.shortest_path_length = grid_info['shortest_path_lengths']
        self.n_clutter_placed = grid_info['clutter_counts']

        # cprint(DEBUG, grid_info)
        # print(f'SEED:{self._level_seed}')
        # print(self.grid.map)
        # print('des file is\n')
        # print(self.grid.des)
        # import pdb; pdb.set_trace()

    def _wizard_accept_death(self, cur_message, r):
        if self.fully_observable:
            match_die = re.match(r"die\?", cur_message)
            if match_die:
                _actions_tmp = self._actions
                self._actions = FULL_ACTIONS
                obs, _, done, info = super().step(_FULL_ACTION2INDEX[ord('y')])
                self._actions = _actions_tmp

                if done:
                    info['episodic_counts'] = self.episodic_counts

                if self.sparse_rewards:
                    r = 0

                return obs, r, done, info
        else:
            return False

    def _is_less_than_one_step_away(self, a, b):
        return (abs(a[0]-b[0]) <= 1 and abs(a[1]-b[1]) <= 1)

    def _get_loc_offsets(self):
        if self.grid_size[0] == 3:
            if self.room_size == 4:
                return (-32,-3)
            elif self.room_size == 5:
                return (-30,-1)
        elif self.grid_size[0] == 4:
            if self.room_size == 4:
                return (-28,0)

        raise ValueError(f'Unsupported room size {self.room_size} for grid size {self.grid_size[0]}.')

    def step(self, action):
        # if self.fully_observable:
            # return self.step_fully_observable(action)
        obs, r, done, info = super().step(action)
        self.episode_reward += r

        # === Check if goal ===
        blstats = obs['blstats']
        loc_offset = self._get_loc_offsets()
        agent_loc = (blstats[0]+loc_offset[0],blstats[1]+loc_offset[1])

        # print('blstats_loc=',blstats[0],blstats[1])
        # print('agent_loc=',agent_loc)

        if not done and agent_loc in self.goal_locs:
            info['end_status'] = self.StepStatus.TASK_SUCCESSFUL
            r += 1.0
            self.episode_reward += 1.0
            done = True

            self.episodic_counts['goal_reached'] = 1
            self.episodic_counts['solved_room_count'] = self.difficulty + 2
            if agent_loc == self.fire_goal_loc:
                self.episodic_counts['solved_fire_room'] = 1

                if not self.reward_kill:
                    r += self.n_monster
                    self.episode_reward += self.n_monster
            else:
                self.episodic_counts['solved_safe_room'] = 1

            if self.sparse_rewards:
                r = self.episode_reward

            return obs, r, done, info

        cur_message = obs['message'].tobytes().decode('utf-8').lower()
        died = self._wizard_accept_death(cur_message, r)
        if died: return died

        match_armor_loc_re = r"you see here .* (scale mail|jacket|armor)"

        match_armor_loc = re.match(match_armor_loc_re, cur_message)
        if match_armor_loc:
            self.autowear_interrupted = True
            _actions_tmp = self._actions
            self._actions = FULL_ACTIONS
            obs, _, done, info = super().step(_FULL_ACTION2INDEX[nethack.Command.PICKUP])
            self._actions = _actions_tmp
            cur_message = obs['message'].tobytes().decode('utf-8').lower()
            died = self._wizard_accept_death(cur_message, r)
            if died: return died

            match_armor_pickup = re.match(r"^([a-z]) - .* (scale mail|jacket|armor)", cur_message)
            if match_armor_pickup:
                self.armor_inventory_index = match_armor_pickup[1]
                if self.armor_inventory_index is not None:
                    wear_actions = f'W{self.armor_inventory_index}'
                    for a in wear_actions:
                        if done: break
                        _actions_tmp = self._actions
                        self._actions = FULL_ACTIONS
                        obs, _, done, info = super().step(_FULL_ACTION2INDEX[ord(a)])
                        self.autowear_interrupted = False
                        self._actions = _actions_tmp

                        cur_message = obs['message'].tobytes().decode('utf-8').lower()
                        died = self._wizard_accept_death(cur_message, r)
                        if died: return died

                        # Track successful wear
                        if a == self.armor_inventory_index:
                            if self.reward_armor:
                                r += 1.0
                                self.episode_reward += 1.0

                            self.episodic_counts['wear_armor'] = 1

                            if self.item_present_index == 0:
                                self.episodic_counts['wear_fireproof'] = 1

        elif not done and self.autowear_interrupted:
            _actions_tmp = self._actions
            self._actions = FULL_ACTIONS
            obs, _, done, info = super().step(_FULL_ACTION2INDEX[ord('W')])
            cur_message = obs['message'].tobytes().decode('utf-8').lower()
            died = self._wizard_accept_death(cur_message, r)
            if died: return died

            match_armor = re.match(r"what do you want to wear\? \[(.+) or \?\*\]", cur_message)
            if not done and match_armor:
                self.armor_inventory_index = match_armor[1]
                if len(self.armor_inventory_index) > 1:
                    self.armor_inventory_index = self.armor_inventory_index[0]
                obs, _, done, info = super().step(_FULL_ACTION2INDEX[
                    ord(self.armor_inventory_index)
                ])
                cur_message = obs['message'].tobytes().decode('utf-8').lower()
                died = self._wizard_accept_death(cur_message, r)
                if died: return died
                
            self._actions = _actions_tmp

        elif f'you kill the {self.minion_name}!' in cur_message:
            if self.reward_kill:
                r += 1.0
                self.episode_reward += 1.0

            self.episodic_counts['monster_kills'] += 1

        self.last_message = cur_message

        if done:
            end_status = info.get('end_status', None)
            if end_status is not None and end_status.name == 'TASK_SUCCESSFUL':
                self.episodic_counts['goal_reached'] = 1
                self.episodic_counts['solved_room_count'] = self.difficulty + 2

                agent_loc = self._last_agent_loc
                if agent_loc is not None and \
                    self._is_less_than_one_step_away(agent_loc, self.fire_goal_loc):
                    self.episodic_counts['solved_fire_room'] = 1

                    if not self.reward_kill:
                        r += self.n_monster
                        self.episode_reward += self.n_monster
                else:
                    self.episodic_counts['solved_safe_room'] = 1

            info['episodic_counts'] = self.episodic_counts

            # print('EPISODIC COUNTS:', info['episodic_counts'])
            # print('AUX PROPERTIES:', self.aux_properties)

        self._last_agent_loc = agent_loc

        if self.sparse_rewards:
            if self.episodic_counts['goal_reached'] == 1:
                r = self.episode_reward
            else:
                r = 0

        return obs, r, done, info

    def reset(self):
        self.autowear_interrupted = False

        self._regenerate_level()
        super().update(des_file=self.grid.des)

        if self.fully_observable:
            super().reset()

            for c in "#wizintrinsic\rt\r\r#wizmap\r#wizwish\ra potion of object detection\r":
                obs, sds = self.env.step(ord(c))
            msg = (
                obs[self._original_observation_keys.index("message")]
                .tobytes()
                .decode('utf-8')
            )

            for c in f"q{msg[0]}":
                obs, sds = self.env.step(ord(c))

            _actions_tmp = self._actions
            self._actions = FULL_ACTIONS
            obs, _, _, _ = super().step(self._actions.index(nethack.MiscDirection.WAIT))
            self._actions = _actions_tmp
        else:
            obs = super().reset()

        return obs

    def reset_agent(self):
        super().update(des_file=self.grid.des)
        return super().reset()

    def seed(self, level_seed, *args, **kwargs):
        super().seed(level_seed, *args, **kwargs)

        self._level_seed = level_seed

        self.np_random, _ = seeding.np_random(level_seed)
        self.grid.seed(level_seed)

    @property
    def aux_properties(self):
        return {
            'fireproof_armor': self.item_present_index == 0,
            'belief_fireproof': self.belief_dist['item_present_index'][0],
            'num_rooms': self.difficulty + 2 
        }

    @property
    def belief_spec(self):
        return {
            'item_present_index': {
                'type': 'categorical',
                'size': len(self.item_info)
            }
        }

    @property
    def belief_tokens(self):
        num_item_options = len(self.item_info)
        item_one_hot = np.zeros(num_item_options, dtype=np.int32)
        sample_index = self.np_random.choice(range(num_item_options), p=self.p)
        item_one_hot[sample_index] = 1

        tokens = {
            'item_present_index': item_one_hot
        }
        return tokens

    def set_belief_dist(self, belief_dist):
        self.belief_dist = belief_dist

    @property
    def des_file(self):
        return self.grid.des

    @property
    def grid_str(self):
        return self.grid.map.__str__()

    def render(self, mode='level'):
        if mode == 'level':
            des_file = self.grid.des
            return np.asarray(get_des_file_rendering(des_file, wizard=True))
        else:
            return super().render(mode=mode)


class MiniHackDoFMedium(MiniHackFireDungeon):
    def __init__(
        self,
        *args,
        **kwargs
    ):
        super().__init__(
            *args,
            n_monster=2,
            max_difficulty=2,
            **kwargs
        )

class MiniHackDoFHard(MiniHackFireDungeon):
    def __init__(
        self,
        *args,
        **kwargs
    ):
        super().__init__(
            *args,
            n_monster=3,
            max_difficulty=6,
            room_size=5,
            # sparse_rewards=True,
            **kwargs
        )

class MiniHackDoFProSparseRewards(MiniHackFireDungeon):
    def __init__(
        self,
        *args,
        **kwargs
    ):
        super().__init__(
            *args,
            n_monster=4,
            max_difficulty=9,
            room_size=4,
            grid_h=4,
            grid_w=4,
            sparse_rewards=True,
            reward_armor=False,
            reward_kill=False,
            **kwargs
        )

class MiniHackDoFProHuntSparseRewards(MiniHackFireDungeon):
    def __init__(
        self,
        *args,
        **kwargs
    ):
        super().__init__(
            *args,
            n_monster=4,
            max_difficulty=9,
            room_size=4,
            grid_h=4,
            grid_w=4,
            sparse_rewards=True,
            reward_armor=False,
            reward_kill=True,
            **kwargs
        )

if hasattr(__loader__, 'name'):
  module_path = __loader__.name
elif hasattr(__loader__, 'fullname'):
  module_path = __loader__.fullname


registration.register(
    id='MiniHack-DoF-v0',
    entry_point=module_path + ':MiniHackFireDungeon',
    max_episode_steps=250
)

registration.register(
    id='MiniHack-DoFMedium-v0',
    entry_point=module_path + ':MiniHackDoFMedium',
    max_episode_steps=250
)

registration.register(
    id='MiniHack-DoFHard-v0',
    entry_point=module_path + ':MiniHackDoFHard',
    max_episode_steps=250
)

registration.register(
    id='MiniHack-DoFProSparseRewards-v0',
    entry_point=module_path + ':MiniHackDoFProSparseRewards',
    max_episode_steps=250
)

registration.register(
    id='MiniHack-DoFProHuntSparseRewards-v0',
    entry_point=module_path + ':MiniHackDoFProHuntSparseRewards',
    max_episode_steps=250
)
