# Obtain Mnist High-Level features
import torch
import torch.nn as nn
import torch.optim as optim
import torchvision.transforms as transforms
from torchvision.datasets import MNIST
from torch.utils.data import DataLoader, random_split

n_layer = 10
torch.manual_seed(0)

# Define a transform to preprocess the MNIST data
transform = transforms.Compose([
    transforms.ToTensor(),  # Convert to tensor
    transforms.Normalize((0.5,), (0.5,))  # Normalize input to range [-1, 1]
])

# Load the MNIST dataset
full_dataset = MNIST(root='/home/causal_ksd/data_MNIST/original', train=True, transform=transform, download=True)
test_dataset = MNIST(root='/home/causal_ksd/data_MNIST/original', train=False, transform=transform, download=True)

# Split the dataset into training and validation sets
train_size = int(0.8 * len(full_dataset))
val_size = len(full_dataset) - train_size
train_dataset, val_dataset = random_split(full_dataset, [train_size, val_size])

# Define data loaders
batch_size = 64
train_loader = DataLoader(train_dataset, batch_size=batch_size, shuffle=True)
val_loader = DataLoader(val_dataset, batch_size=batch_size, shuffle=False)
test_loader = DataLoader(test_dataset, batch_size=batch_size, shuffle=False)

# Define the neural network model
class DenseNN(nn.Module):
    def __init__(self, layer_sizes):
        super(DenseNN, self).__init__()
        self.layers = nn.ModuleList()
        for i in range(1, len(layer_sizes)):
            self.layers.append(nn.Linear(layer_sizes[i-1], layer_sizes[i]))
            if i < len(layer_sizes) - 1:
                self.layers.append(nn.ReLU())  # Add ReLU activation except for the last layer

    def forward(self, x):
        for layer in self.layers:
            x = layer(x)
        return x

# Create the model
if n_layer == 15:
    layer_sizes = [784, 100, 20, 15, 10]
elif n_layer == 10:
    layer_sizes = [784, 100, 20, 10]

model = DenseNN(layer_sizes)

# Define loss function and optimizer
criterion = nn.CrossEntropyLoss()
optimizer = optim.Adam(model.parameters(), lr=0.001)

# Training loop
num_epochs = 20
device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
model.to(device)


best_val_loss = float('inf')  # Initialize with a high value
patience = 5  # Number of epochs to wait before early stopping if validation loss doesn't improve
early_stopping_counter = 0

for epoch in range(num_epochs):
    model.train()
    running_loss = 0.0

    for inputs, labels in train_loader:
        inputs, labels = inputs.to(device), labels.to(device)

        optimizer.zero_grad()

        outputs = model(inputs.view(inputs.size(0), -1))  # Flatten the input
        loss = criterion(outputs, labels)
        loss.backward()
        optimizer.step()

        running_loss += loss.item()

    print(f"Epoch {epoch+1}/{num_epochs}, Loss: {running_loss / len(train_loader)}")

    # Evaluate the model on the validation set
    model.eval()
    val_loss = 0.0

    with torch.no_grad():
        for inputs, labels in val_loader:
            inputs, labels = inputs.to(device), labels.to(device)

            outputs = model(inputs.view(inputs.size(0), -1))  # Flatten the input
            loss = criterion(outputs, labels)
            val_loss += loss.item()

    val_loss /= len(val_loader)
    print(f"Validation Loss: {val_loss}")

    # Check if the validation loss improved
    if val_loss < best_val_loss:
        best_val_loss = val_loss
        early_stopping_counter = 0
        # Save the model weights to a file
        model_weights_path = "/home/causal_ksd/models_MNIST/mnist_dense_nn_weights" + str(n_layer) + ".pth"
        torch.save(model.state_dict(), model_weights_path)
        print(f"Model weights saved to {model_weights_path}")
    else:
        early_stopping_counter += 1

    # Early stopping if validation loss doesn't improve for 'patience' epochs
    if early_stopping_counter >= patience:
        print("Early stopping.")
        break
