import os
import pickle

from collections import OrderedDict

from .negative_database_whole import SIMILAR_FURNITURE_NAMES_DICT_V2

from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden, mkdir_if_missing

from .oxford_pets import OxfordPets


@DATASET_REGISTRY.register()
class ScannetppSimilarNegativeV2(DatasetBase):

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT))
        self.dataset_dir = "scannetpp"
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = self.dataset_dir
        self.preprocessed = os.path.join(self.dataset_dir, "preprocessed.pkl")
        self.split_fewshot_dir = os.path.join(self.dataset_dir, "split_fewshot_scannetpp")
        mkdir_if_missing(self.split_fewshot_dir)

        text_file = os.path.join(self.dataset_dir, "scannetpp_semantic_classes.txt")
        
        classnames = self.read_classnames(text_file)
        
        train_classes_no = -1
        test_classes_no = -1
        train_test_classnames, label2dataset, _ = self.get_indices("scannetpp_sem_seg_train", "scannetpp_sem_seg_val", classnames, train_classes_no=train_classes_no, test_classes_no=test_classes_no)
        
        negative_classes = self.get_negatives(SIMILAR_FURNITURE_NAMES_DICT_V2)

        all_classnames = train_test_classnames + negative_classes
        
        train, negatives, negative_ids = self.read_data(classnames, "scannetpp_sem_seg_train", label2dataset, negative_classes=SIMILAR_FURNITURE_NAMES_DICT_V2) # 
        test = self.read_data(classnames, "scannetpp_sem_seg_val", label2dataset) # 

        # Get the list of Datums for the New Domain (only labels no images)
        train_u = self.create_dataset_from_labels_only(negatives, negative_ids, len(train_test_classnames))

        num_shots = cfg.DATASET.NUM_SHOTS
        if num_shots > 0:
            train = self.generate_fewshot_dataset(train, num_shots=num_shots)

        subsample = cfg.DATASET.SUBSAMPLE_CLASSES
        train, train_u, test = OxfordPets.subsample_classes(train, train_u, test, subsample=subsample)

        super().__init__(train_x=train, train_u=train_u, val=test, test=test, classnames=all_classnames)

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for folder, line in enumerate(lines):
                classnames[folder] = line.strip()

        return classnames

    def get_negatives(self, classes_dict):
        negative_classes = []
        self.prev_len = -1
        for key in classes_dict:
            negative_classes.extend(classes_dict[key])
            if self.prev_len > 0:
                assert len(classes_dict[key]) == self.prev_len
            else:
                self.prev_len = len(classes_dict[key])
        print(f"number of negative classes {len(negative_classes)}")
        return negative_classes

    def get_indices(self, train_dir, test_dir, classnames, train_classes_no=0, test_classes_no=0):
        train_dir = os.path.join(self.image_dir, train_dir)
        train_folders = sorted(f.name for f in os.scandir(train_dir) if f.is_dir())
        if train_classes_no > 0:
            train_folders = train_folders[:train_classes_no]

        test_dir = os.path.join(self.image_dir, test_dir)
        test_folders = sorted(f.name for f in os.scandir(test_dir) if f.is_dir())
        if test_classes_no > 0:
            test_folders = test_folders[:test_classes_no]

        train_indices = [int(folder) for folder in train_folders]
        test_indices = [int(folder) for folder in test_folders]

        train_test_classnames = []
        label2dataset = {}
        remaining_classes = []
        cur_id = 0
        for key in classnames:
            if key in train_indices or key in test_indices:
                train_test_classnames.append(classnames[key]) 
                label2dataset[key] = cur_id
                cur_id += 1
            else:
                remaining_classes.append(classnames[key])

        return train_test_classnames, label2dataset, remaining_classes

    def read_data(self, classnames, split_dir, label2dataset, num_classes=None, negative_classes=None):
        split_dir = os.path.join(self.image_dir, split_dir)
        folders = sorted(f.name for f in os.scandir(split_dir) if f.is_dir())

        if num_classes is not None:
            folders = folders[:num_classes]

        items = []

        print(f"Reading data from {split_dir}")

        classes_list = []
        if negative_classes is not None:
            negative_classes_with_same_occurrences = []
            negative_label_ids = []
        for label, folder in enumerate(folders):
            imnames = listdir_nohidden(os.path.join(split_dir, folder))
            classname = classnames[int(folder)]
            classes_list.append(classname)
            if negative_classes is not None:
                negative_classname_list = negative_classes[classname]
                len_negative =  len(negative_classname_list)
                i = 0
            for imname in imnames:
                impath = os.path.join(split_dir, folder, imname)
                if negative_classes is None:
                    item = Datum(impath=impath, label=label2dataset[int(folder)], classname=classname)
                else:
                    negative_classes_with_same_occurrences.append(negative_classname_list[i])
                    negative_label_ids.append(len_negative * label + i)
                    if i + 1 < len_negative:
                        i += 1
                    else:
                        i = 0
                    item = Datum(impath=impath, label=label2dataset[int(folder)], label_negative=-1, classname=classname)
                items.append(item)

        print(f"Classes for {split_dir}: {classes_list}")

        if negative_classes is not None:
            return items, negative_classes_with_same_occurrences, negative_label_ids
        return items
    
    def create_dataset_from_labels_only(self, classnames, class_ids, num_classes_train_val):
        items = []
        default_impath = ""

        for i, classname in enumerate(classnames):
            item = Datum(impath=default_impath, label=(num_classes_train_val + class_ids[i]), classname=classname, impath_exists=False)
            items.append(item)
        
        return items