import os
import pickle
from collections import OrderedDict

from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden

from .kitti360_labels import kitti360_labels
from .oxford_pets import OxfordPets


@DATASET_REGISTRY.register()
class KITTI360(DatasetBase):

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT))
        self.dataset_dir = "kitti360"
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = self.dataset_dir # os.path.join(self.dataset_dir, "images")

        classnames = [x.name for x in kitti360_labels[:-1]]
        
        train_test_classnames, label2dataset = self.get_indices("kitti_360_train", "kitti_360_val", classnames)
        
        train = self.read_data(classnames, "kitti_360_train", label2dataset)
        # Follow standard practice to perform evaluation on the val set
        # Also used as the val set (so evaluate the last-step model)
        test = self.read_data(classnames, "kitti_360_val", label2dataset)

        subsample = cfg.DATASET.SUBSAMPLE_CLASSES
        train, test = OxfordPets.subsample_classes(train, test, subsample=subsample)

        print(f"All Classes: {train_test_classnames}")

        super().__init__(train_x=train, train_u=test, val=test, test=test, classnames=classnames)

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for folder, line in enumerate(lines):
                classnames[folder] = line.strip()

        return classnames

    def get_indices(self, train_dir, test_dir, classnames, train_classes_num=0, test_classes_num=0):
        train_dir = os.path.join(self.image_dir, train_dir)
        train_folders = sorted(f.name for f in os.scandir(train_dir) if f.is_dir())
        if train_classes_num > 0:
            train_folders = train_folders[:train_classes_num]

        test_dir = os.path.join(self.image_dir, test_dir)
        test_folders = sorted(f.name for f in os.scandir(test_dir) if f.is_dir())
        if test_classes_num > 0:
            test_folders = test_folders[:test_classes_num]

        train_indices = [int(folder) for folder in train_folders]
        test_indices = [int(folder) for folder in test_folders]

        print(f"Train indices w/ Dataset ID: {train_indices}")
        print(f"Test indices w/ Dataset ID: {test_indices}")

        train_test_classnames = []
        label2dataset = {}
        common_ids =[]
        for key, class_name in enumerate(classnames):
            if key in train_indices or key in test_indices:
                train_test_classnames.append(class_name) 
                label2dataset[key] = key
                if key in train_indices and key in test_indices:
                    common_ids.append(key)
        
        print(f"Common ids: {common_ids}")

        return train_test_classnames, label2dataset

    def read_data(self, classnames, split_dir, label2dataset, num_classes=None):
        split_dir = os.path.join(self.image_dir, split_dir)
        folders = sorted(f.name for f in os.scandir(split_dir) if f.is_dir())

        if num_classes is not None:
            folders = folders[:num_classes]

        items = []

        print(f"Reading data from {split_dir}")

        for _, folder in enumerate(folders):
            imnames = listdir_nohidden(os.path.join(split_dir, folder))
            if int(folder) not in label2dataset:
                continue
            classname = classnames[int(folder)]
            for imname in imnames:
                impath = os.path.join(split_dir, folder, imname)
                item = Datum(impath=impath, label=label2dataset[int(folder)], classname=classname)
                items.append(item)

        return items