import os
import pickle
from collections import OrderedDict

from .negative_database import SEMI_HARD_NEGATIVES_LIST

from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden, mkdir_if_missing

from .oxford_pets import OxfordPets


@DATASET_REGISTRY.register()
class ScannetppNegativeSemiHardList(DatasetBase):

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT))
        self.dataset_dir = "scannetpp"
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = self.dataset_dir
        self.preprocessed = os.path.join(self.dataset_dir, "preprocessed.pkl")
        self.split_fewshot_dir = os.path.join(self.dataset_dir, "split_fewshot_scannetpp")
        mkdir_if_missing(self.split_fewshot_dir)

        text_file = os.path.join(self.dataset_dir, "scannetpp_semantic_classes.txt")
        
        classnames = self.read_classnames(text_file)
        
        train_test_classnames, train_indices, test_indices, label2dataset = self.get_positives_negatives("scannetpp_sem_seg_train", "scannetpp_sem_seg_val", classnames)
        negative_classnames = SEMI_HARD_NEGATIVES_LIST
        all_classnames = train_test_classnames + negative_classnames
        
        train = self.read_data(classnames, train_indices, "scannetpp_sem_seg_train", label2dataset, split_type="train") # , num_classes=20
        test = self.read_data(classnames, test_indices, "scannetpp_sem_seg_val", label2dataset, split_type="val") # , num_classes=40

        train_u = self.get_negatives(negative_classnames, cur_id=len(train_test_classnames))

        num_shots = cfg.DATASET.NUM_SHOTS        
        if num_shots >= 1:
            train = self.generate_fewshot_dataset(train, num_shots=num_shots)

        subsample = cfg.DATASET.SUBSAMPLE_CLASSES
        train, train_u, test = OxfordPets.subsample_classes(train, train_u, test, subsample=subsample)

        super().__init__(train_x=train, train_u=train_u, val=test, test=test, classnames=all_classnames)

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for folder, line in enumerate(lines):
                classnames[folder] = line.strip()

        return classnames

    
    def get_positives_negatives(self, train_dir, test_dir, classnames):
        train_dir = os.path.join(self.image_dir, train_dir)
        train_folders = sorted(f.name for f in os.scandir(train_dir) if f.is_dir())

        test_dir = os.path.join(self.image_dir, test_dir)
        test_folders = sorted(f.name for f in os.scandir(test_dir) if f.is_dir())

        train_indices = [int(folder) for folder in train_folders]
        test_indices = [int(folder) for folder in test_folders]

        train_test_classnames = []
        label2dataset = {}
        cur_id = 0
        for key in classnames:
            if key in train_indices or key in test_indices:
                train_test_classnames.append(classnames[key]) 
                label2dataset[key] = cur_id
                cur_id += 1

        return train_test_classnames, train_indices, test_indices, label2dataset
    
    def get_negatives(self, negative_classes, cur_id):
        items = []
        
        for classname in negative_classes:
            item = Datum(impath="", label=cur_id, classname=classname, impath_exists=False)
            items.append(item)
            cur_id += 1

        print(f"number of negative classes {len(negative_classes)}")
        return items

    def read_data(self, classnames, folder_ids, split_dir, label2dataset, split_type="train"):
        split_dir = os.path.join(self.image_dir, split_dir)

        items = []

        print(f"Reading data from {split_dir}")

        classes_list = []
        for _, folder in enumerate(folder_ids):
            imnames = listdir_nohidden(os.path.join(split_dir, str(folder)))
            classname = classnames[int(folder)]
            classes_list.append(classname)
            for imname in imnames:
                impath = os.path.join(split_dir, str(folder), imname)
                item = Datum(impath=impath, label=label2dataset[int(folder)], classname=classname)
                items.append(item)

        print(f"Classes for {split_type}: {classes_list}")

        return items