import os
import pickle
from collections import OrderedDict

from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden, mkdir_if_missing

from .oxford_pets import OxfordPets


@DATASET_REGISTRY.register()
class ScannetppVal(DatasetBase):

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT))
        self.dataset_dir = "scannetpp"
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = self.dataset_dir # os.path.join(self.dataset_dir, "images")
        self.preprocessed = os.path.join(self.dataset_dir, "preprocessed.pkl")
        self.split_fewshot_dir = os.path.join(self.dataset_dir, "split_fewshot_scannetpp")
        mkdir_if_missing(self.split_fewshot_dir)

        text_file = os.path.join(self.dataset_dir, "scannetpp_semantic_classes.txt")
        
        classnames = self.read_classnames(text_file)
        
        train_test_classnames, label2dataset = self.get_indices("scannetpp_sem_seg_val", classnames)
        
        # Follow standard practice to perform evaluation on the val set
        # Also used as the val set (so evaluate the last-step model)
        test = self.read_data(classnames, "scannetpp_sem_seg_val", label2dataset) # , num_classes=40

        num_shots = cfg.DATASET.NUM_SHOTS
        if num_shots >= 1:
            train = self.generate_fewshot_dataset(train, num_shots=num_shots)

        print(f"All Classes: {train_test_classnames}")

        super().__init__(train_x=test, train_u=test, val=test, test=test, classnames=train_test_classnames)

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for folder, line in enumerate(lines):
                classnames[folder] = line.strip()

        return classnames

    def get_indices(self, test_dir, classnames, train_classes_num=0, test_classes_num=0):
        test_dir = os.path.join(self.image_dir, test_dir)
        test_folders = sorted(f.name for f in os.scandir(test_dir) if f.is_dir())
        if test_classes_num > 0:
            test_folders = test_folders[:test_classes_num]

        test_indices = [int(folder) for folder in test_folders]

        print(f"Test indices w/ Dataset ID: {test_indices}")

        train_test_classnames = []
        label2dataset = {}
        cur_id = 0
        for key in classnames:
            if key in test_indices:
                train_test_classnames.append(classnames[key]) 
                label2dataset[key] = cur_id
                cur_id += 1

        return train_test_classnames, label2dataset

    def read_data(self, classnames, split_dir, label2dataset):
        split_dir = os.path.join(self.image_dir, split_dir)
        folders = sorted(f.name for f in os.scandir(split_dir) if f.is_dir())

        items = []

        print(f"Reading data from {split_dir}")

        for _, folder in enumerate(folders):
            imnames = listdir_nohidden(os.path.join(split_dir, folder))
            if int(folder) not in label2dataset:
                continue
            classname = classnames[int(folder)]
            for imname in imnames:
                impath = os.path.join(split_dir, folder, imname)
                item = Datum(impath=impath, label=label2dataset[int(folder)], classname=classname)
                items.append(item)

        return items