import os.path as osp
import copy
import torch
import logging
import torch.nn as nn
from torch.nn import functional as F
from torch.cuda.amp import GradScaler, autocast

from .dassl_utils import load_checkpoint

from .clip import _MODELS, _download, tokenize
from .clip_model import build_clip_model
from .clip_simple_tokenizer import SimpleTokenizer as _Tokenizer

_tokenizer = _Tokenizer()

def load_clip_to_cpu(cfg):
    backbone_name = cfg.MODEL.CLIP_ADAPTER.CLIP_MODEL_NAME
    url = _MODELS[backbone_name]
    model_path = _download(url)

    try:
        # loading JIT archive
        model = torch.jit.load(model_path, map_location="cpu").eval()
        state_dict = None

    except RuntimeError:
        state_dict = torch.load(model_path, map_location="cpu")
    design_details = {"trainer": 'IVLP',
                      "vision_depth": cfg.MODEL.OPENDAS.PROMPT_DEPTH_VISION,
                      "language_depth": cfg.MODEL.OPENDAS.PROMPT_DEPTH_TEXT, "vision_ctx": cfg.MODEL.OPENDAS.N_CTX_VISION,
                      "language_ctx": cfg.MODEL.OPENDAS.N_CTX_TEXT}
    model = build_clip_model(state_dict or model.state_dict(), design_details)

    return model


class TextEncoder(nn.Module):
    def __init__(self, clip_model):
        super().__init__()
        self.transformer = clip_model.transformer
        self.positional_embedding = clip_model.positional_embedding
        self.ln_final = clip_model.ln_final
        self.text_projection = clip_model.text_projection
        self.dtype = clip_model.dtype

    def forward(self, prompts, tokenized_prompts):
        x = prompts + self.positional_embedding.type(self.dtype)
        x = x.permute(1, 0, 2)  # NLD -> LND
        x = self.transformer(x)
        x = x.permute(1, 0, 2)  # LND -> NLD
        x = self.ln_final(x).type(self.dtype)

        # x.shape = [batch_size, n_ctx, transformer.width]
        # take features from the eot embedding (eot_token is the highest number in each sequence)
        x = x[torch.arange(x.shape[0]), tokenized_prompts.argmax(dim=-1)] @ self.text_projection

        return x


class VLPromptLearner(nn.Module):
    def __init__(self, cfg, classnames, clip_model):
        super().__init__()
        n_cls = len(classnames)
        # Make sure Language depth >= 1
        assert cfg.MODEL.OPENDAS.PROMPT_DEPTH_TEXT >= 1
        n_ctx = cfg.MODEL.OPENDAS.N_CTX_TEXT
        ctx_init = cfg.MODEL.OPENDAS.CTX_INIT
        dtype = clip_model.dtype
        ctx_dim = clip_model.ln_final.weight.shape[0]
        vis_dim = clip_model.visual.output_dim
        clip_imsize = clip_model.visual.input_resolution
        cfg_imsize = cfg.MODEL.OPENDAS.INPUT_SIZE[0]
        assert cfg_imsize == clip_imsize, f"cfg_imsize ({cfg_imsize}) must equal to clip_imsize ({clip_imsize})"

        if ctx_init and (n_ctx) <= 4:
            # use given words to initialize context vectors
            ctx_init = ctx_init.replace("_", " ")
            n_ctx = n_ctx
            prompt = tokenize(ctx_init)
            with torch.no_grad():
                embedding = clip_model.token_embedding(prompt).type(dtype)
            ctx_vectors = embedding[0, 1: 1 + n_ctx, :]
            prompt_prefix = ctx_init
        else:
            # random initialization
            ctx_vectors = torch.empty(n_ctx, ctx_dim, dtype=dtype)
            nn.init.normal_(ctx_vectors, std=0.02)
            prompt_prefix = " ".join(["X"] * n_ctx)
        logging.info(f"Visual Language Prompt Tuning (OPENDAS)")
        logging.info(f'Initial text context: "{prompt_prefix}"')
        logging.info(f"Number of context words (tokens) for language prompting: {n_ctx}")
        logging.info(f"Number of context words (tokens) for vision prompting: {cfg.MODEL.OPENDAS.N_CTX_VISION}")
        self.ctx = nn.Parameter(ctx_vectors)

        classnames = [name.replace("_", " ") for name in classnames]
        name_lens = [len(_tokenizer.encode(name)) for name in classnames]
        prompts = [prompt_prefix + " " + name + "." for name in classnames]

        tokenized_prompts = torch.cat([tokenize(p) for p in prompts])  # (n_cls, n_tkn)
        with torch.no_grad():
            embedding = clip_model.token_embedding(tokenized_prompts).type(dtype)

        # These token vectors will be saved when in save_model(),
        # but they should be ignored in load_model() as we want to use
        # those computed using the current class names
        self.register_buffer("token_prefix", embedding[:, :1, :])  # SOS
        self.register_buffer("token_suffix", embedding[:, 1 + n_ctx:, :])  # CLS, EOS

        self.n_cls = n_cls
        self.n_ctx = n_ctx
        self.tokenized_prompts = tokenized_prompts  # torch.Tensor
        self.name_lens = name_lens

    def construct_prompts(self, ctx, prefix, suffix, label=None):
        # dim0 is either batch_size (during training) or n_cls (during testing)
        # ctx: context tokens, with shape of (dim0, n_ctx, ctx_dim)
        # prefix: the sos token, with shape of (n_cls, 1, ctx_dim)
        # suffix: remaining tokens, with shape of (n_cls, *, ctx_dim)

        if label is not None:
            prefix = prefix[label]
            suffix = suffix[label]

        prompts = torch.cat(
            [
                prefix,  # (dim0, 1, dim)
                ctx,  # (dim0, n_ctx, dim)
                suffix,  # (dim0, *, dim)
            ],
            dim=1,
        )

        return prompts

    def forward(self):
        ctx = self.ctx
        if ctx.dim() == 2:
            ctx = ctx.unsqueeze(0).expand(self.n_cls, -1, -1)

        prefix = self.token_prefix
        suffix = self.token_suffix
        prompts = self.construct_prompts(ctx, prefix, suffix)

        return prompts


class CustomCLIP_OpenDAS(nn.Module):
    def __init__(self, cfg, classnames, clip_model):
        super().__init__()
        self.prompt_learner = VLPromptLearner(cfg, classnames, clip_model)
        self.tokenized_prompts = self.prompt_learner.tokenized_prompts
        self.image_encoder = clip_model.visual
        self.text_encoder = TextEncoder(clip_model)
        self.logit_scale = clip_model.logit_scale
        self.dtype = clip_model.dtype
        self.margin = 1.5
        logging.info(f"MARGIN: {self.margin}")
        self.triplet_loss = nn.TripletMarginLoss(margin=self.margin, p=2, eps=1e-7)

    def forward(self, image, label=None, label_u=None, use_triplet=False, use_both=False, triplet_weight=0):
        tokenized_prompts = self.tokenized_prompts
        logit_scale = self.logit_scale.exp()

        prompts = self.prompt_learner()
        text_features = self.text_encoder(prompts, tokenized_prompts)
        image_features = self.image_encoder(image.type(self.dtype))

        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        text_features = text_features / text_features.norm(dim=-1, keepdim=True)
        logits = logit_scale * image_features @ text_features.t()

        if self.prompt_learner.training:
            if use_triplet:
                text_features_list = []
                text_features_u_list = []
                for lbl_x, lbl_u in zip(label, label_u):
                    text_features_list.append(text_features[lbl_x])
                    text_features_u_list.append(text_features[lbl_u])
                text_features_list = torch.stack(text_features_list)
                text_features_u_list = torch.stack(text_features_u_list)

                return self.triplet_loss(image_features, text_features_list, text_features_u_list)
            elif use_both:
                text_features_list = []
                text_features_u_list = []
                for img_f, lbl_x, lbl_u in zip(image_features, label, label_u):
                    # remove x from text features 
                    text_features_without_x = torch.cat([text_features[0:lbl_x], text_features[lbl_x+1:]])
                    # find the most similar item (it cannot be same as lbl_x)
                    lbl_similar = torch.norm(img_f - text_features_without_x, dim=1).argmin()
                    text_features_list.append(text_features[lbl_x])
                    text_features_u_list.append(text_features_without_x[lbl_similar])
                    # text_features_u_list.append(text_features[lbl_u])
                text_features_list = torch.stack(text_features_list)
                text_features_u_list = torch.stack(text_features_u_list)

                return triplet_weight * self.triplet_loss(image_features, text_features_list, text_features_u_list) + F.cross_entropy(logits, label)
            else:
                return F.cross_entropy(logits, label)

        return logits
    
    def get_text_features(self, labels):
        tokenized_prompts = self.tokenized_prompts
        prompts = self.prompt_learner()
        text_features = self.text_encoder(prompts, tokenized_prompts)
        text_features = text_features / text_features.norm(dim=-1, keepdim=True)

        text_features_for_labels = []
        for l in labels:
            text_features_for_labels.append(text_features[l])
        text_features_for_labels = torch.stack(text_features_for_labels)

        return text_features_for_labels
    
    def get_image_features(self, image):
        image_features = self.image_encoder(image.type(self.dtype)).float()
        image_features = image_features / image_features.norm(dim=-1, keepdim=True)
        return image_features

def build_model(cfg, class_names, clip_model):
    logging.info("Building custom CLIP")
    model = CustomCLIP_OpenDAS(cfg, class_names, clip_model)

    logging.info("Turning off all gradients in CustomCLIP_OpenDAS")
    for name, param in model.named_parameters():
       param.requires_grad_(False)

    # Double check
    enabled = set()

    for name, param in model.named_parameters():
        if param.requires_grad:
            enabled.add(name)
    
    logging.info(f"Parameters to be updated: {enabled}")

    # Count the number of parameters to be updated
    num_params = sum(p.numel() for p in model.parameters() if p.requires_grad)
    logging.info(f"Number of parameters to be updated: {num_params}")

    # model.to(device)

    # Note that multi-gpu training could be slow because CLIP's size is
    # big, which slows down the copy operation in DataParallel
    device_count = torch.cuda.device_count()
    if device_count > 1:
        # logging.info(f"Multiple GPUs detected (n_gpus={device_count}), use all of them!")
        logging.info(f"Multi-GPU is not recommended!!!!1")
        # model = nn.DataParallel(model)
        raise NotImplementedError

    return model

def load_model(model, cfg):
    directory = cfg.MODEL.OPENDAS.DIR
    epoch = cfg.MODEL.OPENDAS.LOAD_EPOCH
    logging.info(f"Loading model from {directory}")
    if not directory:
        logging.info("Note that load_model() is skipped as no pretrained model is given")
        raise FileNotFoundError()

    # By default, the best model is loaded
    model_file = "model-best.pth.tar"

    if epoch is not None:
        model_file = "model.pth.tar-" + str(epoch)

    for name in ["VLPromptLearner"]:
        model_path = osp.join(directory, name, model_file)

        if not osp.exists(model_path):
            raise FileNotFoundError('Model not found at "{}"'.format(model_path))

        checkpoint = load_checkpoint(model_path)
        state_dict = checkpoint["state_dict"]
        epoch = checkpoint["epoch"]

        # Ignore fixed token vectors
        if "prompt_learner.token_prefix" in state_dict:
            del state_dict["prompt_learner.token_prefix"]

        if "prompt_learner.token_suffix" in state_dict:
            del state_dict["prompt_learner.token_suffix"]

        logging.info("Loading weights to {} " 'from "{}" (epoch = {})'.format(name, model_path, epoch))
        # set strict=False
        model.load_state_dict(state_dict, strict=False)
    
    return model
