"""Main file."""
import os
import uuid
from absl import app
from absl import flags
from absl import logging

# Required import to setup work units when running through XManager.
from clu import platform
import jax
from ml_collections import config_flags
import tensorflow as tf
import wandb

from ebm_obj.lib import train

FLAGS = flags.FLAGS

config_flags.DEFINE_config_file(
    "config", None, "Training configuration.", lock_config=True)
flags.DEFINE_string("workdir", None, "Work unit directory.")
flags.mark_flags_as_required(["config", "workdir"])
# Flags --jax_backend_target and --jax_xla_backend are available through JAX.


def main(argv):
  del argv
  if FLAGS.config.get("workdir", None):
    FLAGS.workdir = FLAGS.config.workdir
  else:
    FLAGS.workdir = os.path.join(FLAGS.workdir, uuid.uuid4().hex.upper()[0:6])
    FLAGS.config.workdir = FLAGS.workdir
  logging.info("Setting workdir to %s", FLAGS.workdir)
  wandb.init()
  wandb.tensorboard.patch(root_logdir=FLAGS.workdir)
  wandb.config.update(FLAGS.config.to_dict())
  # Hide any GPUs form TensorFlow. Otherwise TF might reserve memory and make
  # it unavailable to JAX.
  tf.config.experimental.set_visible_devices([], "GPU")
  if FLAGS.jax_backend_target:
    logging.info("Using JAX backend target %s", FLAGS.jax_backend_target)
    jax_xla_backend = ("None" if FLAGS.jax_xla_backend is None else
                       FLAGS.jax_xla_backend)
    logging.info("Using JAX XLA backend %s", jax_xla_backend)

  logging.info("JAX process: %d / %d", jax.process_index(), jax.process_count())
  logging.info("JAX devices: %r", jax.devices())

  platform.work_unit().set_task_status(f"process_index: {jax.process_index()}, "
                                       f"process_count: {jax.process_count()}")
  platform.work_unit().create_artifact(platform.ArtifactType.DIRECTORY,
                                       FLAGS.workdir, "workdir")

  train.train_and_evaluate(FLAGS.config, FLAGS.workdir)

  wandb.finish()

if __name__ == "__main__":
  # Provide access to --jax_backend_target and --jax_xla_backend flags.
  jax.config.config_with_absl()
  app.run(main)