#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

import os
import sys
from PIL import Image
from typing import NamedTuple
from scene.colmap_loader import read_extrinsics_text, read_intrinsics_text, qvec2rotmat, \
    read_extrinsics_binary, read_intrinsics_binary, read_points3D_binary, read_points3D_text
from utils.graphics_utils import getWorld2View2, focal2fov, fov2focal
import numpy as np
import json
from pathlib import Path
from plyfile import PlyData, PlyElement
from utils.sh_utils import SH2RGB
from scene.gaussian_model import BasicPointCloud
from tqdm import tqdm
import torch
from utils.general_utils import fps
from multiprocessing.pool import ThreadPool
import imagesize
from loguru import logger
import cv2

class CameraInfo(NamedTuple):
    # #------------------- Pose Optimization ---------------------# #
    camera_id: int
    # #-----------------------------------------------------------# #
    uid: int
    R: np.array
    T: np.array
    FovY: np.array
    FovX: np.array
    image: np.array
    depth: np.array
    flow: np.array
    image_path: str
    image_name: str
    width: int
    height: int
    timestamp: float = 0.0
    pseudo_timestamp: float = 0.0
    fl_x: float = -1.0
    fl_y: float = -1.0
    cx: float = -1.0
    cy: float = -1.0
    # # ------------------ Flow Mask Test -------------- # #
    is_true_image: bool = True
    flow_mask: np.array = None
    # #------------------------------------------------- # #
    # #------------------- Motion Mask Test ------------ # #
    motion_mask_path: str = None
    # #------------------------------------------------- # #
    temporal_coord_only: bool = False
    spatial_coord_only: bool = False

class SceneInfo(NamedTuple):
    point_cloud: BasicPointCloud
    train_cameras: list
    test_cameras: list
    nerf_normalization: dict
    ply_path: str

def getNerfppNorm(cam_info):
    def get_center_and_diag(cam_centers):
        cam_centers = np.hstack(cam_centers)
        avg_cam_center = np.mean(cam_centers, axis=1, keepdims=True)
        center = avg_cam_center
        dist = np.linalg.norm(cam_centers - center, axis=0, keepdims=True)
        diagonal = np.max(dist)
        return center.flatten(), diagonal

    cam_centers = []

    for cam in cam_info:
        W2C = getWorld2View2(cam.R, cam.T)
        C2W = np.linalg.inv(W2C)
        cam_centers.append(C2W[:3, 3:4])

    center, diagonal = get_center_and_diag(cam_centers)
    radius = diagonal * 1.1

    translate = -center

    return {"translate": translate, "radius": radius}

def readColmapCameras(cam_extrinsics, cam_intrinsics, images_folder):
    cam_infos = []
    for idx, key in enumerate(cam_extrinsics):
        sys.stdout.write('\r')
        # the exact output you're looking for:
        sys.stdout.write("Reading camera {}/{}".format(idx+1, len(cam_extrinsics)))
        sys.stdout.flush()

        extr = cam_extrinsics[key]
        intr = cam_intrinsics[extr.camera_id]
        height = intr.height
        width = intr.width

        uid = intr.id
        R = np.transpose(qvec2rotmat(extr.qvec))
        T = np.array(extr.tvec)

        if intr.model=="SIMPLE_PINHOLE" or intr.model=="SIMPLE_RADIAL":
            focal_length_x = intr.params[0]
            FovY = focal2fov(focal_length_x, height)
            FovX = focal2fov(focal_length_x, width)
        elif intr.model=="PINHOLE":
            focal_length_x = intr.params[0]
            focal_length_y = intr.params[1]
            FovY = focal2fov(focal_length_y, height)
            FovX = focal2fov(focal_length_x, width)
        else:
            assert False, "Colmap camera model not handled: only undistorted datasets (PINHOLE or SIMPLE_PINHOLE or SIMPLE_RADIAL cameras) supported!"

        image_path = os.path.join(images_folder, os.path.basename(extr.name))
        image_name = os.path.basename(image_path).split(".")[0]
        image = Image.open(image_path)

        cam_info = CameraInfo(uid=uid, R=R, T=T, FovY=FovY, FovX=FovX, image=image,
                              image_path=image_path, image_name=image_name, width=width, height=height)
        cam_infos.append(cam_info)
    sys.stdout.write('\n')
    return cam_infos

def fetchPly(path, frame_ratio = 1.0):
    plydata = PlyData.read(path)
    vertices = plydata['vertex']
    positions = np.vstack([vertices['x'], vertices['y'], vertices['z']]).T
    colors = np.vstack([vertices['red'], vertices['green'], vertices['blue']]).T / 255.0
    if 'nx' in vertices:
        normals = np.vstack([vertices['nx'], vertices['ny'], vertices['nz']]).T
    else:
        normals = np.zeros_like(positions)
    if 'time' in vertices:
        timestamp = np.array(vertices['time'][:, None]) / frame_ratio
    else:
        timestamp = None

    return BasicPointCloud(points=positions, colors=colors, normals=normals, time=timestamp)

def storePly(path, xyz, rgb):
    # Define the dtype for the structured array
    dtype = [('x', 'f4'), ('y', 'f4'), ('z', 'f4'),
            ('nx', 'f4'), ('ny', 'f4'), ('nz', 'f4'),
            ('red', 'u1'), ('green', 'u1'), ('blue', 'u1')]

    normals = np.zeros_like(xyz)

    elements = np.empty(xyz.shape[0], dtype=dtype)
    attributes = np.concatenate((xyz, normals, rgb), axis=1)
    elements[:] = list(map(tuple, attributes))

    # Create the PlyData object and write to file
    vertex_element = PlyElement.describe(elements, 'vertex')
    ply_data = PlyData([vertex_element])
    ply_data.write(path)

def readColmapSceneInfo(path, images, eval, llffhold=8, num_pts_ratio=1.0):
    try:
        cameras_extrinsic_file = os.path.join(path, "sparse/0", "images.bin")
        cameras_intrinsic_file = os.path.join(path, "sparse/0", "cameras.bin")
        cam_extrinsics = read_extrinsics_binary(cameras_extrinsic_file)
        cam_intrinsics = read_intrinsics_binary(cameras_intrinsic_file)
    except:
        cameras_extrinsic_file = os.path.join(path, "sparse/0", "images.txt")
        cameras_intrinsic_file = os.path.join(path, "sparse/0", "cameras.txt")
        cam_extrinsics = read_extrinsics_text(cameras_extrinsic_file)
        cam_intrinsics = read_intrinsics_text(cameras_intrinsic_file)

    reading_dir = "images" if images == None else images
    cam_infos_unsorted = readColmapCameras(cam_extrinsics=cam_extrinsics, cam_intrinsics=cam_intrinsics, images_folder=os.path.join(path, reading_dir))
    cam_infos = sorted(cam_infos_unsorted.copy(), key = lambda x : x.image_name)

    if eval:
        train_cam_infos = [c for idx, c in enumerate(cam_infos) if idx % llffhold != 0]
        test_cam_infos = [c for idx, c in enumerate(cam_infos) if idx % llffhold == 0]
    else:
        train_cam_infos = cam_infos
        test_cam_infos = []

    nerf_normalization = getNerfppNorm(train_cam_infos)

    ply_path = os.path.join(path, "sparse/0/points3D.ply")
    bin_path = os.path.join(path, "sparse/0/points3D.bin")
    txt_path = os.path.join(path, "sparse/0/points3D.txt")
    if not os.path.exists(ply_path):
        print("Converting point3d.bin to .ply, will happen only the first time you open the scene.")
        try:
            xyz, rgb, _ = read_points3D_binary(bin_path)
        except:
            xyz, rgb, _ = read_points3D_text(txt_path)
        storePly(ply_path, xyz, rgb)
    try:
        pcd = fetchPly(ply_path)
    except:
        pcd = None
    if num_pts_ratio > 1.001:
        num_pts = int((num_pts_ratio - 1) * pcd.points.shape[0])
        mean_xyz = pcd.points.mean(axis=0)
        min_rand_xyz = mean_xyz - np.array([0.5, 0.5, 0.5])
        max_rand_xyz = mean_xyz + np.array([0.5, 2.0, 0.5])
        xyz = np.concatenate([pcd.points,
                              np.random.random((num_pts, 3)) * (max_rand_xyz - min_rand_xyz) + min_rand_xyz],
                              axis=0)
        colors = np.concatenate([pcd.colors,
                              SH2RGB(np.random.random((num_pts, 3)) / 255.0)],
                              axis=0)
        normals = np.concatenate([pcd.normals,
                              np.zeros((num_pts, 3))],
                              axis=0)
        pcd = BasicPointCloud(points=xyz, colors=colors, normals=normals)

    scene_info = SceneInfo(point_cloud=pcd,
                           train_cameras=train_cam_infos,
                           test_cameras=test_cam_infos,
                           nerf_normalization=nerf_normalization,
                           ply_path=ply_path)
    return scene_info

def readCamerasFromTransforms(path, transformsfile, white_background, extension=".png", time_duration=None, frame_ratio=1, dataloader=False, target_cam=[]):
    cam_infos = []

    with open(os.path.join(path, transformsfile)) as json_file:
        contents = json.load(json_file)
    if "camera_angle_x" in contents:
        fovx = contents["camera_angle_x"]

    frames = contents["frames"]
    tbar = tqdm(range(len(frames)))
    def frame_read_fn(idx_frame):
        idx = idx_frame[0]
        frame = idx_frame[1]
        timestamp = frame.get('time', 0.0) / frame_ratio
        if time_duration is not None and 'time' in frame:
            if timestamp < time_duration[0] or timestamp > time_duration[1]:
                return

        if len(target_cam) > 0:
            cam = os.path.basename(frame["file_path"]).split("_")[0]
            if not cam in target_cam:
                return
        cam_name = os.path.join(path, frame["file_path"] + extension)

        # NeRF 'transform_matrix' is a camera-to-world transform
        c2w = np.array(frame["transform_matrix"])
        # change from OpenGL/Blender camera axes (Y up, Z back) to COLMAP (Y down, Z forward)
        c2w[:3, 1:3] *= -1

        # get the world-to-camera transform and set R, T
        w2c = np.linalg.inv(c2w)
        R = np.transpose(w2c[:3,:3])  # R is stored transposed due to 'glm' in CUDA code
        T = w2c[:3, 3]

        image_path = os.path.join(path, cam_name) # .replace('hdImgs_unditorted', 'hdImgs_unditorted_rgba').replace('.jpg', '.png')
        image_name = Path(cam_name).stem

        if not dataloader:
            with Image.open(image_path) as image_load:
                im_data = np.array(image_load.convert("RGBA"))

            bg = np.array([1,1,1]) if white_background else np.array([0, 0, 0])

            norm_data = im_data / 255.0
            arr = norm_data[:,:,:3] * norm_data[:, :, 3:4] + bg * (1 - norm_data[:, :, 3:4])
            if norm_data[:, :, 3:4].min() < 1:
                arr = np.concatenate([arr, norm_data[:, :, 3:4]], axis=2)
                image = Image.fromarray(np.array(arr*255.0, dtype=np.byte), "RGBA")
            else:
                image = Image.fromarray(np.array(arr*255.0, dtype=np.byte), "RGB")

            width, height = image.size[0], image.size[1]
        else:
            image = np.empty(0)
            width, height = imagesize.get(image_path)

        depth=None
        flow=None
        if 'depth_path' in frame:
            depth_name = frame["depth_path"]
            if not extension in frame["depth_path"]:
                depth_name = frame["depth_path"] + extension
            depth_path = os.path.join(path, depth_name)
            depth = Image.open(depth_path).copy()
        tbar.update(1)
        if 'fl_x' in frame and 'fl_y' in frame and 'cx' in frame and 'cy' in frame:
            FovX = FovY = -1.0
            fl_x = frame['fl_x']
            fl_y = frame['fl_y']
            cx = frame['cx']
            cy = frame['cy']
            return CameraInfo(uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                        image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp,
                        fl_x=fl_x, fl_y=fl_y, cx=cx, cy=cy)
        elif 'fl_x' in contents and 'fl_y' in contents and 'cx' in contents and 'cy' in contents:
            FovX = FovY = -1.0
            fl_x = contents['fl_x']
            fl_y = contents['fl_y']
            cx = contents['cx']
            cy = contents['cy']
            return CameraInfo(uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                        image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp,
                        fl_x=fl_x, fl_y=fl_y, cx=cx, cy=cy)
        else:
            fovy = focal2fov(fov2focal(fovx, width), height)
            FovY = fovy
            FovX = fovx
            return CameraInfo(uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                            image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp)

    with ThreadPool() as pool:
        cam_infos = pool.map(frame_read_fn, zip(list(range(len(frames))), frames))
        pool.close()
        pool.join()

    cam_infos = [cam_info for cam_info in cam_infos if cam_info is not None]

    return cam_infos

def readNerfSyntheticInfo(path, white_background, eval, extension=".png", num_pts=100_000, time_duration=None, num_extra_pts=0, frame_ratio=1, dataloader=False, target_cam=[]):

    print("Reading Training Transforms")
    train_cam_infos = readCamerasFromTransforms(path, "transforms_train.json", white_background, extension,
                                                time_duration=time_duration, frame_ratio=frame_ratio, dataloader=dataloader,
                                                target_cam=target_cam)
    print("Reading Test Transforms")
    test_cam_infos = readCamerasFromTransforms(path, "transforms_test.json" if not path.endswith('lego') else "transforms_val.json", white_background, extension, time_duration=time_duration, frame_ratio=frame_ratio, dataloader=dataloader)

    if not eval:
        train_cam_infos.extend(test_cam_infos)
        test_cam_infos = []

    print(f"{len(train_cam_infos)} Training cameras, {len(test_cam_infos)} Test cameras loaded")
    nerf_normalization = getNerfppNorm(train_cam_infos)

    ply_path = os.path.join(path, "points3d.ply")
    if not os.path.exists(ply_path):
        # Since this data set has no colmap data, we start with random points
        print(f"Generating random point cloud ({num_pts})...")

        # We create random points inside the bounds of the synthetic Blender scenes
        xyz = np.random.random((num_pts, 3)) * 2.6 - 1.3
        shs = np.random.random((num_pts, 3)) / 255.0
        pcd = BasicPointCloud(points=xyz, colors=SH2RGB(shs), normals=np.zeros((num_pts, 3)))

        storePly(ply_path, xyz, SH2RGB(shs) * 255)
    try:
        pcd = fetchPly(ply_path)
    except:
        pcd = None

    if pcd.points.shape[0] > num_pts:
        mask = np.random.randint(0, pcd.points.shape[0], num_pts)
        # mask = fps(torch.from_numpy(pcd.points).cuda()[None], num_pts).cpu().numpy()
        if pcd.time is not None:
            times = pcd.time[mask]
        else:
            times = None
        xyz = pcd.points[mask]
        rgb = pcd.colors[mask]
        normals = pcd.normals[mask]

        # 计算点云中心
        center = np.mean(xyz, axis=0)
        # 计算每个点到中心的距离
        distances = np.linalg.norm(xyz - center, axis=1)
        # 计算距离的均值和标准差
        mean_dist = np.mean(distances)
        std_dist = np.std(distances)
        # 设置阈值(例如:均值+3倍标准差)
        threshold = mean_dist + 3 * std_dist
        # 剔除距离过大的点
        valid_mask = distances < threshold
        xyz = xyz[valid_mask]
        rgb = rgb[valid_mask]
        normals = normals[valid_mask]

        if times is not None:
            time_mask = (times[:,0] < time_duration[1]) & (times[:,0] > time_duration[0])
            xyz = xyz[time_mask]
            rgb = rgb[time_mask]
            normals = normals[time_mask]
            times = times[time_mask]
        pcd = BasicPointCloud(points=xyz, colors=rgb, normals=normals, time=times)

    if num_extra_pts > 0:
        times = pcd.time
        xyz = pcd.points
        rgb = pcd.colors
        normals = pcd.normals
        bound_min, bound_max = xyz.min(0), xyz.max(0)
        radius = 60.0 # (bound_max - bound_min).mean() + 10
        phi = 2.0 * np.pi * np.random.rand(num_extra_pts)
        theta = np.arccos(2.0 * np.random.rand(num_extra_pts) - 1.0)
        x = radius * np.sin(theta) * np.cos(phi)
        y = radius * np.sin(theta) * np.sin(phi)
        z = radius * np.cos(theta)
        xyz_extra = np.stack([x, y, z], axis=1)
        normals_extra = np.zeros_like(xyz_extra)
        rgb_extra = np.ones((num_extra_pts, 3)) / 2

        xyz = np.concatenate([xyz, xyz_extra], axis=0)
        rgb = np.concatenate([rgb, rgb_extra], axis=0)
        normals = np.concatenate([normals, normals_extra], axis=0)

        if times is not None:
            times_extra = torch.zeros(((num_extra_pts, 3))) + (time_duration[0] + time_duration[1]) / 2
            times = np.concatenate([times, times_extra], axis=0)

        pcd = BasicPointCloud(points=xyz,
                              colors=rgb,
                              normals=normals,
                              time=times)

    scene_info = SceneInfo(point_cloud=pcd,
                           train_cameras=train_cam_infos,
                           test_cameras=test_cam_infos,
                           nerf_normalization=nerf_normalization,
                           ply_path=ply_path)
    return scene_info

def readCamerasFromTransforms2(path, transformsfile, white_background, extension=".png", time_duration=None, frame_ratio=1, dataloader=False,
                               target_cam=[], mode='train', scene_scale_ratio=1.0, frame_skip=8, time_skip=5, temporal_coord_only=False, spatial_coord_only=False):
    cam_infos = []

    with open(os.path.join(path, transformsfile)) as json_file:
        contents = json.load(json_file)
    if "camera_angle_x" in contents:
        fovx = contents["camera_angle_x"]
    if "frames" in contents:
        frames = contents["frames"]
    else:
        frames = contents
    tbar = tqdm(range(len(frames)))
    def frame_read_fn(idx_frame):
        idx = idx_frame[0]
        frame = idx_frame[1]
        timestamp = frame.get('time', 0.0) / frame_ratio
        pseudo_timestamp = frame.get('time_pseudo', 0.0) / frame_ratio
        is_true_image = bool(frame.get('is_true_image', True))
        if not is_true_image:
            return
        # # ------------------- Downsample Camera Poses ---------------------# #
        if mode=='train' and not is_true_image:
            frame_idx = int(frame["file_path"].split('/')[-1].split('.')[0].split('_')[-1]) # diffusion/frame_00000.png
            frame_idx = frame_idx - 1 if frame_idx > 240 else frame_idx
            if frame_idx % frame_skip:
                return
        # # ------------------- Downsample Time Step ------------------------# #
        # if time_skip > 0 and not is_true_image:
        #     tt = int(frame["file_path"].split('/')[-3].split('_')[-1])
        #     if tt % time_skip:
        #         return

        #--------------------------------------------------------------------# #
        if time_duration is not None and 'time' in frame:
            if timestamp < time_duration[0] or timestamp > time_duration[1]:
                return

        if len(target_cam) > 0:
            if is_true_image:
                cam = frame["file_path"].split('/')[-1].split('.')[0].split('_')[0]
            else:
                cam = frame["file_path"].split('/')[-3]
            if not cam in target_cam:
                return
        if not frame["file_path"].endswith(extension):
            image_name = frame["file_path"] + extension
        else:
            image_name = frame["file_path"]

        # NeRF 'transform_matrix' is a camera-to-world transform
        c2w = np.array(frame["transform_matrix"])
        # c2w[:3, :2] *= -1 # vggt train
        # change from OpenGL/Blender camera axes (Y up, Z back) to COLMAP (Y down, Z forward)
        # c2w[:3, 1:3] *= -1

        # get the world-to-camera transform and set R, T
        w2c = np.linalg.inv(c2w)
        R = np.transpose(w2c[:3,:3])  # R is stored transposed due to 'glm' in CUDA code
        T = w2c[:3, 3] * scene_scale_ratio

        image_path = os.path.join(path, image_name) # .replace('hdImgs_unditorted', 'hdImgs_unditorted_rgba').replace('.jpg', '.png')
        # image_name = Path(image_name).stem
        if not os.path.exists(image_path):
            logger.warning(f"Image path {image_path} does not exist, skipping this frame.")
            return
        #! Motion Mask Path
        apply_motion_mask = False
        if apply_motion_mask:
            if not is_true_image:
                motion_mask_path = image_path.replace("diffusion", "motion_masks")
            else:
                motion_mask_path = os.path.join(path, "motion_masks", frame["file_path"])
        else:
            motion_mask_path = None
        # #! Difix Path
        # if not is_true_image:
        #     image_path = image_path.replace("data", "outputs/Difix3D")
        # assert os.path.exists(image_path)

        if not dataloader:
            with Image.open(image_path) as image_load:
                im_data = np.array(image_load.convert("RGBA"))

            bg = np.array([1,1,1]) if white_background else np.array([0, 0, 0])

            norm_data = im_data / 255.0
            arr = norm_data[:,:,:3] * norm_data[:, :, 3:4] + bg * (1 - norm_data[:, :, 3:4])
            if norm_data[:, :, 3:4].min() < 1:
                arr = np.concatenate([arr, norm_data[:, :, 3:4]], axis=2)
                image = Image.fromarray(np.array(arr*255.0, dtype=np.byte), "RGBA")
            else:
                image = Image.fromarray(np.array(arr*255.0, dtype=np.byte), "RGB")

            width, height = image.size[0], image.size[1]
        else:
            image = np.empty(0)
            width, height = imagesize.get(image_path)

        depth=None
        if "depth_path" in frame:
            depth_path = os.path.join(path, frame["depth_path"].replace(".npy", "_depth.npy"))
        else:
            depth_path = os.path.join(path, "depth", frame["file_path"].replace(".png", "_depth.npy"))
        if os.path.isfile(depth_path):
            depth = torch.tensor(np.load(depth_path))

        # if 'depth_path' in frame:
        #     depth_name = frame["depth_path"]
        #     if not extension in frame["depth_path"]:
        #         depth_name = frame["depth_path"] + extension
        #     depth_path = os.path.join(path, depth_name)
        #     depth = Image.open(depth_path).copy()
        # if "depth_path" in frame and frame["depth_path"] is not None:
        #     depth_path = os.path.join(path, frame["depth_path"])
        #     if os.path.isfile(depth_path):
        #         depth = torch.tensor(np.load(depth_path))

        flow=None
        # if "flow_path" in frame:
        #     flow_path = os.path.join(path, frame["flow_path"])
        # else:
        #     flow_path = os.path.join(path, "flow", frame["file_path"].replace(".png", "_flow.npy")) ### gmflow
        # if os.path.isfile(flow_path):
        #     flow = torch.tensor(np.load(flow_path))

        tbar.update(1)
        if 'fl_x' in frame and 'fl_y' in frame and 'cx' in frame and 'cy' in frame:
            cx = frame['cx']
            cy = frame['cy']
            ratio = int(frame["width"]) / width
            cx /= ratio
            cy /= ratio
            # width = frame["width"]
            # height = frame["height"]
            fl_x = frame['fl_x']
            fl_y = frame['fl_y']
            FovX = focal2fov(fl_x, frame["width"])
            fl_x = fov2focal(FovX, width)
            FovY = focal2fov(fl_y, frame["height"])
            fl_y = fov2focal(FovY, height)
            if 'id' in frame:
                camera_id = frame['id']
            else:
                camera_id = int(frame["file_path"].split('/')[-1][3:5])
            cam_info = CameraInfo(camera_id=camera_id, uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                        image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp, pseudo_timestamp=pseudo_timestamp,
                        fl_x=fl_x, fl_y=fl_y, cx=cx, cy=cy, is_true_image=True, motion_mask_path=motion_mask_path,
                        temporal_coord_only=temporal_coord_only, spatial_coord_only=spatial_coord_only)
            return cam_info
        elif 'fx' in frame and 'fy' in frame and 'cx' in frame and 'cy' in frame:
            camera_id = frame['id']
            cx = frame['cx']
            cy = frame['cy']
            cx = cx / frame["width"] * width
            cy = cy / frame["height"]* height
            fl_x = frame['fx']
            fl_y = frame['fy']
            FovX = focal2fov(fl_x, frame["width"])
            fl_x = fov2focal(FovX, width)
            FovY = focal2fov(fl_y, frame["height"])
            fl_y = fov2focal(FovY, height)
            cam_info = CameraInfo(camera_id=camera_id, uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                        image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp, pseudo_timestamp=pseudo_timestamp,
                        fl_x=fl_x, fl_y=fl_y, cx=cx, cy=cy, is_true_image=is_true_image, motion_mask_path=motion_mask_path,
                        temporal_coord_only=temporal_coord_only, spatial_coord_only=spatial_coord_only)
            return cam_info

        elif 'fl_x' in contents and 'fl_y' in contents and 'cx' in contents and 'cy' in contents:
            if 'id' in frame:
                camera_id = frame['id']
            else:
                camera_id = int(image_name.split('/')[-1].split('_')[0][3:]) # images/cam01_0000.png -> 1
            fl_x = contents['fl_x']
            fl_y = contents['fl_y']
            cx = contents['cx']
            cy = contents['cy']
            cx = cx / contents["w"] * width
            cy = cy / contents["h"] * height
            FovX = focal2fov(fl_x, contents["w"])
            fl_x = fov2focal(FovX, width)
            FovY = focal2fov(fl_y, contents["h"])
            fl_y = fov2focal(FovY, height)
            return CameraInfo(camera_id=camera_id, uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                        image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp, pseudo_timestamp=pseudo_timestamp,
                        fl_x=fl_x, fl_y=fl_y, cx=cx, cy=cy, is_true_image=is_true_image, motion_mask_path=motion_mask_path,
                        temporal_coord_only=temporal_coord_only, spatial_coord_only=spatial_coord_only)
        else:
            fovy = focal2fov(fov2focal(fovx, width), height)
            FovY = fovy
            FovX = fovx
            return CameraInfo(uid=idx, R=R, T=T, FovY=FovY, FovX=FovX, image=image, depth=depth, flow=flow,
                            image_path=image_path, image_name=image_name, width=width, height=height, timestamp=timestamp)

    with ThreadPool() as pool:
        cam_infos = pool.map(frame_read_fn, zip(list(range(len(frames))), frames))
        pool.close()
        pool.join()

    cam_infos = [cam_info for cam_info in cam_infos if cam_info is not None]
    if mode=='train':
        # 展平嵌套列表
        flattened_cam_infos = []
        for item in cam_infos:
            if isinstance(item, list):
                flattened_cam_infos.extend(item)
            else:
                flattened_cam_infos.append(item)
        cam_infos = flattened_cam_infos


    return cam_infos

def readNerfSyntheticInfo2(path, white_background, eval, extension=".png", num_pts=100_000, time_duration=None, num_extra_pts=0, frame_ratio=1, dataloader=False, target_cam=[],
                           transforms_file='cameras.json', ply_file="mast3r_dense_pts_merged.ply", scene_scale_ratio=1.0, frame_skip=8, time_skip=5,
                           temporal_coord_only=False, spatial_coord_only=False):

    print("Reading Training Transforms")
    print(f"{transforms_file=}")
    train_cam_infos = readCamerasFromTransforms2(path, transforms_file, white_background, extension,
                                                time_duration=time_duration, frame_ratio=frame_ratio, dataloader=dataloader, scene_scale_ratio=scene_scale_ratio,
                                                target_cam=target_cam, frame_skip=frame_skip, time_skip=time_skip,
                                                temporal_coord_only=temporal_coord_only, spatial_coord_only=spatial_coord_only)
    print("Reading Test Transforms")
    # # target_cam = ["cam00"]
    # # target_cam = ["time_0000"]
    # test_cam_infos = readCamerasFromTransforms2(path, transforms_file, white_background, extension,
    #                                         time_duration=time_duration, frame_ratio=frame_ratio, dataloader=dataloader, scene_scale_ratio=scene_scale_ratio,
    #                                         target_cam=target_cam, mode='test', frame_skip=frame_skip, time_skip=time_skip,
    #                                         temporal_coord_only=temporal_coord_only, spatial_coord_only=spatial_coord_only)
    test_cam_infos = train_cam_infos
    assert len(test_cam_infos)

    if not eval:
        train_cam_infos.extend(test_cam_infos)
        test_cam_infos = []

    print(f"{len(train_cam_infos)} Training views with {len(set([cam.timestamp for cam in train_cam_infos]))} timestamps, {len(test_cam_infos)} Test views loaded")
    print(f"{len([c for c in train_cam_infos if c.flow is not None])} Training cameras with flow prior")
    print(f"{len([c for c in train_cam_infos if c.depth is not None])} Training cameras with depth prior")
    nerf_normalization = getNerfppNorm(train_cam_infos)

    ply_path = os.path.join(path, ply_file)
    if not os.path.exists(ply_path):
        # Since this data set has no colmap data, we start with random points
        print(f"Generating random point cloud ({num_pts})...")

        # We create random points inside the bounds of the synthetic Blender scenes
        xyz = np.random.random((num_pts, 3)) * 2.6 - 1.3
        shs = np.random.random((num_pts, 3)) / 255.0
        pcd = BasicPointCloud(points=xyz, colors=SH2RGB(shs), normals=np.zeros((num_pts, 3)))

        storePly(ply_path, xyz, SH2RGB(shs) * 255)
    pcd = fetchPly(ply_path, frame_ratio=frame_ratio)

    if pcd.points.shape[0] > num_pts:
        mask = np.random.randint(0, pcd.points.shape[0], num_pts)
        # mask = fps(torch.from_numpy(pcd.points).cuda()[None], num_pts).cpu().numpy()

        xyz = pcd.points[mask] * scene_scale_ratio
        rgb = pcd.colors[mask]
        normals = pcd.normals[mask]

        if pcd.time is not None:
            times = pcd.time[mask]
            time_mask = (times[:,0] < time_duration[1]) & (times[:,0] >= time_duration[0])
            xyz = xyz[time_mask]
            rgb = rgb[time_mask]
            normals = normals[time_mask]
            times = times[time_mask]
        else:
            times = None
        pcd = BasicPointCloud(points=xyz, colors=rgb, normals=normals, time=times)

    if num_extra_pts > 0:
        times = pcd.time
        xyz = pcd.points
        rgb = pcd.colors
        normals = pcd.normals
        bound_min, bound_max = xyz.min(0), xyz.max(0)
        radius = 60.0 # (bound_max - bound_min).mean() + 10
        phi = 2.0 * np.pi * np.random.rand(num_extra_pts)
        theta = np.arccos(2.0 * np.random.rand(num_extra_pts) - 1.0)
        x = radius * np.sin(theta) * np.cos(phi)
        y = radius * np.sin(theta) * np.sin(phi)
        z = radius * np.cos(theta)
        xyz_extra = np.stack([x, y, z], axis=1)
        normals_extra = np.zeros_like(xyz_extra)
        rgb_extra = np.ones((num_extra_pts, 3)) / 2

        xyz = np.concatenate([xyz, xyz_extra], axis=0)
        rgb = np.concatenate([rgb, rgb_extra], axis=0)
        normals = np.concatenate([normals, normals_extra], axis=0)

        if times is not None:
            times_extra = torch.zeros(((num_extra_pts, 3))) + (time_duration[0] + time_duration[1]) / 2
            times = np.concatenate([times, times_extra], axis=0)

        pcd = BasicPointCloud(points=xyz,
                              colors=rgb,
                              normals=normals,
                              time=times)

    if pcd.time is not None:
        logger.info(f"Point cloud time duration: {pcd.time.min()} - {pcd.time.max()}")
    scene_info = SceneInfo(point_cloud=pcd,
                           train_cameras=train_cam_infos,
                           test_cameras=test_cam_infos,
                           nerf_normalization=nerf_normalization,
                           ply_path=ply_path)
    return scene_info

sceneLoadTypeCallbacks = {
    "Colmap": readColmapSceneInfo,
    "Blender" : readNerfSyntheticInfo,
    "Blender_new" : readNerfSyntheticInfo2
}