#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

from scene.cameras import Camera
import numpy as np
from utils.general_utils import PILtoTorch
from utils.graphics_utils import fov2focal
from PIL import Image
import torch
import cv2
WARNED = False

def loadCam(args, id, cam_info, resolution_scale):
    orig_w, orig_h = cam_info.width, cam_info.height# cam_info.image.size

    if args.resolution in [1, 2, 3, 4, 8]:
        resolution = round(orig_w/(resolution_scale * args.resolution)), round(orig_h/(resolution_scale * args.resolution))
        scale = resolution_scale * args.resolution
    else:  # should be a type that converts to float
        if args.resolution == -1:
            if orig_w > 1600:
                global WARNED
                if not WARNED:
                    print("[ INFO ] Encountered quite large input images (>1.6K pixels width), rescaling to 1.6K.\n "
                        "If this is not desired, please explicitly specify '--resolution/-r' as 1")
                    WARNED = True
                global_down = orig_w / 1600
            else:
                global_down = 1
        else:
            global_down = orig_w / args.resolution

        scale = float(global_down) * float(resolution_scale)
        resolution = (int(orig_w / scale), int(orig_h / scale))

    cx = cam_info.cx / scale
    cy = cam_info.cy / scale
    fl_y = cam_info.fl_y / scale
    fl_x = cam_info.fl_x / scale

    loaded_mask = None
    if not args.dataloader:
        resized_image_rgb = PILtoTorch(cam_info.image, resolution)
        gt_image = resized_image_rgb[:3, ...]

        # if resized_image_rgb.shape[0] == 4:
        #     alpha_mask = resized_image_rgb[3:4, ...]
        #     loaded_mask = alpha_mask if loaded_mask is None else alpha_mask * loaded_mask

        if resized_image_rgb.shape[0] == 4:
            gt_alpha_mask = resized_image_rgb[3:4, ...]
        else:
            gt_alpha_mask = torch.ones((1, resolution[1], resolution[0]))
        # #----------------------- Motion Mask Test ------------------------------# #
        if cam_info.motion_mask_path is not None:
            motion_mask = cv2.imread(cam_info.motion_mask_path)
            motion_mask = torch.tensor(motion_mask[:,:,0:1] > 0.1).float()
            motion_mask = torch.nn.functional.interpolate(
                        motion_mask.permute(2,0,1)[None,], size=(resolution[1], resolution[0]),
                        mode="bicubic", align_corners=True).squeeze()
            gt_alpha_mask *= motion_mask
        loaded_mask = gt_alpha_mask if loaded_mask is None else gt_alpha_mask * loaded_mask
    else:
        gt_image = cam_info.image

    if cam_info.depth is not None:
        if isinstance(cam_info.depth, Image.Image):
            depth = PILtoTorch(cam_info.depth, resolution) * 255 / 10000
        else:
            depth = torch.nn.functional.interpolate(
                    torch.tensor(cam_info.depth).unsqueeze(0).unsqueeze(0), size=(resolution[1],resolution[0]), mode="bicubic", align_corners=False
                ).squeeze()
    else:
        depth = None

    if cam_info.flow is not None:
        # print(cam_info.flow.shape, resolution) # (h,w,c) (h,w)
        origin_h, origin_w, _ = cam_info.flow.shape
        target_h = resolution[1]
        target_w = resolution[0]
        flow = torch.nn.functional.interpolate(cam_info.flow.permute(2,0,1)[None,], size=(target_h,target_w), mode='bilinear', align_corners=True).squeeze().permute(1,2,0) #(H,W,C)
        flow[:, 0] = flow[:, 0] * target_w / origin_w
        flow[:, 1] = flow[:, 1] * target_h / origin_h
    else:
        flow = None

    # return Camera(colmap_id=cam_info.uid, R=cam_info.R, T=cam_info.T,
    #               FoVx=cam_info.FovX, FoVy=cam_info.FovY,
    #               image=gt_image, gt_alpha_mask=loaded_mask,
    #               image_name=cam_info.image_name, uid=id, data_device=args.data_device,
    #               timestamp=cam_info.timestamp,
    #               cx=cx, cy=cy, fl_x=fl_x, fl_y=fl_y, depth=depth, flow=flow, resolution=resolution, image_path=cam_info.image_path,
    #               meta_only=args.dataloader
    #               )

    # #--------------------- Flow Mask Test ---------------------# #
    if cam_info.flow_mask is not None:
        if isinstance(cam_info.flow_mask, Image.Image):
            flow_mask = PILtoTorch(cam_info.flow_mask, resolution)
        else:
            flow_mask = torch.nn.functional.interpolate(
                        torch.tensor(cam_info.flow_mask).permute(2,0,1)[None,], size=(resolution[1],resolution[0]), mode="bicubic", align_corners=True
                    ).squeeze()
    else:
        flow_mask = None
    # return Camera(colmap_id=cam_info.uid, R=cam_info.R, T=cam_info.T,
    #               FoVx=cam_info.FovX, FoVy=cam_info.FovY,
    #               image=gt_image, gt_alpha_mask=loaded_mask,
    #               image_name=cam_info.image_name, uid=id, data_device=args.data_device,
    #               timestamp=cam_info.timestamp,
    #               cx=cx, cy=cy, fl_x=fl_x, fl_y=fl_y, depth=depth, flow=flow, resolution=resolution, image_path=cam_info.image_path,
    #               meta_only=args.dataloader, flow_mask=flow_mask, is_true_image=cam_info.is_true_image
    #               )
    # #----------------------------------------------------------# #

    # #---------------------- Pose Optimization -----------------------# #
    return Camera(colmap_id=cam_info.uid, camera_id=cam_info.camera_id,
                  image=gt_image, gt_alpha_mask=loaded_mask,
                  image_name=cam_info.image_name, uid=id, data_device=args.data_device,
                  timestamp=cam_info.timestamp, pseudo_timestamp=cam_info.pseudo_timestamp,
                  depth=depth, flow=flow, resolution=resolution, image_path=cam_info.image_path, motion_mask_path=cam_info.motion_mask_path,
                  meta_only=args.dataloader, flow_mask=flow_mask, is_true_image=cam_info.is_true_image,
                  temporal_coord_only=cam_info.temporal_coord_only, spatial_coord_only=cam_info.spatial_coord_only
                  )
    # #---------------------------------------------------------------# #


def cameraList_from_camInfos(cam_infos, resolution_scale, args):
    camera_list = []

    for id, c in enumerate(cam_infos):
        camera_list.append(loadCam(args, id, c, resolution_scale))

    return camera_list

def camera_to_JSON(id, camera : Camera):
    Rt = np.zeros((4, 4))
    Rt[:3, :3] = camera.R.transpose()
    Rt[:3, 3] = camera.T
    Rt[3, 3] = 1.0

    W2C = np.linalg.inv(Rt)
    pos = W2C[:3, 3]
    rot = W2C[:3, :3]
    serializable_array_2d = [x.tolist() for x in rot]
    camera_entry = {
        'id' : id,
        'img_name' : camera.image_name,
        'width' : camera.width,
        'height' : camera.height,
        'position': pos.tolist(),
        'rotation': serializable_array_2d,
        'fy' : fov2focal(camera.FovY, camera.height),
        'fx' : fov2focal(camera.FovX, camera.width)
    }
    return camera_entry
