#include <iostream>
#include <sstream>
#include <fstream>
#include <set>
#include <random>
#include "mm_CredalGame.h"
#include "mm_Knuth.h"

namespace mm{

static void
split( const std::string& line,
       std::vector<std::string>& words )
{
    words.clear();
    std::stringstream ss( line );
    while( ss.good() ){
        std::string buf;
        ss >> buf;
        if( buf.find_first_not_of(' ') != std::string::npos ) words.push_back( buf );
    }
}

CredalGame::CredalGame( const unsigned numPeg,
                        const unsigned numColor,
                        const std::vector<std::string>& cfgLines,
                        const std::vector<bool>& corners,
                        int seed )
    : Game{ numPeg, numColor }
{
    _seed = seed;
    preCalcLieprobs( cfgLines, corners );
    getAllCodes( numPeg, numColor, _all );
}

void
CredalGame::preCalcLieprobs( const std::vector<std::string>& cfgLines,
                             const std::vector<bool>& corners )
{
    _lb=0;
    _ub=1;
    double std1=0;
    for( unsigned i=0; i<cfgLines.size(); i++ ){
        std::vector<std::string> words;
        split( cfgLines[i], words );
        if(( words.size() == 3 )&&( words[0] == "lie_prob" )){
            _lb = std::stod( words[1] );
            _ub = std::stod( words[2] );
        }
        if(( words.size() == 2 )&&( words[0] == "std" )){
            std1 = std::stod( words[1] );
        }
    }
    if(( _ub>1 )||( _lb<0 )||( _ub < _lb )) throw std::runtime_error( "invalid config file" );
    unsigned limit = 41;
    _lieProbs.resize(limit);
    if(std1>0){
        std::default_random_engine generator;
        generator.seed( _seed );
        for( unsigned i=0; i<limit; i++ ){
            double number;
            if( rand()%2 ){
                std::normal_distribution<double> distribution(_lb,std1);
                number = distribution(generator);
                if( number < _lb ) number = _lb + _lb - number;
                if( number > _ub ) number = _ub;
            }
            else{
                std::normal_distribution<double> distribution(_ub,std1);
                number = distribution(generator);
                if( number > _ub ) number = _ub + _ub - number;
                if( number < _lb ) number = _lb;
            }
            _lieProbs[i] = number;
        }
    }
    else{
        for( unsigned i=0; i<limit; i++ ) _lieProbs[i] = _lb + (_ub-_lb)*double(rand())/double(RAND_MAX);
    }
}

std::pair<unsigned,unsigned>
CredalGame::feedback( const std::vector<unsigned>& guess )
{
    if( guess == _code ) return std::pair<unsigned,unsigned>( guess.size(), 0 );
    _historyG.push_back( guess );
    unsigned size = guess.size();
    if( size != _code.size()) throw std::runtime_error("code length mismatch");
    std::pair<unsigned,unsigned> correct = get_feedback( guess, _code );
    std::set<std::pair<unsigned,unsigned> > lies;
    for( unsigned i=0; i<_all.size(); i++ ){
        if( guess == _all[i] ) continue;
        std::pair<unsigned,unsigned> f = get_feedback( guess, _all[i] );
        if( f == correct ) continue;
        lies.insert( f );
    }
    if( lies.empty() ) throw std::runtime_error("no available lie");
    _historyM.push_back( lies.size() );
    double r = double(rand())/double(RAND_MAX);
    if( r > _lieProbs[_historyF.size()] ){
        _historyF.push_back( correct );
        return correct;
    }
    std::vector<std::pair<unsigned,unsigned> > liesv(lies.begin(),lies.end());
    std::pair<unsigned,unsigned> f = liesv[rand() % lies.size()];
    _historyF.push_back( f );
    return f;
}

double
CredalGame::stats( std::ostream& os )
{
    double sum=0, max=0, codeprob=0;
    std::vector<unsigned> maxindex;
    unsigned numG = _historyG.size();
    for( unsigned i=0; i<_all.size(); i++ ){
        bool display = false;
        double prob = 1;
        for( unsigned ii=0; ii<_historyG.size(); ii++ ){
            if( get_feedback( _historyG[ii], _all[i] ) == _historyF[ii] ) prob *= 1-_lieProbs[ii];
            else prob *= _lieProbs[ii]/double(_historyM[ii]);
            if( display ) os << prob << "\n";
        }
        sum += prob;
        if( max < prob*0.9999 ){
            max = prob;
            maxindex.clear();
        }
        if( max < prob*1.0001 ) maxindex.push_back(i);
        if( _all[i] == _code ) codeprob = prob;
    }
    os << "correct answer prob: " << codeprob/sum << "\n"
       << "map prob: " << max/sum << "\n"
       << "map code: " << maxindex.size();
    for( unsigned i=0; i<maxindex.size(); i++ ) os << " " << code_to_string(_all[maxindex[i]]);
    os << "\n";
    return codeprob/sum;
}

void
CredalGame::writeProbFile( const std::string& filename )
{
    std::ofstream os( filename.c_str() );
    for( unsigned i=0; i<_lieProbs.size(); i++ ) os << _lieProbs[i] << "\n";
    os.close();
}
    
}
