#include <iostream>
#include <sstream>
#include <set>
#include "mm_BayesGame.h"
#include "mm_Knuth.h"

namespace mm{

static void
split( const std::string& line,
       std::vector<std::string>& words )
{
    words.clear();
    std::stringstream ss( line );
    while( ss.good() ){
        std::string buf;
        ss >> buf;
        if( buf.find_first_not_of(' ') != std::string::npos ) words.push_back( buf );
    }
}

BayesGame::BayesGame( const unsigned numPeg,
                      const unsigned numColor,
                      const std::vector<std::string>& config )
    : Game{ numPeg, numColor }
{
    _lieProb = 0;
    for( unsigned i=0; i<config.size(); i++ ){
        std::vector<std::string> words;
        split( config[i], words );
        if(( words.size() == 2 )&&( words[0] == "lie_prob" )) _lieProb = std::stod( words[1] );
    }
    if(( _lieProb>1 )||( _lieProb<0 )) throw std::runtime_error( "invalid config file" );
    getAllCodes( numPeg, numColor, _all );
}

std::pair<unsigned,unsigned>
BayesGame::feedback( const std::vector<unsigned>& guess )
{
    if( guess == _code ) return std::pair<unsigned,unsigned>( guess.size(), 0 );
    _historyG.push_back( guess );
    unsigned size = guess.size();
    if( size != _code.size()) throw std::runtime_error("code length mismatch");
    std::pair<unsigned,unsigned> correct = get_feedback( guess, _code );
    std::set<std::pair<unsigned,unsigned> > lies;
    for( unsigned i=0; i<_all.size(); i++ ){
        if( guess == _all[i] ) continue;
        std::pair<unsigned,unsigned> f = get_feedback( guess, _all[i] );
        if( f == correct ) continue;
        lies.insert( f );
    }
    if( lies.empty() ) throw std::runtime_error("no available lie");
    _historyM.push_back( lies.size() );
    double r = double(rand())/double(RAND_MAX);
    if( r > _lieProb ){
        _historyF.push_back( correct );
        return correct;
    }
    std::vector<std::pair<unsigned,unsigned> > liesv(lies.begin(),lies.end());
    std::pair<unsigned,unsigned> f = liesv[rand() % lies.size()];
    _historyF.push_back( f );
    return f;
}

double
BayesGame::stats( std::ostream& os )
{
    double sum=0, max=0, codeprob=0;
    std::vector<unsigned> maxindex;
    for( unsigned i=0; i<_all.size(); i++ ){
        bool display = false;
        double prob = 1;
        for( unsigned ii=0; ii<_historyG.size(); ii++ ){
            if( get_feedback( _historyG[ii], _all[i] ) == _historyF[ii] ) prob *= 1-_lieProb;
            else prob *= _lieProb/double(_historyM[ii]);
            if( display ) os << prob << "\n";
        }
        sum += prob;
        if( max < prob*0.9999 ){
            max = prob;
            maxindex.clear();
        }
        if( max < prob*1.0001 ) maxindex.push_back(i);
        if( _all[i] == _code ) codeprob = prob;
    }
    os << "correct answer prob: " << codeprob/sum << "\n"
       << "map prob: " << max/sum << "\n"
       << "map code: " << maxindex.size();
    for( unsigned i=0; i<maxindex.size(); i++ ) os << " " << code_to_string(_all[maxindex[i]]);
    os << "\n";
    return codeprob/sum;
}

}
