#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <pthread.h>
#include "mm_CredalGame.h"
#include "mm_Knuth.h"
#include "mm_Keyboard.h"

struct GameParms{
    unsigned numPeg,numColor,numGuess,index,result=0,repeat;
    double prob;
    std::string playerName,display,error,gameType,saveDir;
    std::vector<std::string> config;
    std::vector<bool> corners;
};

static bool
play( mm::Game*  game,
      GameParms* parms,
      std::stringstream& ss )
{
    mm::Player* player;
    if( parms->playerName == "random" ) player = new mm::Player( parms->numPeg, parms->numColor );
    else if( parms->playerName == "Knuth" ) player = new mm::Knuth( parms->numPeg, parms->numColor);
    else throw std::runtime_error( "unknown player" );
    for( unsigned ii=0; ii<parms->numGuess; ii++ ){
        std::vector<unsigned> guess;
        player->guess( guess );
        std::pair<unsigned,unsigned> feedback = game->feedback( guess );
        ss << mm::code_to_string(guess) << " "
           << mm::pair_to_string(feedback) << "\n";
        if( feedback.first == parms->numPeg ){
            parms->result = ii+1;
            return true;
        }
        try{
            player->read( guess, feedback );
        }
        catch( std::exception& e ){
            if( parms->gameType == "original" ) throw e;
            else{
                parms->error = e.what();
                return false;
            }
        }
    }
    delete player;
    return false;
}

static void*
game_thread( void* arg )
{
    GameParms* parms = (GameParms*)arg;
    try{
        std::stringstream ss;
        ss << "Game " << parms->index << ":\n";
        mm::Game* game;
        if( parms->gameType == "original" ) game = new mm::Game( parms->numPeg, parms->numColor );
        else if( parms->gameType == "Bayesian" ) game = new mm::BayesGame( parms->numPeg, parms->numColor, parms->config );
        else if( parms->gameType == "credal" ) game = new mm::CredalGame( parms->numPeg, parms->numColor,
                                                                          parms->config, parms->corners, parms->index );
        else throw std::runtime_error( "unknown game type" );
        ss << "hidden " << game->code() << "\n";
        parms->result = 0;
        unsigned repeat = 1;
        if(( parms->repeat > 1 )&&( parms->gameType != "original" )) repeat = parms->repeat;
        bool success = false;
        for( unsigned i=0; i<repeat; i++ ){
            if( play(game,parms,ss) ){
                success = true;
                break;
            }
        }
        if(!success){
            parms->prob = game->stats( ss );
            if( parms->gameType == "credal" ) ((mm::CredalGame*)game)->writeProbFile(parms->saveDir);
        }
        parms->display = ss.str();
        delete game;
    }
    catch( std::exception& e ){
        parms->error = e.what();
    }
    return NULL;
}

int
main( int argc, const char* argv[] )
{
    int numPeg   = 4;
    int numColor = 6;
    int numGame  = 1;
    int numGuess = 12;
    int position = 1;
    int seed     = 0;
    int repeat   = 1;
    bool deterministic = false;
    std::string playerName( "Knuth" );
    std::string gameType( "original" );
    std::string configFile;
    std::string saveDir;
    while( position < argc ){
        if( argv[position][0] != '-' ){
            std::cout << "Wrong usage: invalue argument \""
                      << argv[position] << "\".\n";
            return 1;
        }
        std::string buf( argv[position] );
        if(( buf == "-numPeg" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numPeg) < 1 )||( numPeg <= 0 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-numColor" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numColor) < 1 )||( numColor <= 1 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-numGame" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numGame) < 1 )||( numGame <= 0 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-numGuess" )&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &numGuess) < 1 )||( numGuess <= 0 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if(( buf == "-gameConfig" )&&( argc > position+2 )){
            gameType  = std::string( argv[position+1] );
            configFile= std::string( argv[position+2] );
            position += 3;
        }
        else if(( buf == "-player" )&&( argc > position+1 )){
            playerName = std::string( argv[position+1] );
            position += 2;
        }
        else if(( buf == "-randomseed")&&( argc > position+1 )){
            if(( sscanf( argv[position+1], "%d", &seed ) < 1 )){
                std::cout << "Wrong usage: invalue argument \""
                          << argv[position] << " "
                          << argv[position+1] << "\".\n";
                return 1;
            }
            position += 2;
        }
        else if( buf == "-deterministic" ){
            deterministic = true;
            position++;
        }
        else if(( buf == "-repeat" )&&( argc > position+1 )){
            repeat = std::stoi( argv[position+1] );
            position += 2;
        }
        else if(( buf == "-saveDir" )&&( argc > position+1 )){
            saveDir = std::string( argv[position+1] );
            position += 2;
            system((std::string("mkdir ")+saveDir).c_str());
        }
        else{
            std::cout << "Wrong usage: invalue argument \""
                      << argv[position] << "\".\n";
            return 1;
        }
    }
    if( seed >= 0 ) srand( seed );
    else if( deterministic ) srand( 0 );
    else srand( (int)time(NULL));
    std::vector<std::string> cfgLines;
    if( configFile.size() > 0 ){
        std::ifstream is( configFile.c_str());
        std::string buffer;
        while( is.good()){
            std::getline( is, buffer );
            cfgLines.push_back( buffer );
        }
    }
    std::vector<bool> corners(41);
    for( unsigned i=0; i<41; i++ ) corners[i] = rand()%2;
    std::cout << "start " << numGame << " games with " << numPeg
              << " pegs and " << numColor << " colors.\n";
    unsigned fail=0,sum=0;
    time_t start,end;
    time(&start);
    if(( playerName == "human" )|| deterministic ){
        try{
            for(unsigned i=0; i<numGame; i++){
                std::cout << "Game " << i << ":\n";
                mm::Game* game;
                if( gameType == "original" ) game = new mm::Game( numPeg, numColor );
                else if( gameType == "Bayesian" ) game = new mm::BayesGame( numPeg, numColor, cfgLines );
                else if( gameType == "credal" ) game = new mm::CredalGame( numPeg, numColor, cfgLines, corners );
                else throw std::runtime_error( "unknown game type" );
                if( playerName != "human" ) std::cout << "hidden " << game->code() << "\n";
                mm::Player* player;
                if( playerName == "random" ) player = new mm::Player( numPeg, numColor );
                else if( playerName == "Knuth" ) player = new mm::Knuth( numPeg, numColor);
                else if( playerName == "human" ) player = new mm::Keyboard( numPeg, numColor);
                else throw std::runtime_error( "unknown player" );
                bool solved = false;
                for( unsigned ii=0; ii<numGuess; ii++ ){
                    std::vector<unsigned> guess;
                    player->guess( guess );
                    std::pair<unsigned,unsigned> feedback = game->feedback( guess );
                    std::cout << mm::code_to_string(guess) << " "
                              << mm::pair_to_string(feedback) << "\n";
                    if( feedback.first == numPeg ){
                        solved = true;
                        sum += ii+1;
                        break;
                    }
                    player->read( guess, feedback );
                }
                if( !solved ) fail++;
                if( playerName == "human" ){
                    if( solved ) std::cout << "congrats!\n";
                    else std::cout << "failed. hidden code is " << game->code() << "\n";
                }
                delete player;
                delete game;
            }
        }
        catch( std::exception& e ){
            std::cout << "Error: " << e.what() << "\n";
            return 1;
        }
    }
    else{
        std::vector<GameParms> parms( numGame );
        std::vector<pthread_t> threads( numGame );
        for( unsigned i=0; i<numGame; i++ ){
            parms[i].numPeg = numPeg;
            parms[i].numColor = numColor;
            parms[i].numGuess = numGuess;
            parms[i].index = i;
            parms[i].playerName = playerName;
            parms[i].gameType = gameType;
            parms[i].config = cfgLines;
            parms[i].repeat = repeat;
            parms[i].saveDir = saveDir + "/game_" + std::to_string(i) + ".prob";
            parms[i].corners = corners;
            pthread_create( &(threads[i]), NULL, game_thread, &(parms[i]));
        }
        for( unsigned i=0; i<numGame; i++ ){
            pthread_join( threads[i], NULL );
            if( !parms[i].error.empty() ){
                if( gameType == "original" ){
                    std::cout << "Error: " << parms[i].error << "\n";
                    return 1;
                }
                else std::cout << "Warning: " << parms[i].error << "\n";
            }
            std::cout << parms[i].display;
            if( parms[i].result == 0 ){
                fail++;
                if( !saveDir.empty() ){
                    std::string filename = saveDir + "/game_" + std::to_string(i);
                    std::ofstream os( filename.c_str() );
                    os << parms[i].display;
                    os.close();
                }
            }
            else sum += parms[i].result;
        }
    }
    std::cout << "Fails: " << fail << "/" << numGame << "\n";
    if( fail < numGame ) std::cout << "Avg turns: " << double(sum)/double(numGame-fail) << "\n";
    time(&end);
    std::cout << "Run time: " << difftime(end,start) << " seconds\n";
    return 0;
}

