# -*- coding: utf-8 -*-
import torch
import torch.nn as nn
from model.layer import *


def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, bn_type='', expand=1, **kwargs_spikes):
        super(BasicBlock, self).__init__()
        self.kwargs_spikes = kwargs_spikes
        self.nb_steps = kwargs_spikes['timestep']
        self.expand = expand
        self.conv1 = nn.Conv2d(in_planes, planes * expand, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes * expand)
        self.spike1 = LIFLayer(**kwargs_spikes)
        self.conv2 = nn.Conv2d(planes, planes * expand, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes * expand)
        self.shortcut = SequentialModule()
        if stride != 1 or in_planes != planes * self.expansion:
            self.shortcut = SequentialModule(
                nn.Conv2d(in_planes, planes * self.expansion * expand, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes * expand)
            )
        self.spike2 = LIFLayer(**kwargs_spikes)

    def forward(self, x, rate0):
        out, rate = self.spike1(*self.bn1(*self.conv1(x, rate0)))
        out, rate = self.bn2(*self.conv2(out, rate))

        v1, v2 = self.shortcut(x, rate0)
        out, rate = out + v1, rate + v2

        out, rate = self.spike2(out, rate)
        return out, rate


class ResNet(nn.Module):
    def __init__(self, block, num_block_layers, num_classes=10, in_channel=3, bn_type='', **kwargs_spikes):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.bn_type = bn_type
        self.kwargs_spikes = kwargs_spikes
        self.timestep = kwargs_spikes['timestep']
        self.rate_flag = kwargs_spikes.get("rate_flag", False)

        self.conv0 = SequentialModule(
            nn.Conv2d(in_channel, self.in_planes, kernel_size=3, padding=1, stride=1, bias=False),
            nn.BatchNorm2d(self.in_planes),
            LIFLayer(**kwargs_spikes)
        )
        self.layer1 = self._make_layer(block, 64, num_block_layers[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_block_layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_block_layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_block_layers[3], stride=2)

        self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))
        self.classifier = SequentialModule(
            nn.Linear(512 * block.expansion, num_classes),
        )

        wrap_model(self, self.timestep, rate_flag=self.rate_flag)
        self.register_forward_hook(affine_forward_hook)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, self.bn_type, **self.kwargs_spikes))
            self.in_planes = planes * block.expansion
        return SequentialModule(*layers)

    def forward(self, x):
        out, _ = torch.broadcast_tensors(x, torch.zeros((self.timestep,) + x.shape))
        if self.rate_flag and self.training:
            rate = out.mean(dim=0).clone()
        else:
            rate = torch.zeros_like(out[0])

        out, rate = self.conv0(out, rate)
        out, rate = self.layer1(out, rate)
        out, rate = self.layer2(out, rate)
        out, rate = self.layer3(out, rate)
        out, rate = self.layer4(out, rate)
        out, rate = self.avg_pool(out, rate)

        out, rate = out.view(out.shape[0], out.shape[1], -1), rate.view(rate.shape[0], -1)
        out, rate = self.classifier(out, rate)
        return out, rate


class ResNet19(nn.Module):
    def __init__(self, block, num_block_layers, num_classes=10, in_channel=3, bn_type='', **kwargs_spikes):
        super(ResNet19, self).__init__()
        self.in_planes = 128
        self.bn_type = bn_type
        self.kwargs_spikes = kwargs_spikes
        self.nb_steps = kwargs_spikes['timestep']
        self.rate_flag = kwargs_spikes.get("rate_flag", False)

        self.conv0 = SequentialModule(
            nn.Conv2d(in_channel, self.in_planes, kernel_size=3, padding=1, stride=1, bias=False),
            nn.BatchNorm2d(self.in_planes),
            LIFLayer(**kwargs_spikes)
        )
        self.layer1 = self._make_layer(block, 128, num_block_layers[0], stride=1)
        self.layer2 = self._make_layer(block, 256, num_block_layers[1], stride=2)
        self.layer3 = self._make_layer(block, 512, num_block_layers[2], stride=2)
        self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))
        self.classifier = SequentialModule(
            nn.Linear(512 * block.expansion, 256, bias=True),
            LIFLayer(**kwargs_spikes),
            nn.Linear(256, num_classes, bias=True),
        )

        wrap_model(self, self.nb_steps, rate_flag=self.rate_flag)
        self.register_forward_hook(affine_forward_hook)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, self.bn_type, **self.kwargs_spikes))
            self.in_planes = planes * block.expansion
        return SequentialModule(*layers)

    def forward(self, x):
        out, _ = torch.broadcast_tensors(x, torch.zeros((self.nb_steps,) + x.shape))
        if self.rate_flag and self.training:
            rate = out.mean(dim=0).clone()
        else:
            rate = torch.zeros_like(out[0])

        out, rate = self.conv0(out, rate)
        out, rate = self.layer1(out, rate)
        out, rate = self.layer2(out, rate)
        out, rate = self.layer3(out, rate)
        out, rate = self.avg_pool(out, rate)
        out, rate = out.view(out.shape[0], out.shape[1], -1), rate.view(rate.shape[0], -1)
        out, rate = self.classifier(out, rate)
        return out, rate


def rate_resnet18(num_classes=10, in_channel=3, neuron_dropout=0.0, **kwargs):
    return ResNet(BasicBlock, [2, 2, 2, 2], num_classes, in_channel=in_channel, **kwargs)


def rate_resnet19(num_classes=10, in_channel=3, neuron_dropout=0.0, **kwargs):
    return ResNet19(BasicBlock, [3, 3, 2], num_classes, in_channel=in_channel, **kwargs)
