import torch


def geomed(vectors, alphas=-1, initial_point=-1, iterations=-1, nu=-1):
    """
    This function computes smoothed weighted geometric median using Weiszfeld's method.

    :param vectors: vectors to be aggregated
    :param alphas: weights of the vectors, the length of alphas should be the same as vectors
    :param initial_point: initial point of iteration
    :param iterations: numbers of iteration
    :param nu: smoothness parameter
    :return: the aggregated result
    """

    length = len(vectors)

    # default setting
    if alphas == -1:
        alphas = [1.]*length
    if initial_point == -1:
        initial_point = torch.zeros_like(vectors[0])
        if torch.cuda.is_available():
            initial_point = initial_point.cuda()
    if iterations == -1:
        iterations = 10
    if nu == -1:
        nu = 0.1

    # check
    if len(alphas) != length:
        raise ValueError("wrong dimension of alphas")
    if nu <= 0:
        raise ValueError("parameter nu should be positive")
    if iterations <= 0:
        raise ValueError("parameter iterations should be positive")

    # initialize iteration
    z = torch.clone(initial_point)
    z_new = torch.zeros_like(initial_point)

    # iteration
    for t in range(iterations):
        beta_sum = 0
        z_new.zero_()
        for l in range(length):
            beta = alphas[l]/max(nu, torch.norm(z.sub(vectors[l]), 2))
            beta_sum += beta
            z_new.add_(beta, vectors[l])
        z = z_new.div(beta_sum)

    return z
