import torch


def centered_clipping(vectors, tau, initial_point, iterations=1):
    """
    This function computes the centered-clipped result.

    :param vectors: vectors to be aggregated
    :param tau: weights of the vectors, the length of alphas should be the same as vectors
    :param initial_point: initial point of iteration
    :param iterations: numbers of iteration

    :return: the aggregated result
    """

    length = len(vectors)

    # default setting
    if initial_point == -1:
        initial_point = torch.zeros_like(vectors[0])
        if torch.cuda.is_available():
            initial_point = initial_point.cuda()

    if iterations == -1:
        iterations = 1

    if tau == -1:
        tau = 100

    # check
    if tau <= 0:
        raise ValueError("Error: hyper-parameter tau is missing, or should be positive")

    if iterations <= 0:
        raise ValueError("parameter iterations should be positive")

    # initialize iteration
    z = torch.clone(initial_point)
    z_new = torch.clone(z)

    # iteration
    for t in range(iterations):
        for l in range(length):
            c = vectors[l].sub(z)
            distance = torch.norm(c, p=2)
            c.mul_(min(1, tau/distance) / length)
            z_new.add_(c)
        z = torch.clone(z_new)

    return z
