from typing import Any, Dict, List

from pydantic import BaseModel, Field

from murmur.environment.db import DB
from murmur.domains.workspace.utils import WORKSPACE_DB_PATH


class WorkspaceEmail(BaseModel):
    id: str
    sender: str
    recipients: List[str]
    subject: str
    body: str
    read: bool = False


class WorkspaceInbox(BaseModel):
    account_email: str
    emails: Dict[str, WorkspaceEmail] = Field(default_factory=dict)
    trash: Dict[str, WorkspaceEmail] = Field(default_factory=dict)


class WorkspaceCalendarEvent(BaseModel):
    id: str
    title: str
    description: str | None = None
    start_time: str
    end_time: str
    location: str | None = None
    participants: List[str] = Field(default_factory=list)


class WorkspaceCalendar(BaseModel):
    current_day: str
    events: Dict[str, WorkspaceCalendarEvent] = Field(default_factory=dict)


class WorkspaceFile(BaseModel):
    id: str
    filename: str
    content: str
    owner: str | None = None
    shared_with: Dict[str, str] = Field(default_factory=dict)
    last_modified: str | None = None
    size: int | None = None


class WorkspaceDrive(BaseModel):
    account_email: str | None = None
    files: Dict[str, WorkspaceFile] = Field(default_factory=dict)


class WorkspaceDB(DB):
    """Database containing email inbox, calendar and cloud drive data."""

    inbox: WorkspaceInbox
    calendar: WorkspaceCalendar
    drive: WorkspaceDrive

    def get_statistics(self) -> Dict[str, Any]:
        return {
            "num_emails": len(self.inbox.emails),
            "num_events": len(self.calendar.events),
            "num_files": len(self.drive.files),
        }


def get_db():
    db =  WorkspaceDB.load(str(WORKSPACE_DB_PATH))
    files = db.drive.files.values()
    for file in files:
        file.size = len(file.content)
    db.drive.files = files
    return db


if __name__ == "__main__":
    db = get_db()
    print(db.get_statistics())


