import torch
import torch.nn as nn
import torch.nn.functional as F


class FConv2d(nn.Conv2d):
    def forward(self, input, weight, bias):
        _weight = weight 
        _bias = bias if bias is not None else self.bias
        if self.padding_mode != 'zeros':
            raise NotImplementedError("not implemented for non-zero padding")
        return F.conv2d(input, _weight, _bias, self.stride,
                        self.padding, self.dilation, self.groups)

class FBatchNorm2d(nn.BatchNorm2d):
    def forward(self, input, weight, bias):
        self._check_input_dim(input)
        _weight = weight if weight is not None else self.weight
        _bias = bias if bias is not None else self.bias

        # exponential_average_factor is set to self.momentum
        # (when it is available) only so that it gets updated
        # in ONNX graph when this node is exported to ONNX.
        if self.momentum is None:
            exponential_average_factor = 0.0
        else:
            exponential_average_factor = self.momentum

        if self.training and self.track_running_stats:
            # TODO: if statement only here to tell the jit to skip emitting this when it is None
            if self.num_batches_tracked is not None:
                self.num_batches_tracked = self.num_batches_tracked + 1
                if self.momentum is None:  # use cumulative moving average
                    exponential_average_factor = 1.0 / float(self.num_batches_tracked)
                else:  # use exponential moving average
                    exponential_average_factor = self.momentum

        r"""
        Decide whether the mini-batch stats should be used for normalization rather than the buffers.
        Mini-batch stats are used in training mode, and in eval mode when buffers are None.
        """
        if self.training:
            bn_training = True
        else:
            bn_training = (self.running_mean is None) and (self.running_var is None)

        r"""
        Buffers are only updated if they are to be tracked and we are in training mode. Thus they only need to be
        passed when the update should occur (i.e. in training mode when they are tracked), or when buffer stats are
        used for normalization (i.e. in eval mode when buffers are not None).
        """
        return F.batch_norm(
            input,
            # If buffers are not to be tracked, ensure that they won't be updated
            self.running_mean if not self.training or self.track_running_stats else None,
            self.running_var if not self.training or self.track_running_stats else None,
            _weight, _bias, bn_training, exponential_average_factor, self.eps)
    
class WLinear(nn.Linear):
    def forward(self, input, weight, bias):
        _weight = weight if weight is not None else self.weight
        _bias = bias if bias is not None else self.bias
        return F.linear(input, _weight, _bias)